package com.xforceplus.phoenix.tools.util;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import lombok.experimental.UtilityClass;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;

@Slf4j
@UtilityClass
public class TaxRateUtil {

    private static final String REGEX = "%|\\s";
    private static final BigDecimal HUNDRED = new BigDecimal(100);
    private static final DecimalFormat FORMAT = new DecimalFormat("#.######");

    /**
     * 税率除以100(如果大于等于1的情况下)
     *
     * @param textRate 税率
     * @return 格式化后的税率数字
     */
    public static String formatDivide(final String textRate) {
        if (StringUtils.isBlank(textRate)) {
            return textRate;
        }
        final String textRateNumber = textRate.replaceAll(REGEX, StringUtils.EMPTY);
        if (NumberUtils.isCreatable(textRateNumber)) {
            BigDecimal rate = new BigDecimal(textRateNumber);
            rate = rate.compareTo(BigDecimal.ONE) < 0 ? rate : rate.divide(HUNDRED);
            return FORMAT.format(rate);
        }
        return textRate;
    }

    /**
     * 税率乘以100 (输入，输出) (0.09，9) (0.2，20) (0.062，6.2) (0.13%，13) (0.13%*，13) (1%，1%) (9%，9%)
     *
     * @param formatRate 税率
     * @return 格式化后的税率数字
     */
    public static String formatMultiply(String formatRate) {
        if (StringUtils.isBlank(formatRate)) {
            return formatRate;
        }
        if (formatRate.indexOf(".") > 0) {
            formatRate = formatRate.replaceAll("0+?$", "").replaceAll("[.]$", "");
        }
        if (formatRate.indexOf("%") > 0) {
            formatRate = formatRate.replaceAll("%", "");
        }
        final String rateAmount = getAmount(formatRate);
        if (!NumberUtils.isCreatable(rateAmount)) {
            return formatRate;
        }
        BigDecimal rate = new BigDecimal(rateAmount);
        try {
            rate = rate.compareTo(BigDecimal.ONE) < 0 ? rate.multiply(HUNDRED).stripTrailingZeros() : rate;
            return rate.toPlainString();
        } catch (Exception e) {
            log.error("formatMultiply error formatRate:{}", formatRate, e);
        }
        return formatRate;
    }

    public static String getAmount(String str) {
        final Matcher matcher = Pattern.compile("\\d+\\.\\d+").matcher(str);
        return matcher.find() ? matcher.group() : str;
    }

}
