package com.xforceplus.pscc.common.bot.task;

import com.xforceplus.pscc.common.bot.BotConfig;
import com.xforceplus.pscc.common.bot.BotService;
import com.xforceplus.pscc.common.intercept.TraceContext;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicInteger;
import javax.servlet.http.HttpServletRequest;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

@Slf4j
public class DingdingWarning implements ExceptionWarning {

    private static final Map<Object, Integer> countMap = new ConcurrentHashMap<>();
    private static final AtomicInteger countAll = new AtomicInteger(0);
    private Long timePoint = System.currentTimeMillis();

    private BotConfig botConfig;
    private BotService botService;
    private ExecutorService executorService;
    private String appName;
    private String env;
    private String ip;

    public DingdingWarning(BotConfig botConfig, BotService botService, ExecutorService executorService, String appName, String env) {
        this.botConfig = botConfig;
        this.botService = botService;
        this.executorService = executorService;
        this.appName = appName;
        this.env = env;
        try {
            this.ip = InetAddress.getLocalHost().getHostAddress();
        } catch (UnknownHostException e) {
            this.ip = "未知";
            e.printStackTrace();
        }
    }


    /**
     * http请求异常预警
     *
     * @param e       异常
     * @param request 请求
     */
    @Override
    public void httpWarn(Exception e, HttpServletRequest request) {
        String traceId = TraceContext.getContextTraceId();
        executorService.submit(() -> {
            try {
                if (!needWarn(e)) {
                    return;
                }
                botService.sendInfo(String.format(
                    "traceId: %s\nappName: %s\nip: %s\nrequestURI: %s\nmethodName: %s\nenv: %s\nexceptionMessage: %s",
                    traceId,
                    appName,
                    ip,
                    request.getRequestURI(),
                    request.getMethod(),
                    env,
                    getStackTraceString(e))
                );
            } catch (Exception ecp) {
                log.error("钉钉机器人计数告警错误：{}", ecp.getMessage());
            }
        });
    }

    /**
     * janus消息异常预警
     *
     * @param e           异常
     * @param requestName 请求队列名
     */
    @Override
    public void januxWarn(Exception e, String requestName) {
        String traceId = TraceContext.getContextTraceId();
        executorService.submit(() -> {
            try {
                if (!needWarn(e)) {
                    return;
                }
                botService.sendInfo(String.format(
                    "traceId: %s\nappName: %s\nip: %s\nrequestURI: %s\nmethodName: %s\nenv: %s\nexceptionMessage: %s",
                    traceId,
                    appName,
                    ip,
                    requestName,
                    "Janus",
                    env,
                    getStackTraceString(e))
                );
            }catch(Exception ecp) {
                log.error("钉钉机器人计数告警错误：{}", ecp.getMessage());
            }
        });
    }

    private boolean needWarn(Exception e) {
        boolean needWarn = false;
        String errClz = e.getClass().toString();
        // 策略1 只限制全局异常告警频率（如10分钟内只允许2条任何异常被告警）
        if(botConfig.getWarningType().equals(0)) {
            needWarn = countAll(2);
        }
        // 策略2 放宽全局异常告警频率限制，对逐类异常的频率单独限制（如10分钟内允许10条任何异常被告警，但某msg的告警只能出现5次）
        else if(botConfig.getWarningType().equals(1)) {
            needWarn = countAll(10) && countOne(errClz);
        }
        return needWarn;
    }

    /**
     * 对服务内任何异常，a时间内可以报b条异常
     */
    private boolean countAll(Integer count) {
        count = botConfig.getWarningCountThres() == null ? count : botConfig.getWarningCountThres();
        if ((System.currentTimeMillis() - timePoint) / 60000 < botConfig.getWarningMinutesThres()) {
            if (countAll.get() < count) {
                countAll.addAndGet(1);
                return true;
            } else {
                return false;
            }
        } else {
            timePoint = System.currentTimeMillis();
            countAll.set(1);
            countMap.clear();
            return true;
        }
    }

    /**
     * 内部-方法级-消息异常预警
     *
     * @param e             异常
     * @param bussinessInfo 提示信息
     */
    @Override
    public void exceptionWarn(Exception e, String bussinessInfo) {
        String traceId = TraceContext.getContextTraceId();
        executorService.submit(() -> {
            try {
                if (!needWarn(e)) {
                    return;
                }
                botService.sendInfo(String.format(
                    "traceId: %s\nappName: %s\nip: %s\nbussinessInfo: %s\nmethodName: %s\nenv: %s\nexceptionMessage: %s",
                    traceId,
                    appName,
                    ip,
                    bussinessInfo,
                    "method",
                    env,
                    getStackTraceString(e))
                );
            } catch (Exception ecp) {
                log.error("钉钉机器人计数告警错误：{}", ecp.getMessage());
            }
        });
    }

    /**
     * 对服务内各个异常进行计数，a次内可以告警
     */
    private boolean countOne(String key) {
        if (StringUtils.isEmpty(key)) {
            key = "未知异常";
        }
        int count = countMap.get(key) != null ? countMap.get(key) + 1 : 1;
        countMap.put(key, count);
        if(count <= botConfig.getWarningCountThresByOne()) {
            return true;
        }else {
            countAll.addAndGet(-1);
            return false;
        }
    }

    private String getStackTraceString(Exception e) {
        StringWriter sw = new StringWriter();
        e.printStackTrace(new PrintWriter(sw,true));
        String stack = sw.toString();
        stack = stack.length() >= 200 ? stack.substring(0, 200)+"..." : stack;
        return stack;
    }

}
