package com.xforceplus.pscc.common.exception;

import cn.hutool.core.util.StrUtil;
import com.xforceplus.pscc.common.bot.task.ExceptionWarning;
import com.xforceplus.pscc.common.dto.CommonResponse;
import com.xforceplus.pscc.common.intercept.TraceContext;
import java.util.List;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.annotation.Order;
import org.springframework.http.HttpStatus;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.validation.BindingResult;
import org.springframework.validation.FieldError;
import org.springframework.validation.ObjectError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;

/**
 * @ClassName GlobalExceptionHandler
 * @Description 全局处理自定义的业务异常
 * @Author zjj
 * @Date 2021/12/29 15:59
 * @Version 1.0
 */
@ControllerAdvice
@Order()
@Slf4j
public class GlobalExceptionHandler {

    @Autowired
    ExceptionWarning exceptionWarning;

    @ExceptionHandler(Exception.class)
    @ResponseBody
    public CommonResponse<?> handleException(Exception e, HttpServletRequest request, HttpServletResponse response) {
        exceptionWarning.httpWarn(e, request);
        log.error(e.getMessage(), e);
        response.setStatus(HttpStatus.INTERNAL_SERVER_ERROR.value());
        return CommonResponse.failed("未知异常，请联系管理员。" + StrUtil.maxLength(TraceContext.getContextTraceId(), 8));
    }

    @ExceptionHandler(BusinessException.class)
    @ResponseBody
    public CommonResponse<?> handleBusinessException(BusinessException e, HttpServletResponse response) {
        log.error(e.getMessage(), e);
        if (e.getErrorCode() != null) {
            response.setStatus(e.getErrorCode().getErrCode());
        } else {
            response.setStatus(HttpStatus.OK.value());
        }
        return CommonResponse.failed(e.getMessage());
    }

    @ExceptionHandler(HttpMessageNotReadableException.class)
    @ResponseBody
    public CommonResponse<?> handleHttpMessageNotReadableException(HttpMessageNotReadableException e, HttpServletResponse response) {
        log.error(e.getMessage(), e);
        response.setStatus(HttpStatus.INTERNAL_SERVER_ERROR.value());
        return CommonResponse.failed("参数检验失败，请检查填写的参数是否合法！");
    }

    @ExceptionHandler(value = IllegalArgumentException.class)
    @ResponseBody
    public CommonResponse<?> handleIllegalArgumentException(IllegalArgumentException e, HttpServletResponse response) {
        log.error(e.getMessage(), e);
        response.setStatus(HttpStatus.INTERNAL_SERVER_ERROR.value());
        return CommonResponse.failed("非法参数异常" + StrUtil.maxLength(TraceContext.getContextTraceId(), 8));
    }

    @ExceptionHandler(MethodArgumentNotValidException.class)
    @ResponseBody
    public CommonResponse<?> handleMethodArgumentNotValidException(MethodArgumentNotValidException e) {
        log.error(e.getMessage(), e);
        BindingResult bindingResult = e.getBindingResult();
        StringBuilder errorMsg = new StringBuilder();
        if (bindingResult.hasErrors()) {
            List<ObjectError> errors = bindingResult.getAllErrors();
            for (ObjectError objectError : errors) {
                FieldError fieldError = (FieldError) objectError;
                if (log.isDebugEnabled()) {
                    log.error("Data check failure : object: {},field: {},errorMessage: {}",
                        fieldError.getObjectName(), fieldError.getField(), fieldError.getDefaultMessage());
                }
                errorMsg.append(fieldError.getField()).append("--").append(fieldError.getDefaultMessage());
                errorMsg.append(",");
            }
            errorMsg = new StringBuilder(errorMsg.substring(0, errorMsg.length() - 1));
        }
        return CommonResponse.failed(errorMsg + " - " + StrUtil.maxLength(TraceContext.getContextTraceId(), 8));
    }

}
