package com.xforceplus.pscc.common.intercept;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.google.common.collect.Maps;
import java.util.Map;
import javax.servlet.http.HttpServletRequest;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.AfterReturning;
import org.aspectj.lang.annotation.Before;
import org.aspectj.lang.annotation.Pointcut;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

/**
 * 切面更换成filter
 *
 * @author admin
 */
@Slf4j
//@Aspect
//@Component
public class WebLogAspect {

    ThreadLocal<Long> startTime = new ThreadLocal<>();

    @Pointcut("execution(public * com.xforceplus.ultraman..*Controller.*(..)) &&"
        + " !execution(public * com.xforceplus.ultraman.oqsengine.sdk.controller..*Controller.*(..))")
    public void webLog() {
    }

    @Before("webLog()")
    public void doBefore(JoinPoint joinPoint) throws Throwable {
        startTime.set(System.currentTimeMillis());
        try {
            //接收到请求，记录请求内容
            final ServletRequestAttributes attributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
            if (attributes == null) {
                return;
            }
            final HttpServletRequest request = attributes.getRequest();
            writeAppRequest(joinPoint, request);
        } catch (Exception ex) {
            log.error("拦截参数记录下请求内容出现异常：" + ex);
        }
    }

    @AfterReturning(returning = "ret", pointcut = "webLog()")
    public void doAfterReturning(Object ret) {
        try {
            // 处理完请求，返回内容
            final Long time = (System.currentTimeMillis() - startTime.get());
            startTime.remove();
            writeAppResponse(ret, time);
        } catch (Exception e) {
            log.error("拦截返回记录下返回内容出现异常：" + e);
        }
    }

    private void writeAppRequest(JoinPoint joinPoint, HttpServletRequest request) {

        final StringBuffer requestParm = new StringBuffer();
        if (joinPoint.getArgs() != null && joinPoint.getArgs().length > 0) {
            for (int i = 0; i < joinPoint.getArgs().length; i++) {
                requestParm.append(JSON.toJSONString(joinPoint.getArgs()[i], SerializerFeature.DisableCircularReferenceDetect));
            }
        }
        final Map<String, Object> logEntity = Maps.newHashMap();
        logEntity.put("Title", "请求报文-" + request.getMethod());
        logEntity.put("TraceId", TraceContext.getContextTraceId());
        logEntity.put("Name", request.getRequestURI().substring(request.getRequestURI().lastIndexOf("/") + 1));
        logEntity.put("Url", request.getRequestURL().toString());
        logEntity.put("Ip", request.getRemoteAddr());
        logEntity.put("RequestStr", requestParm.toString());
        log.info(JSON.toJSONString(logEntity));
    }

    private void writeAppResponse(Object ret, Long time) {
        final Map<String, Object> logEntity = Maps.newHashMap();
        logEntity.put("Title", "返回报文");
        logEntity.put("TraceId", TraceContext.getContextTraceId());
        logEntity.put("ResponseStr", ret);
        logEntity.put("Time", time);
        log.info(JSON.toJSONString(logEntity));
    }

}
