package com.xforceplus.pscc.common.janus;

import com.xforceplus.apollo.client.netty.IMessageListener;
import com.xforceplus.apollo.client.netty.MCFactory;
import com.xforceplus.apollo.client.utils.SealedMessageBuilder;
import com.xforceplus.apollo.msg.SealedMessage;
import com.xforceplus.pscc.common.bot.task.ExceptionWarning;
import com.xforceplus.pscc.common.dto.CommonResponse;
import com.xforceplus.pscc.common.exception.ApplicationException;
import com.xforceplus.pscc.common.exception.BusinessException;
import com.xforceplus.pscc.common.intercept.TraceContext;
import java.util.HashMap;
import java.util.Map;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;

/**
 * @Description:
 * @Author: zhujunjie
 * @Date: 2022/4/6 18:32
 */
@Slf4j
@Data
public class JanusListener implements IMessageListener {

    private MCFactory mcFactory;

    private ExceptionWarning exceptionWarning;

    private static JanusListener instance;

    public JanusListener(MCFactory mcFactory, ExceptionWarning exceptionWarning) {
        this.mcFactory = mcFactory;
        this.exceptionWarning = exceptionWarning;
        instance = this;
    }

    public static JanusListener getInstance() {
        return instance;
    }

    private final static Map<String, MessageHandler> handleMap = new HashMap<>();

    public void putHandleMap(String name, MessageHandler handler) {
        handleMap.put(name, handler);
    }

    @Override
    public boolean onMessage(SealedMessage sealedMessage) {
        boolean flag = true;
        SealedMessage msg = null;
        SealedMessage.Header header = sealedMessage.getHeader();
        String playLoadId = header.getPayLoadId();
        String msgId = header.getMsgId();
        final String traceId = TraceContext.putContextTraceId(msgId);
        String requestName = header.getRequestName();
        try {
            String body = sealedMessage.getPayload().getObj().toString();
            log.info("消息队列监听器接收到【{}】队列,payLoadId【{}】,msgId【{}】",
                requestName, playLoadId, msgId);
            log.info("消息内容：{}", body);
            CommonResponse response = null;

            if ("request_receipt".equals(requestName)) {
                if ("属地已接收".equals(body)) {
                    //无需处理
                    return true;
                }
                MessageHandler messageHandler = handleMap.get(header.getOthers().get("sourceRequestName"));
                if (messageHandler != null) {
                    messageHandler.receiveReceipt(sealedMessage);
                } else {
                    log.info("回执无需处理!");
                }
            } else {
                MessageHandler messageHandler = handleMap.get(requestName);
                if (messageHandler != null) {
                    response = messageHandler.handle(sealedMessage);
                } else {
                    throw new ApplicationException("无法找到【{}】队列处理规则，无法进行处理！！！", requestName);
                }
            }

            if (response != null) {
                msg = SealedMessageBuilder.buildReceiptMessage(sealedMessage, true, traceId + ":" + response.getMessage());
            }
        } catch (Exception e) {
            flag = false;
            msg = SealedMessageBuilder.buildReceiptMessage(sealedMessage, flag, traceId + ":" + e.getMessage());
            if(!(e instanceof BusinessException)) {
                exceptionWarning.januxWarn(e, requestName);
            }
            log.error("ReceiveMsgListener.onMessage() error! --- netty消息处理异常！", e);
        }

        try {
            if (msg != null && !"request_receipt".equals(requestName)) {
                mcFactory.sendMessage(msg);
            }
        } catch (Exception e) {
            exceptionWarning.januxWarn(e, requestName);
            log.error("ReceiveMsgListener.onMessage() error! --- netty发送回执异常！", e);
        }
        TraceContext.removeContextTraceId();
        return flag;
    }

}
