package com.xforceplus.pscc.common.janus;

import com.alibaba.fastjson2.JSON;
import com.alibaba.fastjson2.JSONObject;
import com.google.common.collect.Maps;
import com.xforceplus.pscc.common.exception.ApplicationException;
import com.xforceplus.pscc.common.intercept.TraceContext;
import com.xforceplus.pscc.common.janus.JanusConfig.ActionData;
import com.xforceplus.pscc.common.util.OkHttpClientUtil;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import okhttp3.Response;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

/**
 * 统一调用集成平台接口
 *
 * @author 朱俊杰
 * @version 1.0
 * @date 2022/6/31 10:35
 */
@Slf4j
@AllArgsConstructor
@Service
public class JanusServiceImpl implements JanusService {

    private static final String PREFIX = "pscc.common.janus";

    private final JanusConfig janusConfig;

    @Override
    public ActionData getActionData(ActionName actionName) {
        if (MapUtils.isEmpty(janusConfig.getActionMap())) {
            throw new ApplicationException("未配置参数:集成平台action,配置prefix:" + PREFIX + ".actionMap");
        }
        ActionData actionData = janusConfig.getActionMap().get(actionName.getKey());
        if (actionData == null) {
            throw new ApplicationException("未找到对应action,key:{},name:{}", actionName.getKey(), actionName.getName());
        }
        return actionData;
    }

    /**
     * 根据请求参数请求，serialNo为traceId
     *
     * @param actionName 请求action key
     * @param body       请求参数
     * @return 结果
     */
    @Override
    public JSONObject send(ActionName actionName, Map<String, Object> body) {
        final ActionData actionData = getActionData(actionName);
        return send(actionData, actionName, null, null, JSON.toJSONString(body), TraceContext.getContextTraceId(), JSONObject.class);
    }

    /**
     * 根据请求参数请求，serialNo为traceId （自定义返回类型）
     *
     * @param actionName 请求action key
     * @param body       请求参数
     * @return 结果
     */
    @Override
    public <T> T send(ActionName actionName, Map<String, Object> body, Class<T> tClass) {
        final ActionData actionData = getActionData(actionName);
        return send(actionData, actionName, null, null, JSON.toJSONString(body), TraceContext.getContextTraceId(), tClass);
    }


    /**
     * 根据请求参数请求，传入流水号
     *
     * @param actionName 请求action key
     * @param body       请求参数
     * @param serialNo   传入流水号
     * @return 结果
     */
    @Override
    public JSONObject send(ActionName actionName, Map<String, Object> body, String serialNo) {
        final ActionData actionData = getActionData(actionName);
        return send(actionData, actionName, null, null, JSON.toJSONString(body), serialNo, JSONObject.class);
    }

    /**
     * 根据请求参数请求，传入流水号 （自定义返回类型）
     *
     * @param actionName 请求action key
     * @param body       请求参数
     * @param serialNo   传入流水号
     * @return 结果
     */
    @Override
    public <T> T send(ActionName actionName, Map<String, Object> body, String serialNo, Class<T> tClass) {
        final ActionData actionData = getActionData(actionName);
        return send(actionData, actionName, null, null, JSON.toJSONString(body), serialNo, tClass);
    }

    /**
     * 全参数 返回JSONObject
     *
     * @param actionName 请求action key
     * @param header     请求头
     * @param param      url请求参数
     * @param body       请求体
     * @param serialNo   传入流水号
     * @return 结果 JSONObject
     */
    @Override
    public JSONObject send(ActionName actionName, Map<String, String> header,
        Map<String, String> param, Map<String, Object> body, String serialNo) {
        final ActionData actionData = getActionData(actionName);
        return send(actionData, actionName, header, param, JSON.toJSONString(body), serialNo, JSONObject.class);
    }

    /**
     * （自定义返回类型）
     *
     * @param actionName 请求action key
     * @param headerMap  请求头
     * @param param      url请求参数
     * @param body       请求体
     * @param serialNo   传入流水号
     * @param tClass     返回类型
     * @return 结果
     */
    @Override
    public <T> T send(ActionName actionName, Map<String, String> headerMap,
        Map<String, String> param, Map<String, Object> body, String serialNo, Class<T> tClass) {
        final ActionData actionData = getActionData(actionName);
        return send(actionData, actionName, headerMap, param, JSON.toJSONString(body), serialNo, tClass);
    }

    /**
     * （自定义返回类型） body为原始字符串
     *
     * @param actionName 请求action key
     * @param headerMap  请求头
     * @param param      url请求参数
     * @param body       请求体
     * @param serialNo   传入流水号
     * @param tClass     返回类型
     * @return 结果
     */
    @Override
    public <T> T send(ActionName actionName, Map<String, String> headerMap,
        Map<String, String> param, String body, String serialNo, Class<T> tClass) {
        final ActionData actionData = getActionData(actionName);
        return send(actionData, actionName, headerMap, param, body, serialNo, tClass);
    }

    private <T> T send(ActionData actionData, ActionName actionName, Map<String, String> header,
        Map<String, String> param, String body, String serialNo, Class<T> tClass) {

        check();

        Map<String, String> headerMap = header;
        Response resp;
        String url = janusConfig.getHttpUrl();
        //String bodyStr = JSON.toJSONString(body);
        try {
            if (CollectionUtils.isEmpty(header)) {
                headerMap = new HashMap<>();
            }
            headerMap.put("rpcType", "http");
            headerMap.put("Authentication", janusConfig.getAuthentication());
            headerMap.put("action", actionData.getAction());
            headerMap.put("Accept-Encoding", "deflate");
            serialNo = StringUtils.isBlank(serialNo) ? UUID.randomUUID().toString() : serialNo;
            headerMap.put("serialNo", serialNo);
            if (StringUtils.isBlank(headerMap.get("tenantId"))) {
                headerMap.put("tenantId", janusConfig.getTenantId());
            }
            if (StringUtils.isBlank(headerMap.get("tenantCode"))) {
                headerMap.put("tenantCode", janusConfig.getTenantCode());
            }
            if (StringUtils.isBlank(headerMap.get("uiaSign"))) {
                headerMap.put("uiaSign", janusConfig.getUiaSign());
            }
            log.info("调用集成平台接口 serialNo:[{}] ,调用地址:[{}],调用方法名:[{}],参数:{},header:{}", serialNo, url, actionName.getName(), body, headerMap);
            //去除空字段
            Set<String> keySet = new HashSet<>(headerMap.keySet());
            for (String key : keySet) {
                if (headerMap.get(key) == null) {
                    headerMap.remove(key);
                }
            }

            if (StringUtils.equals("get", actionData.getMethod())) {
                Map<String, Object> bodyMap = Maps.newHashMap();
                if (StringUtils.isNotBlank(body)) {
                    if (!JSON.isValid(body)) {
                        throw new ApplicationException("格式错误！请传入正确得消息体,body:" + body);
                    }
                    bodyMap = JSON.parseObject(body);
                }
                if (MapUtils.isNotEmpty(param)) {
                    bodyMap.putAll(param);
                }
                resp = OkHttpClientUtil.doGet(janusConfig.getHttpUrl(), headerMap, bodyMap);
            } else {
                resp = OkHttpClientUtil.doPost(janusConfig.getHttpUrl(), headerMap, param, body);

            }
            if (resp.body() == null) {
                throw new ApplicationException("返回值为空");
            }

            String message = resp.body().string();

            if (StringUtils.isBlank(message)) {
                throw new ApplicationException("返回值为空");
            }

            if (!resp.isSuccessful()) {
                throw new ApplicationException("返回失败!返回码:{},message:{}", resp.code(), message);
            }

            log.info("调用集成平台接口成功 serialNo:[{}] ,调用地址:[{}],调用方法名:[{}],返回值:{}", serialNo, url, actionName.getName(), message);

            return JSONObject.parseObject(message, tClass);
        } catch (Exception e) {
            log.error("调用集成平台接口失败 serialNo:[{}] ,调用地址:[{}],调用方法名:[{}],错误原因:{}", serialNo, url, actionName.getName(), e.getMessage());
            throw new ApplicationException(e);
        }
    }

    private void check() {
        if (StringUtils.isBlank(janusConfig.getHttpUrl())) {
            throw new ApplicationException("未配置参数:集成平台url,配置prefix:" + PREFIX + ".httpUrl");
        }
        if (StringUtils.isBlank(janusConfig.getAuthentication())) {
            throw new ApplicationException("未配置参数:集成平台authentication,配置prefix:" + PREFIX + ".authentication");
        }
    }


    public static void main(String[] args) {
        String a = "{";
        System.out.println(JSON.isValid(a));
    }

}
