package com.xforceplus.pscc.common.wapper;

import cn.hutool.core.date.SystemClock;
import com.xforceplus.pscc.common.exception.ApplicationException;
import com.xforceplus.ultraman.oqsengine.pojo.dto.entity.IEntityClass;
import com.xforceplus.ultraman.oqsengine.sdk.business.meta.EntityId;
import com.xforceplus.ultraman.oqsengine.sdk.business.meta.EntityInstance;
import com.xforceplus.ultraman.oqsengine.sdk.business.meta.service.BusinessFacade;
import com.xforceplus.ultraman.oqsengine.sdk.query.dsl.ExpFactory;
import com.xforceplus.ultraman.oqsengine.sdk.service.EntityService;
import com.xforceplus.ultraman.oqsengine.sdk.transactional.annotation.OqsTransactional;
import com.xforceplus.ultraman.oqsengine.sdk.util.RequestBuilder;
import com.xforceplus.ultraman.oqsengine.sdk.vo.dto.ConditionOp;
import com.xforceplus.ultraman.oqsengine.sdk.vo.dto.ConditionQueryRequest;
import com.xforceplus.ultraman.oqsengine.sdk.vo.dto.FieldCondition;
import io.vavr.control.Either;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.Callable;
import java.util.function.Function;
import java.util.stream.Collectors;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Component;

@Component
@Slf4j(topic = "ULTRAMAN")
public class EntityMapperWrapper implements EntityMapper, ApplicationContextAware {

    private EntityMapperWrapper entityMapperWrapper;
    @Autowired(required = false)
    private BusinessFacade businessFacade;
    @Autowired(required = false)
    private EntityService entityService;

    @Override
    public void setApplicationContext(@NotNull ApplicationContext applicationContext) throws BeansException {
        Map<String, EntityMapperWrapper> map = applicationContext.getBeansOfType(EntityMapperWrapper.class);
        map.forEach((key, value) -> entityMapperWrapper = value);
    }

    @Override
    public IEntityClass getEntityClass(String entityCode) {
        return businessFacade.load(entityCode);
    }

    @Override
    public IEntityClass getEntityClass(EntityEnum entityEnum) {
        return businessFacade.load(entityEnum.getValue());
    }

    @Override
    public <T> T transactionalExecute(Callable<T> callable) {
        final Either<String, T> either = entityService.transactionalExecute(callable);
        return either.getOrElseThrow((Function<String, ApplicationException>) ApplicationException::new);
    }

    @Override
    public List<IEntityClass> getEntityClasss() {
        return entityService.getEntityClasss();
    }

    @Override
    public Long create(EntityEnum entityEnum, Map<String, Object> entity) {
        final IEntityClass entityClass = getEntityClass(entityEnum.getValue());
        Long id = businessFacade.create(entityClass, entity);
        if (id == null || id <= 0) {
            throw new ApplicationException("保存失败！id：" + id);
        }
        return id;
    }

    /**
     * 循环创建 保证强一致 创建后可以立刻查询
     *
     * @param entityEnum 表
     * @param entityList 参数
     * @return 数量
     */
    @Override
    @OqsTransactional(rollbackFor = {Exception.class})
    public Integer createLoop(EntityEnum entityEnum, List<Map<String, Object>> entityList) {
        if (CollectionUtils.isEmpty(entityList)) {
            throw new ApplicationException("保存内容不能为空");
        }
        final IEntityClass entityClass = getEntityClass(entityEnum.getValue());
        entityList.forEach(entity -> {
            Long id = businessFacade.create(entityClass, entity);
            if (id == null || id <= 0) {
                throw new ApplicationException("保存失败！id：" + id);
            }
        });
        return entityList.size();
    }

    /**
     * 批量创建 未保证强一致 创建后无法立刻查询
     *
     * @param entityEnum 表
     * @param entityList 参数
     * @return 数量
     */
    @Override
    public Integer createMulti(EntityEnum entityEnum, List<Map<String, Object>> entityList) {
        if (CollectionUtils.isEmpty(entityList)) {
            throw new ApplicationException("保存内容不能为空");
        }
        final IEntityClass entityClass = getEntityClass(entityEnum.getValue());
        Integer count = businessFacade.createMulti(entityClass, entityList);
        if (count == null || count <= 0 || count != entityList.size()) {
            throw new ApplicationException("保存失败！");
        }
        return count;
    }

    @Override
    public void deleteById(EntityEnum entityEnum, long id) {
        final IEntityClass entityClass = getEntityClass(entityEnum.getValue());
        Integer i = businessFacade.deleteOne(new EntityId(entityClass, id));
        if (i == null || i < 1) {
            throw new ApplicationException("删除失败！id：" + id);
        }
    }

    @Override
    public Integer deleteByIds(EntityEnum entityEnum, List<Long> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            throw new ApplicationException("查询参数不能为空");
        }
        final IEntityClass entityClass = getEntityClass(entityEnum.getValue());
        return businessFacade.deleteMulti(ids.stream().map(id -> new EntityId(entityClass, id)).collect(Collectors.toList()));
    }

    @Override
    public Integer deleteByCondition(EntityEnum entityEnum, RequestBuilder requestBuilder) {
        if (requestBuilder == null) {
            throw new ApplicationException("查询参数不能为空");
        }
        ConditionQueryRequest request = requestBuilder.build();
        if (CollectionUtils.isEmpty(request.getConditions().getFields())) {
            throw new ApplicationException("查询参数不能为空");
        }
        if (null == request.getPageNo()) {
            request.setPageNo(1);
        }
        if (null == request.getPageSize()) {
            request.setPageSize(10000);
        }
        final IEntityClass entityClass = getEntityClass(entityEnum.getValue());
        return businessFacade.deleteByConditionByBatch(entityClass, ExpFactory.createFrom(request));
    }

    @Override
    public void updateById(EntityEnum entityEnum, long id, Map<String, Object> entity) {
        if (MapUtils.isEmpty(entity)) {
            throw new ApplicationException("参数不能为空");
        }
        final IEntityClass entityClass = getEntityClass(entityEnum.getValue());
        entity.put("update_time", SystemClock.now());
        Integer i = businessFacade.updateById(new EntityId(entityClass, id), entity);
        if (i == null || i < 1) {
            throw new ApplicationException("更新失败！id：" + id);
        }
    }

    /**
     * 循环更新  保证强一致 更新后可以立刻查询
     *
     * @param entityEnum 表
     * @param entity     参数
     * @return 数量
     */
    @OqsTransactional(rollbackFor = {Exception.class})
    @Override
    public Integer updateByIds(EntityEnum entityEnum, List<Long> ids, Map<String, Object> entity) {
        if (CollectionUtils.isEmpty(ids)) {
            throw new ApplicationException("查询参数不能为空");
        }
        if (MapUtils.isEmpty(entity)) {
            throw new ApplicationException("参数不能为空");
        }
        final IEntityClass entityClass = getEntityClass(entityEnum.getValue());
        ids.forEach(id -> {
            Integer i = businessFacade.updateById(new EntityId(entityClass, id), entity);
            if (i == null || i < 1) {
                throw new ApplicationException("更新失败！id：" + id);
            }
        });
        return ids.size();
    }

    /**
     * 批量更新 未保证强一致 更新后无法立刻查询
     *
     * @param entityEnum 表
     * @param entityList 参数
     * @return 数量
     */
    @Override
    public Integer updateMulti(EntityEnum entityEnum, List<Map<String, Object>> entityList) {
        if (CollectionUtils.isEmpty(entityList)) {
            throw new ApplicationException("参数不能为空");
        }
        final IEntityClass entityClass = getEntityClass(entityEnum.getValue());
        return businessFacade.updateMulti(entityClass, entityList);
    }

    /**
     * 根据条件更新  保证强一致 更新后可以立刻查询
     *
     * @param entityEnum 表
     * @param entity     参数
     * @return 数量
     */
    @Override
    public Integer updateByCondition(EntityEnum entityEnum, RequestBuilder requestBuilder, Map<String, Object> entity) {
        if (MapUtils.isEmpty(entity)) {
            throw new ApplicationException("参数不能为空");
        }
        List<Long> ids = findIdsByCondition(entityEnum, requestBuilder);
        if (CollectionUtils.isEmpty(ids)) {
            return 0;
        }
        if (ids.size() == 1) {
            updateById(entityEnum, ids.get(0), entity);
            return 1;
        } else {
            return entityMapperWrapper.updateByIds(entityEnum, ids, entity);
        }
    }

    @Override
    public Integer replaceById(EntityEnum entityEnum, long id, Map<String, Object> entity) {
        if (MapUtils.isEmpty(entity)) {
            throw new ApplicationException("参数不能为空");
        }
        final IEntityClass entityClass = getEntityClass(entityEnum.getValue());
        entity.put("update_time", SystemClock.now());
        return businessFacade.replaceById(new EntityId(entityClass, id), entity);
    }

    @Override
    public Map<String, Object> findOne(EntityEnum entityEnum, long id) {
        IEntityClass entityClass = getEntityClass(entityEnum.getValue());
        final Optional<EntityInstance> instance = businessFacade.findOne(new EntityId(entityClass, id));
        return instance.map(EntityInstance::value).orElse(null);
    }

    @Override
    public <T> T findOne(EntityEnum entityEnum, long id, Function<Map<String, Object>, T> toEntityFunction) {
        return toEntityFunction.apply(findOne(entityEnum, id));
    }

    @Override
    public List<Map<String, Object>> findByIds(EntityEnum entityEnum, List<Long> ids) {
        if (CollectionUtils.isEmpty(ids)) {
            throw new ApplicationException("查询参数不能为空");
        }
        RequestBuilder requestBuilder = new RequestBuilder().field("id", ConditionOp.in, ids);
        return findByCondition(entityEnum, requestBuilder);
    }

    @Override
    public <T> List<T> findByIds(EntityEnum entityEnum, List<Long> ids, Function<Map<String, Object>, T> toEntityFunction) {
        final List<Map<String, Object>> condition = findByIds(entityEnum, ids);
        return condition.stream().map(toEntityFunction).collect(Collectors.toList());
    }

    @Override
    public Long count(EntityEnum entityEnum, RequestBuilder requestBuilder) {
        ConditionQueryRequest request = requestBuilder.build();
        if (null == request.getPageNo()) {
            request.setPageNo(1);
        }
        if (null == request.getPageSize()) {
            request.setPageSize(10000);
        }

        IEntityClass entityClass = getEntityClass(entityEnum.getValue());
        return businessFacade.count(entityClass, ExpFactory.createFrom(request));
    }

    @Override
    public List<Map<String, Object>> findByCondition(EntityEnum entityEnum, Map<String, Object> entityMap) {
        if (MapUtils.isEmpty(entityMap)) {
            throw new ApplicationException("查询条件不能为空!");
        }
        RequestBuilder requestBuilder = new RequestBuilder();
        entityMap.forEach((key, value) -> {
            requestBuilder.field(key, ConditionOp.eq, value);
        });
        return findByCondition(entityEnum, requestBuilder);
    }

    @Override
    public List<Map<String, Object>> findByCondition(EntityEnum entityEnum, RequestBuilder requestBuilder) {

        ConditionQueryRequest request = requestBuilder.build();
        if (null == request.getPageNo()) {
            request.setPageNo(1);
        }
        if (null == request.getPageSize()) {
            request.setPageSize(10000);
        }
        List<FieldCondition> fields = request.getConditions().getFields();
        if (CollectionUtils.isNotEmpty(fields)) {
            //过滤eq ne空串
            request.getConditions().setFields(filterBlank(fields));
        }

        if (CollectionUtils.isEmpty(request.getConditions().getFields()) && CollectionUtils.isEmpty(request.getConditions().getEntities())) {
            throw new ApplicationException("查询参数不能为空");
        }

        final IEntityClass entityClass = getEntityClass(entityEnum.getValue());
        return businessFacade.findByCondition(entityClass, ExpFactory.createFrom(request)).getRows().stream().map(EntityInstance::value).collect(Collectors.toList());
    }

    public List<FieldCondition> filterBlank(List<FieldCondition> list) {
        if (null == list) {
            return list;
        }

        List<FieldCondition> list2 = new ArrayList<>();
        for (FieldCondition field : list) {
            if (ConditionOp.eq.name().equals(field.getOperation().name()) ||
                ConditionOp.ne.name().equals(field.getOperation().name())) {
                if (CollectionUtils.isEmpty(field.getValue()) || field.getValue().size() > 1) {
                    throw new ApplicationException(field.getCode() + "的参数错误!");
                } else {
                    if (StringUtils.isNotBlank(field.getValue().get(0))) {
                        list2.add(field);
                    }
                }
            } else {
                list2.add(field);
            }
        }
        return list2;
    }

    @Override
    public List<Long> findIdsByCondition(EntityEnum entityEnum, RequestBuilder requestBuilder) {
        requestBuilder.item("id");
        List<Map<String, Object>> result = findByCondition(entityEnum, requestBuilder);
        return result.stream().map(map -> MapUtils.getLong(map, "id")).collect(Collectors.toList());
    }

    @Override
    public <T> List<T> findByCondition(EntityEnum entityEnum, RequestBuilder requestBuilder, Function<Map<String, Object>, T> toEntityFunction) {
        final List<Map<String, Object>> condition = findByCondition(entityEnum, requestBuilder);
        return condition.stream().map(toEntityFunction).collect(Collectors.toList());
    }

    @Override
    public <T> T findOneByCondition(EntityEnum entityEnum, RequestBuilder requestBuilder, Function<Map<String, Object>, T> toEntityFunction) {
        requestBuilder.pageSize(1);
        requestBuilder.pageNo(1);
        final List<Map<String, Object>> condition = findByCondition(entityEnum, requestBuilder);
        if (CollectionUtils.isEmpty(condition)) {
            return null;
        } else {
            return toEntityFunction.apply(condition.get(0));
        }
    }

}
