package com.xforceplus.purconfig.client.util;

import com.xforceplus.purconfig.client.model.config.item.RecogCustomFieldBean;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.StringUtils;

import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

/**
 * Describe: 自定义字段参数验证(目前影像采集 、发票管理都有使用)
 *
 * @Author xiezhongyong
 * @Date 2019/9/20
 * @Version 1.0
 */
@Slf4j
public class CheckCustomFieldUtil {


    public static final String VERIFY_BIZ_TAG_ERR = "自定义字段不符合配置规则";
    public static final String GET_RECOG_CUSTOM_FIELD_ERR = "自定义字段配置获取异常";
    public static final String RECOG_CUSTOM_FIELD_STATE_ERR = "自定义字段配置未启用";
    public static final String RECOG_CUSTOM_FIELD_LENGTH_ERR = "自定义字段内存超长(最大200)";

    /**
     * 分隔符
     */
    public static final String SEPARATOR = ",";
    /**
     * 日期正则表达式 格式：yyyyMMdd
     **/
    public static final String DATE_TIME8_REGEX = "^(?:(?!0000)[0-9]{4}(?:(?:0[1-9]|1[0-2])(?:0[1-9]|1[0-9]|2[0-8])|(?:0[13-9]|1[0-2])(?:29|30)|(?:0[13578]|1[02])31)|(?:[0-9]{2}(?:0[48]|[2468][048]|[13579][26])|(?:0[48]|[2468][048]|[13579][26])00)0229)$";
    /**
     * 字段内容最大长度
     */
    public static final int FIELD_MAX_LENGTH = 200;


    /**
     * 校验自定义字段 内容 是否符合规则配置
     *
     * @param values
     * @param customFieldBean
     */
    public static void checkRecogCustomField(List<String> values, RecogCustomFieldBean customFieldBean) {

        log.info("##### checkRecogCustomField input param --> values：{}；customFieldBean：{}", values, customFieldBean);
        if (null == customFieldBean) {
            log.warn("##### 读取 自定义字段配置：{}; 参数检查失败", customFieldBean);
            throw new RuntimeException(GET_RECOG_CUSTOM_FIELD_ERR);
        }

        // 判断是否启用自定义字段
        boolean state = RecogCustomFieldBean.StateEnum.NO == customFieldBean.getEnabledFlag();
        if (state && null != values) {
            for (String value : values) {
                // 如果是未启用并且 默认值不为空 则报错
                if (!StringUtils.isEmpty(value)) {
                    log.warn("##### 自定义字段未开启，values：{}; 参数检查失败", values);
                    throw new RuntimeException(RECOG_CUSTOM_FIELD_STATE_ERR);
                }
            }
        }

        List<RecogCustomFieldBean.FieldInfo> fieldList = customFieldBean.getFieldList();

        // 理论上前端每次 list会有3个值（默认：空字符），如果size 小于 规则字段数量 报错
        // 如果默认值 数量超过 3个报错
        if (values.size() < fieldList.size() || values.size() > 3) {
            log.warn("##### values 数量：{}；自定义字段数量：{}; 参数检查失败", values.size(), fieldList.size());
            throw new RuntimeException(VERIFY_BIZ_TAG_ERR);
        }

        for (int i = 0; i < values.size(); i++) {
            RecogCustomFieldBean.FieldInfo fieldInfo = null;
            // 避免 自定义字段2个，values 传默认值有 3个的情况
            if (i < fieldList.size()) {
                fieldInfo = fieldList.get(i);
            }
            // 获取自定义字段value 并且去除空格
            String value = values.get(i).trim();

            if (value.length() > FIELD_MAX_LENGTH) {
                log.warn("##### value 长度：{}；最大长度：{} 参数检查失败", value.length(), FIELD_MAX_LENGTH);
                throw new RuntimeException(RECOG_CUSTOM_FIELD_LENGTH_ERR);
            }

            boolean empty = StringUtils.isEmpty(value);

            // 自定义字段未配置，前端却传值的情况
            if (null == fieldInfo && !empty) {
                log.warn("##### 未查询到自定义字段配置，value:{}; 参数检查失败", value);
                throw new RuntimeException(VERIFY_BIZ_TAG_ERR);
            }

            // 判断是否必填
            boolean requiredFlag = null != fieldInfo && RecogCustomFieldBean.StateEnum.YES == fieldInfo.getRequiredFlag() ;
            if (requiredFlag && empty) {
                log.warn("##### 自定义字段配置requiredFlag：{}，value:{} ；参数必填； 检查失败", fieldInfo.getRequiredFlag(), value);
                throw new RuntimeException(VERIFY_BIZ_TAG_ERR);
            }

            // 非必填并且为空直接调过本次循环
            if (empty) {
                log.warn("##### 自定义字段配置value:{} ；参数非必填； 跳过本次循环", value);
                continue;
            }

            RecogCustomFieldBean.FieldTypeEnum fieldType = fieldInfo.getFieldType();

            switch (fieldType) {
                case RADIO:
                    checkRadio(fieldInfo, value);
                    break;
                case CHECKBOX:
                    checkCheckBox(fieldInfo, value);
                    break;
                case DATETIME:
                    checkDatetime(value);
                    break;
                case TEXT:
                    // 文本目前暂时没有验证
                    break;
                default:
                    break;
            }

        }

    }

    /**
     * 检查单选 参数合法
     *
     * @param fieldInfo
     * @param value
     */
    private static void checkRadio(RecogCustomFieldBean.FieldInfo fieldInfo, String value) {
        List<RecogCustomFieldBean.ValueInfo> valueInfos = fieldInfo.getValueList();
        List<String> checkValues = valueInfos.stream().map(v -> v.getValue()).collect(Collectors.toList());

        if (!checkValues.contains(value)) {
            log.warn("##### value:{}， checkValues：{}; RADIO 参数检查失败", value, checkValues);
            throw new RuntimeException(VERIFY_BIZ_TAG_ERR);
        }
    }

    /**
     * 检查复选 参数合法
     *
     * @param fieldInfo
     * @param value
     */
    private static void checkCheckBox(RecogCustomFieldBean.FieldInfo fieldInfo, String value) {
        List<RecogCustomFieldBean.ValueInfo> valueInfos = fieldInfo.getValueList();
        List<String> checkValues = valueInfos.stream().map(v -> v.getValue()).collect(Collectors.toList());
        List<String> valueList = Arrays.asList(value.split(SEPARATOR)).stream().map(s -> (s.trim())).collect(Collectors.toList());

        // 判断是否有重复值
        if (valueList.stream().distinct().count() < valueList.size()) {
            log.warn("##### value:{}， checkValues：{}; 数值重复，CHECKBOX 参数检查失败", value, checkValues);
            throw new RuntimeException(VERIFY_BIZ_TAG_ERR);
        }

        for (String val : valueList) {
            if (!checkValues.contains(val)) {
                log.warn("##### value:{}， checkValues：{}; CHECKBOX 参数检查失败", value, checkValues);
                throw new RuntimeException(VERIFY_BIZ_TAG_ERR);
            }
        }
    }

    /**
     * 检查日期 参数合法
     *
     * @param value
     */
    private static void checkDatetime(String value) {
        //日期格式 20190909
        boolean matches = value.matches(DATE_TIME8_REGEX);
        if (!matches) {
            log.warn("##### value:{}; DATETIME 参数检查失败", value);
            throw new RuntimeException(VERIFY_BIZ_TAG_ERR);
        }
    }

}
