package com.xforceplus.ant.coop.rule.center.client.data.cc.tool;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.xforceplus.ant.coop.rule.center.client.api.cc.TemplateCommonConfigApi;
import com.xforceplus.ant.coop.rule.center.client.data.cc.bean.fieldlimit.*;
import com.xforceplus.ant.coop.rule.center.client.data.cc.enums.FieldLimitEnum;
import com.xforceplus.ant.coop.rule.center.client.data.cc.enums.FixedEnumValueEnum;
import com.xforceplus.ant.coop.rule.center.client.data.cc.enums.LimitStatusEnum;
import com.xforceplus.ant.coop.rule.center.client.data.cc.exception.ValidateException;
import com.xforceplus.ant.coop.rule.center.client.data.cc.request.CheckInvalidCondition;
import com.xforceplus.ant.coop.rule.center.client.data.cc.response.RuleTemplateDetail;
import com.xforceplus.ant.coop.rule.center.client.data.utils.BaseResult;
import com.xforceplus.ant.coop.rule.center.client.utils.FieldLimitUtils;
import com.xforceplus.ant.coop.rule.center.common.util.ReflectUtils;
import com.xforceplus.xplatframework.exception.ResultCode;
import com.xforceplus.xplatframework.utils.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * Describe: 模板配置通用处理工具类
 *
 * @Author xiezhongyong
 * @Date 2020-06-12
 */
@Slf4j
public class CommonConfigTool {


    /**
     * 条件表达式执行
     *
     * @param api
     * @param exp    表达式
     * @param params 条件
     * @return true: 匹配成功，false：匹配失败/非法表达式等
     */
    public static boolean checkConditionExp(TemplateCommonConfigApi api, String exp, Map<String, Object> params) {
        CheckInvalidCondition condition = new CheckInvalidCondition();
        condition.setConditionExp(exp);
        condition.setParams(params);
        try {
            BaseResult<String> response = api.checkConditionExp(condition);
            log.info("##### 条件表达式执行 响应：{}", JsonUtils.writeObjectToFastJson(response));
            if (ResultCode.SUCCESS.code().equals(response.getCode())) {
                return true;
            }
            return false;
        } catch (Exception e) {
            log.warn("##### 条件表达式执行异常：{}", e);
            return false;
        }
    }

    /**
     * 验证字段限制失效条件
     *
     * @param api
     * @param fieldLimit
     * @param params
     * @return fales：失效条件不成立，true: 失效条件成立
     */
    public static boolean checkInvalidCondition(TemplateCommonConfigApi api, FieldLimit fieldLimit, Map<String, Object> params) {
        // 判断 limit 是否具备 invalidStatus，invalidCondition 字段
        if (null == ReflectUtils.getDeclaredMethod(fieldLimit, "getInvalidCondition") || null == ReflectUtils.getDeclaredMethod(fieldLimit, "getInvalidStatus")) {
            log.warn("##### fieldLimit:{}；不存在失效条件：", fieldLimit);
            return false;
        }
        // 失效状态验证
        Object invalidStatus = ReflectUtils.getValue(fieldLimit, "invalidStatus");
        if (null == invalidStatus || !LimitStatusEnum.ENABLED.code().equals(invalidStatus)) {
            log.info("##### fieldLimit:{}；invalidStatus is null/ invalidStatus 已停用：", fieldLimit);
            return false;
        }

        // 失效条件验证
        Object invalidCondition = ReflectUtils.getValue(fieldLimit, "invalidCondition");
        if (null == invalidCondition || StringUtils.isEmpty(invalidCondition.toString())) {
            log.info("##### fieldLimit:{}；invalidCondition is null/为空：", fieldLimit);
            return false;
        }
        // 进行 条件表达式验证
        return checkConditionExp(api, invalidCondition.toString(), params);
    }

    /**
     * 验证 字段限制是否启用和失效条件验证
     *
     * @param fieldLimits
     * @param fieldLimitEnum
     * @return true: 字段限制启用&失效条件不成立; false: 字段限制停用/字段限制不存/字段限制启用但是失效条件处理
     */
    public static boolean checkLimitAntInvalidCondition(TemplateCommonConfigApi api, List<JSONObject> fieldLimits, FieldLimitEnum fieldLimitEnum, Map<String, Object> params) {
        boolean checkLimitAntStatus = checkLimitAntStatus(fieldLimits, fieldLimitEnum);
        if (!checkLimitAntStatus) {
            // 未启用或者不存在，返回false
            return checkLimitAntStatus;
        }
        // 只有 limit 存在并且启用才进行失效条件验证
        FieldLimit limit = FieldLimitUtils.getFieldLimitByJsonObjects(fieldLimits, fieldLimitEnum);
        if (checkInvalidCondition(api, limit, params)) {
            // 失效条件成立说明当前限制失效了，直接返回false
            return false;
        }
        return true;
    }

    /**
     * 验证 字段限制是否启用
     *
     * @param fieldLimits
     * @param fieldLimitEnum
     * @return true: 字段限制存在&启用；false: 字段限制不存在/存在并且停用
     */
    public static boolean checkLimitAntStatus(List<JSONObject> fieldLimits, FieldLimitEnum fieldLimitEnum) {
        Object limit = FieldLimitUtils.getFieldLimitByJsonObjects(fieldLimits, fieldLimitEnum);
        if (null == limit) {
            return false;
        }
        // 验证状态
        if (null != ReflectUtils.getDeclaredMethod(limit, "getStatus")) {
            Object status = ReflectUtils.getValue(limit, "status");
            if (LimitStatusEnum.ENABLED.code().equals(status.toString())) {
                return true;
            }
        } else {
            // 无status 对象只要limit不为空即可
            return true;
        }

        return false;
    }

    /**
     * 批量参数非空验证（同时执行失效条件验证）PS: 如果主表明细同时出现的配置中字段重复会有问题
     *
     * @param items           字段限制
     * @param conditionParams 需要验证的字段 & 条件填充参数&需要验证的字段
     * @return
     */
    public static void validateNoEmpty(TemplateCommonConfigApi api, List<RuleTemplateDetail.ConfigItem> items, Map<String, Object> conditionParams) {
        validateNoEmpty(api, items, conditionParams, conditionParams);
    }

    /**
     * 批量参数非空验证（同时执行失效条件验证）
     *
     * @param items           字段限制
     * @param validateFields  需要验证的字段
     * @param conditionParams 条件填充参数
     * @return
     */
    public static void validateNoEmpty(TemplateCommonConfigApi api, List<RuleTemplateDetail.ConfigItem> items, Map<String, Object> validateFields, Map<String, Object> conditionParams) {
        if (CollectionUtils.isEmpty(items) || CollectionUtils.isEmpty(validateFields)) {
            return;
        }
        for (RuleTemplateDetail.ConfigItem item : items) {
            if (!validateFields.containsKey(item.getFieldName())) {
                continue;
            }
            // 验证NoEmpty 是否启用有效，如果启用的情况先检查是否有失效条件，失效条件是否成立（成立返回false 等同于NoEmpty 不起任何作用）
            if (!checkLimitAntInvalidCondition(api, item.getFieldLimits(), FieldLimitEnum.NO_EMPTY, conditionParams)) {
                // 字段未配置 noEmpty/失效条件成立 跳过
                continue;
            }
            Object valueObj = validateFields.get(item.getFieldName());
            if (null == valueObj) {
                throw new ValidateException(item.getFieldDisplayName() + "不可为空");
            }
            String fieldValue = valueObj.toString();
            if (fieldValue.startsWith("[") && fieldValue.endsWith("]")) {
                List<String> fieldValues = JSON.parseArray(fieldValue, String.class);
                fieldValue = CollectionUtils.isEmpty(fieldValues) ? "" : fieldValues.get(0);
            }
            if (StringUtils.isEmpty(fieldValue)) {
                String message = item.getFieldDisplayName() + "不可为空";
                log.warn("##### 根据模板进行非空校验--校验未通过！ validateResult:{}", message);
                throw new ValidateException(message);
            }
        }
    }


    /**
     * 批量参数禁改验证+字段值是否和禁改默认值一致（同时执行失效条件验证）PS: 如果主表明细同时出现的配置中字段重复会有问题
     *
     * @param items           字段限制
     * @param conditionParams 需要验证的字段 & 条件填充参数
     * @return
     */
    public static void validateNoChangeFixedValue(TemplateCommonConfigApi api, List<RuleTemplateDetail.ConfigItem> items, Map<String, Object> conditionParams) {
        validateNoChangeFixedValue(api, items, conditionParams, conditionParams);
    }

    /**
     * 批量参数禁改验证+字段值是否和禁改默认值一致（同时执行失效条件验证）
     *
     * @param items           字段限制
     * @param validateFields  需要验证的字段
     * @param conditionParams 条件填充参数
     * @return
     */
    public static void validateNoChangeFixedValue(TemplateCommonConfigApi api, List<RuleTemplateDetail.ConfigItem> items, Map<String, Object> validateFields, Map<String, Object> conditionParams) {
        if (CollectionUtils.isEmpty(items) || CollectionUtils.isEmpty(validateFields)) {
            return;
        }
        for (RuleTemplateDetail.ConfigItem item : items) {
            if (!validateFields.containsKey(item.getFieldName())) {
                continue;
            }
            // 验证NoChange 是否启用有效，如果启用的情况先检查是否有失效条件，失效条件是否成立（成立返回false 等同于NoEmpty 不起任何作用）
            if (!checkLimitAntInvalidCondition(api, item.getFieldLimits(), FieldLimitEnum.NO_CHANGE, conditionParams)) {
                // 字段未配置 NoChange/失效条件成立 跳过
                log.info("##### 字段未配置 NoChange/失效条件成立 跳过：{}", JsonUtils.writeObjectToFastJson(item));
                continue;
            }

            // 业务字段值
            String fieldValue = null == validateFields.get(item.getFieldName()) ? null : validateFields.get(item.getFieldName()).toString();
            // 默认值获取
            DefaultValueLimit defaultValueLimit = FieldLimitUtils.getFieldLimitByJsonObjects(item.getFieldLimits(), FieldLimitEnum.DEFAULT_VALUE);
            if (null != defaultValueLimit && null != defaultValueLimit.getDefaultValue() && null != fieldValue) {
                log.info("#####根据模板进行固定值NoChange校验-defaultValueLimit-开始校验 modelField：{}", JSON.toJSONString(item));
                if (fieldValue.startsWith("[") && fieldValue.endsWith("]")) {
                    List<String> fieldValues = JSON.parseArray(fieldValue, String.class);
                    fieldValue = CollectionUtils.isEmpty(fieldValues) ? "" : fieldValues.get(0);
                }
                if (!defaultValueLimit.getDefaultValue().equals(fieldValue)) {
                    String message = item.getFieldDisplayName() + "值[" + fieldValue + "]与购方配置固定值[" + defaultValueLimit.getDefaultValue() + "]不一致";
                    log.info("根据模板进行默认值NoChange校验--校验未通过！ validateResult:{}", message);
                    throw new ValidateException(message);
                }
            }
            // 固定枚举值获取
            FixedEnumValueLimit fixedEnumValueLimit = FieldLimitUtils.getFieldLimitByJsonObjects(item.getFieldLimits(), FieldLimitEnum.FIXED_ENUM_VALUE);
            if (null != fixedEnumValueLimit && CollectionUtils.isEmpty(fixedEnumValueLimit.getEnumValues()) && null != fieldValue) {
                log.info("#####根据模板进行固定值NoChange校验-fixedEnumValueLimit-开始校验 modelField：{}", JSON.toJSONString(item));
                List<FixedEnumValueLimit.FixedEnumValue> enumValues = fixedEnumValueLimit.getEnumValues();
                if (!CollectionUtils.isEmpty(enumValues)) {
                    // 配置固定枚举值列表
                    List<String> itemEnumValues = enumValues.stream().filter(v -> FixedEnumValueEnum.Y.code().equals(v.getFix())).map(FixedEnumValueLimit.FixedEnumValue::getCode).collect(Collectors.toList());
                    // 当前字段枚举值列表
                    List<String> fieldValues = new ArrayList<>();
                    if (fieldValue.startsWith("[") && fieldValue.endsWith("]")) {
                        fieldValues = JSON.parseArray(fieldValue, String.class);
                    } else {
                        fieldValues.add(fieldValue);
                    }
                    if (!itemEnumValues.containsAll(fieldValues) || !fieldValues.containsAll(itemEnumValues)) {
                        String message = item.getFieldDisplayName() + "值" + JSON.toJSONString(fieldValues) + "与购方配置固定值" + JSON.toJSONString(itemEnumValues) + "不一致";
                        log.warn("根据模板进行固定枚举值NoChange校验--校验未通过！ validateResult:{}", message);
                        throw new ValidateException(message);
                    }
                }
            }


        }
    }

}
