package com.xforceplus.ant.coop.rule.center.client.data.cc.tool;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.xforceplus.ant.coop.rule.center.client.api.cc.BizConfigOpenApi;
import com.xforceplus.ant.coop.rule.center.client.data.cc.bizconfig.BizConfig;
import com.xforceplus.ant.coop.rule.center.client.data.cc.bizconfig.BizMainConfig;
import com.xforceplus.ant.coop.rule.center.client.data.cc.bizconfig.enums.BizConfigTypeEnum;
import com.xforceplus.ant.coop.rule.center.client.data.cc.enums.CommonFlagEnum;
import com.xforceplus.ant.coop.rule.center.client.data.cc.enums.ConfigTypeEnum;
import com.xforceplus.ant.coop.rule.center.client.data.cc.enums.bizcfg.IBasicConfigEnum;
import com.xforceplus.ant.coop.rule.center.client.data.cc.request.GetBizConfigDetail;
import com.xforceplus.ant.coop.rule.center.client.data.cc.response.BizConfigDetail;
import com.xforceplus.ant.coop.rule.center.client.data.utils.BaseResult;
import com.xforceplus.xplatframework.exception.ResultCode;
import com.xforceplus.xplatframework.utils.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.Map;

/**
 * Describe: 业务配置处理工具类
 *
 * @Author xiezhongyong
 * @Date 2022/4/2
 */
@Slf4j
public class BizConfigTool {

    /**
     * 读取业务配置
     *
     * @param api             api client
     * @param configType      配置类型
     * @param basicConfigEnum 配置项
     * @param configObjId     购方配置ID(租户ID)
     * @return BizConfig 返回对应的配置信息 返回类型参考 ${@link com.xforceplus.ant.coop.rule.center.client.data.cc.bizconfig.enums.BizConfigTypeEnum}
     */
    public static <T> T getBizConfigDetail(BizConfigOpenApi api, ConfigTypeEnum configType, IBasicConfigEnum basicConfigEnum, Long configObjId) {
        return getBizConfigDetail(api, configType, basicConfigEnum, configObjId, null, null);
    }
    /**
     * 读取业务配置
     *
     * @param api             api client
     * @param configType      配置类型
     * @param basicConfigEnum 配置项
     * @param configObjId     购方配置ID(租户ID)
     * @param sellerTaxNo     销方税号
     * @param purchaserTaxNo  购方税号
     * @return BizConfig 返回对应的配置信息 返回类型参考 ${@link com.xforceplus.ant.coop.rule.center.client.data.cc.bizconfig.enums.BizConfigTypeEnum}
     */
    public static <T> T getBizConfigDetail(BizConfigOpenApi api, ConfigTypeEnum configType, IBasicConfigEnum basicConfigEnum, Long configObjId,
                                           String sellerTaxNo, String purchaserTaxNo) {
        return getBizConfigDetail(api, configType, basicConfigEnum, configObjId, null, null, null, CommonFlagEnum.ACCEPT, sellerTaxNo, purchaserTaxNo);
    }

    /**
     * 读取业务配置
     *
     * @param api             api client
     * @param configType      配置类型
     * @param basicConfigEnum 配置项
     * @param configObjId     购方配置ID(租户ID)
     * @param bindObjId       绑定关联ID(购销对关系)
     * @param billTypeName    单据类型中文名称
     * @param limiters        业务配置头部限定条件
     * @param commonFlag      是否接受通用模板
     * @return BizConfig 返回对应的配置信息 返回类型参考 ${@link com.xforceplus.ant.coop.rule.center.client.data.cc.bizconfig.enums.BizConfigTypeEnum}
     */
    public static <T> T getBizConfigDetail(BizConfigOpenApi api, ConfigTypeEnum configType, IBasicConfigEnum basicConfigEnum, Long configObjId, Long bindObjId, String billTypeName, Map<String, Object> limiters, CommonFlagEnum commonFlag) {
        return getBizConfigDetail(api, configType, basicConfigEnum, configObjId, bindObjId, billTypeName, limiters, commonFlag, null, null);
    }

    /**
     * 读取业务配置
     *
     * @param api             api client
     * @param configType      配置类型
     * @param basicConfigEnum 配置项
     * @param configObjId     购方配置ID(租户ID)
     * @param bindObjId       绑定关联ID(购销对关系)
     * @param billTypeName    单据类型中文名称
     * @param limiters        业务配置头部限定条件
     * @param commonFlag      是否接受通用模板
     * @param sellerTaxNo     销方税号
     * @param purchaserTaxNo  购方税号
     * @return BizConfig 返回对应的配置信息 返回类型参考 ${@link com.xforceplus.ant.coop.rule.center.client.data.cc.bizconfig.enums.BizConfigTypeEnum}
     */
    public static <T> T getBizConfigDetail(BizConfigOpenApi api, ConfigTypeEnum configType, IBasicConfigEnum basicConfigEnum,
                                           Long configObjId, Long bindObjId, String billTypeName,
                                           Map<String, Object> limiters, CommonFlagEnum commonFlag,
                                           String sellerTaxNo, String purchaserTaxNo) {
        try {
            // 字段过滤
            BaseResult<BizConfigDetail> response = getBizConfigResponse(api, configType, basicConfigEnum.modelType(), basicConfigEnum.code(), configObjId, bindObjId, billTypeName, limiters, commonFlag, sellerTaxNo, purchaserTaxNo);
            BizConfigDetail result = response.getResult();
            if (ResultCode.SUCCESS.code().equals(response.getCode()) && null != result) {
                return getConfig(result);
            }
            log.warn("##### 读取配置信息错误或者未找到配置,响应内容: {}", response);
            return null;
        } catch (Exception e) {
            log.warn("##### 租户ID:{}；模板modelType:{}读取配置异常：{}", configObjId, basicConfigEnum.modelType(), e);
        }
        return null;
    }


    /**
     * 读取业务配置
     *
     * @param api          api client
     * @param configType   配置类型
     * @param modelType    model类型
     * @param configCode   配置代码(字段)
     * @param configObjId  购方配置ID(租户ID)
     * @param bindObjId    绑定关联ID(购销对关系)
     * @param billTypeName 单据类型中文名称
     * @param limiters     业务配置头部限定条件
     * @param commonFlag   是否接受通用模板
     * @return BaseResult 响应对象
     */
    public static BaseResult<BizConfigDetail> getBizConfigResponse(BizConfigOpenApi api, ConfigTypeEnum configType, Integer modelType,
                                                                   String configCode, Long configObjId, Long bindObjId, String billTypeName,
                                                                   Map<String, Object> limiters, CommonFlagEnum commonFlag,
                                                                   String sellerTaxNo, String purchaserTaxNo) {
        GetBizConfigDetail request = new GetBizConfigDetail();
        request.setConfigType(configType.code());
        request.setBindObjId(bindObjId);
        request.setConfigObjId(configObjId);
        if (null != commonFlag) {
            request.setCommonFlag(commonFlag.code().toString());
        }
        request.setModelType(modelType);
        request.setBillTypeName(billTypeName);
        request.setConfigCode(configCode);
        request.setSellerTaxNo(sellerTaxNo);
        request.setPurchaserTaxNo(purchaserTaxNo);
        try {
            // 复杂条件
            GetBizConfigDetail.BizConfigComplexParam complexParam = new GetBizConfigDetail.BizConfigComplexParam();
            if (!CollectionUtils.isEmpty(limiters)) {
                complexParam.setLimiters(limiters);
            }
            request.setComplexParam(complexParam);
            log.info("##### 读取业务配置bizconfig,接口请求参数：{}", JsonUtils.writeObjectToFastJson(request));
            BaseResult<BizConfigDetail> response = getBizConfigResponse(api, request);
            log.info("##### 读取业务配置bizconfig,接口响应数据：{}", JsonUtils.writeObjectToFastJson(response));
            return response;
        } catch (Throwable e) {
            log.error("##### 租户ID:{}；模板modelType:{}, configCode: {}, 读取业务配置bizconfig异常：{}", configObjId, modelType, configCode, e);
        }
        // 不抛异常，防止中断上层业务
        return null;
    }

    /**
     * 读取业务配置
     *
     * @param api
     * @param request
     * @return
     */
    public static BaseResult<BizConfigDetail> getBizConfigResponse(BizConfigOpenApi api, GetBizConfigDetail request) {
        return api.getBizConfigDetailPost(request);
    }

    public static <T> T getConfig(BizConfigDetail bizConfigDetail) {
        BizConfigTypeEnum configTypeEnum = BizConfigTypeEnum.fromValue(bizConfigDetail.getConfigType());
        if (StringUtils.isEmpty(bizConfigDetail.getConfigResult())) {
            return null;
        }
        JSONObject configResult = bizConfigDetail.getConfigResult();
        BizConfig bizConfig = (BizConfig) JSON.parseObject(configResult.toJSONString(), configTypeEnum.configClass());
        // 主信息设置
        bizConfig.setMainConfig(JSON.parseObject(JSON.toJSONString(bizConfigDetail), BizMainConfig.class));
        log.info("##### 业务配置响应 转换为BizConfig对象后内容: {}", bizConfig);
        return (T) bizConfig;
    }

}