package com.xforceplus.ant.coop.rule.center.client.utils;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.xforceplus.ant.coop.rule.center.client.data.cc.bean.fieldlimit.FieldLimit;
import com.xforceplus.ant.coop.rule.center.client.data.cc.enums.FieldLimitEnum;
import com.xforceplus.xplatframework.utils.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * Describe: 字段限制工具类
 *
 * @Author xiezhongyong
 * @Date 2020-05-11
 */
@Slf4j
public class FieldLimitUtils {

    /**
     * String 转 FieldLimit （db 输出）
     *
     * @param fieldLimitStr
     * @return
     */
    public static List<FieldLimit> string2FieldLimit(String fieldLimitStr) {
        try {
            List<FieldLimit> ruleFieldLimits = new ArrayList<>();
            if (StringUtils.isEmpty(fieldLimitStr)) {
                return ruleFieldLimits;
            }

            List<JSONObject> jsonObjects = JSONArray.parseArray(fieldLimitStr, JSONObject.class);
            for (JSONObject fieldLimit : jsonObjects) {
                String limitType = fieldLimit.getString("limitType");
                if (StringUtils.isEmpty(limitType)) {
                    throw new RuntimeException("fieldLimit 格式错误，无法加载");
                }
                FieldLimitEnum ruleFieldLimitEnum = FieldLimitEnum.fromValue(limitType);
                if (null == ruleFieldLimitEnum) {
                    throw new RuntimeException(String.format("fieldLimit ->limitType = %s 不合法", limitType));
                }

                Object fieldLimitObj = fieldLimit.toJavaObject(ruleFieldLimitEnum.limitClass());
                ruleFieldLimits.add((FieldLimit) fieldLimitObj);
            }
            return ruleFieldLimits;
        } catch (RuntimeException e) {
            throw e;
        } catch (Exception e) {
            log.error("##### string2FieldLimit 异常: {}", e);
            throw new RuntimeException("##### string2FieldLimit 异常" + e);
        }
    }

    /**
     * jsonObjects 转 FieldLimit
     *
     * @param jsonObjects
     * @return
     */
    public static List<FieldLimit> jsonObjects2FieldLimit(List<JSONObject> jsonObjects) {
        try {
            return string2FieldLimit(jsonObjects.toString());
        } catch (RuntimeException e) {
            throw e;
        } catch (Exception e) {
            log.error("##### jsonObjects2FieldLimit 异常: {}", e);
            throw new RuntimeException("##### jsonObjects2FieldLimit 异常" + e);
        }
    }

    /**
     * 解析 fieldLimit 后获取 对应的字段限制对象
     *
     * @param fieldLimits
     * @param limitEnum
     * @return
     */
    public static <T> T getFieldLimit(List<FieldLimit> fieldLimits, FieldLimitEnum limitEnum) {

        for (FieldLimit fieldLimit : fieldLimits) {
            if (fieldLimit.getLimitType().equals(limitEnum.code())) {
                return (T) fieldLimit;
            }
        }
        return null;
    }

    /**
     * 解析 fieldLimit 后获取 对应的字段限制对象
     *
     * @param fieldLimits
     * @param limitEnum
     * @return
     */
    public static <T> T getFieldLimitByJsonObjects(List<JSONObject> fieldLimits, FieldLimitEnum limitEnum) {
        return getFieldLimit(JsonUtils.writeObjectToFastJson(fieldLimits), limitEnum);
    }

    /**
     * 解析 fieldLimit 后获取 对应的字段限制对象
     *
     * @param fieldLimit
     * @param limitEnum
     * @return
     */
    public static <T> T getFieldLimit(String fieldLimit, FieldLimitEnum limitEnum) {
        try {
            if (StringUtils.isEmpty(fieldLimit)) {
                return null;
            }
            List<FieldLimit> itemFieldLimits = string2FieldLimit(fieldLimit);

            Map<String, FieldLimit> itemFieldLimitMap = itemFieldLimits.stream().collect(
                    Collectors.toMap(FieldLimit::getLimitType, Function.identity(), (v1, v2) -> v2));

            FieldLimit itemFieldLimit = itemFieldLimitMap.get(limitEnum.code());

            return (T) itemFieldLimit;
        } catch (Exception e) {
            log.error("##### getFieldLimit 异常：{}", e.getMessage());
        }
        return null;
    }

}
