package com.xforceplus.seller.config.client.parse.impl;

import com.google.common.collect.Lists;
import com.xforceplus.phoenix.file.utils.CommonTools;
import com.xforceplus.seller.config.client.constant.PriceMethodTypeEnum;
import com.xforceplus.seller.config.client.model.MsConfigItemBean;
import com.xforceplus.seller.config.client.parse.ConfigParser;
import com.xforceplus.seller.config.client.parse.bean.InvoiceSplitrRuleDTO;
//import com.xforceplus.seller.invoice.client.model.PriceMethod;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.util.List;
import java.util.regex.Pattern;

/**
 * @Author zzw
 * @Description 拆票规则解析器
 * @Date
 * @return
 **/
@Component
public class InvoiceSplitParser implements ConfigParser {
    private static final Logger logger = LoggerFactory.getLogger(InvoiceSplitParser.class);

    @Override
    public InvoiceSplitrRuleDTO parse(List<MsConfigItemBean> cfgConfigItemEntities) {
        InvoiceSplitrRuleDTO invoiceSplitrRuleDTO = new InvoiceSplitrRuleDTO();
        for (MsConfigItemBean tmp : cfgConfigItemEntities) {
            invoiceSplitrRuleDTO.setRuldId(tmp.getConfigId());
            String fieldName = tmp.getConfigItemName();
            List<String>  itemValueList = tmp.getConfigItemValue();

            //当配置项值 设置了错误的格式，取出的集合是空则给一个默认值
            if (CollectionUtils.isEmpty(itemValueList)) {
                itemValueList = Lists.newArrayList("");
            }
            String itemValue = itemValueList.get(0);
            if ("invoiceLimit".equals(fieldName)) {
                if (CommonTools.isEmpty(itemValue)){
                    throw  new RuntimeException("开票限额必填项");
                }else if (new BigDecimal(itemValue).compareTo(BigDecimal.ZERO)<=0){
                    throw  new RuntimeException("开票限额必须大于0");
                }
                invoiceSplitrRuleDTO.setInvoiceLimit(new BigDecimal(itemValue));
            }
            else if("priceMethod".equals(fieldName)) {
                if (!NumberUtils.isNumber(itemValue)) {
                    itemValue = "0";
                    logger.info("价格方式：0-不含税 1-含税 当前传入值不在范围内.使用默认价格方式：0-不含税");
                }
                PriceMethodTypeEnum priceMethodTypeEnum = PriceMethodTypeEnum.fromValue(Integer.parseInt(itemValue));
//                int priceMethod = priceMethodTypeEnum == null ? 0 : priceMethodTypeEnum.getValue();
//                PriceMethod priceMethodEnum = PriceMethod.value(priceMethod);
                invoiceSplitrRuleDTO.setPriceMethod(priceMethodTypeEnum);
            }
            else if ("splitFiledList".equals(fieldName)) {
                invoiceSplitrRuleDTO.setSplitFiledList(itemValueList);
            }
            else if ("discountRateLimit".equals(fieldName)) {
                if (!NumberUtils.isNumber(itemValue) || StringUtils.isEmpty(itemValue)) {
                    itemValue = "100";
                }
                BigDecimal discountRateLimit = new BigDecimal(itemValue);
                if (BigDecimal.ZERO.compareTo(discountRateLimit) > 0 || new BigDecimal("100").compareTo(discountRateLimit) < 0) {
                    discountRateLimit = new BigDecimal("100");
                }
                invoiceSplitrRuleDTO.setDiscountRateLimit(discountRateLimit);
            }
            else if ("mergeBySplitField".equals(fieldName)) {
                if (!NumberUtils.isNumber(itemValue) || StringUtils.isEmpty(itemValue)) {
                    itemValue = "0";
                }
                int mergeBySplitField = Integer.parseInt(itemValue);
                if (mergeBySplitField< 0 || mergeBySplitField > 1) {
                    mergeBySplitField = 0;
                }
                invoiceSplitrRuleDTO.setMergeBySplitField(mergeBySplitField != 0);
            }
            else if ("saleListOption".equals(fieldName)) {
                invoiceSplitrRuleDTO.setSaleListOption(itemValue);
            }
            else if ("discountMode".equals(fieldName)) {
                invoiceSplitrRuleDTO.setDiscountMode(itemValue);
            }
            else if ("taxDeviceType".equals(fieldName)) {
                 invoiceSplitrRuleDTO.setTaxDeviceType(itemValue);
            }
            else if ("itemSort".equals(fieldName)) {
                 invoiceSplitrRuleDTO.setItemSort(itemValue);
            }
            else if ("salesListMaxRow".equals(fieldName)) {
                //判断是否为整数
                if(null == itemValue || !Pattern.compile("^[\\d]*$").matcher(itemValue).matches()){
                    logger.info("销货清单最大行数只能为整数，传入值：{}",itemValue);
                    throw  new RuntimeException("销货清单最大行数只能为整数");
                }
                invoiceSplitrRuleDTO.setSalesListMaxRow(itemValue.equals("") ? 0: Integer.parseInt(itemValue));
            }
            else if ("unitPriceAmountOps".equals(fieldName)) {
                invoiceSplitrRuleDTO.setUnitPriceAmountOps(itemValue);
            }
            else if ("amountSplitRule".equals(fieldName)) {
                invoiceSplitrRuleDTO.setAmountSplitRule(itemValue);
            }
            else if ("integerOps".equals(fieldName)) {
                 invoiceSplitrRuleDTO.setIntegerOps(Boolean.parseBoolean(itemValue.equals("") ? "false" : itemValue));
            }
            else if ("invoiceItemMaxRow".equals(fieldName)) {
                if (CommonTools.isEmpty(itemValue)){
                    throw  new RuntimeException("票面行数必填项");
                }else if (new BigDecimal(itemValue).compareTo(BigDecimal.ZERO)<0){
                    throw  new RuntimeException("票面行数不能小于0");
                }
                invoiceSplitrRuleDTO.setInvoiceItemMaxRow(CommonTools.isEmpty(itemValue) ? 0: Integer.parseInt(itemValue));
            }
            else if ("invoiceType".equals(fieldName)) {
                invoiceSplitrRuleDTO.setInvoiceType(itemValue);
            }
            else if ("invoiceRemarkFiledList".equals(fieldName)) {
                invoiceSplitrRuleDTO.setInvoiceRemarkFiledList(itemValueList);
            }else if("invoiceItemRemarkFiledList".equals(fieldName)){
                invoiceSplitrRuleDTO.setInvoiceItemRemarkFiledList(itemValueList);
            }else if("goodsServicesNameFieldsList".equals(fieldName)){
                invoiceSplitrRuleDTO.setGoodsServicesNameFieldsList(itemValueList);
            }else if("itemSpecFieldsList".equals(fieldName)){
                invoiceSplitrRuleDTO.setItemSpecFieldsList(itemValueList);
            }else if("goodsServicesNameMainFieldsList".equals(fieldName)){
                invoiceSplitrRuleDTO.setGoodsServicesNameMainFieldsList(itemValueList);
            }else if("itemSpecMainFieldsList".equals(fieldName)){
                invoiceSplitrRuleDTO.setItemSpecMainFieldsList(itemValueList);
            } else if ("customRemarkSize".equals(fieldName)) {
                if(!CommonTools.isEmpty(itemValue)){
                    if(itemValue.matches("^-?\\d+$")){
                        invoiceSplitrRuleDTO.setCustomRemarkSize(Integer.valueOf(itemValue));
                    }else{
                        throw  new RuntimeException("自定义备注长度必须为整数");
                    }
                }
            }else if("bizBillDiscountRate".equals(fieldName)){
                // 业务单折扣率, 页面传值会是携带 % 的格式
                if(!CommonTools.isEmpty(itemValue) ){
                    // 存 #0.00 格式或者 %格式
                    if(itemValue.contains("%")){
                        // 洗掉%
                        itemValue = StringUtils.replace(itemValue,"%","");
                        // 转 #0.00
                        invoiceSplitrRuleDTO.setBizBillDiscountRate(new BigDecimal(itemValue).divide(new BigDecimal("100")));
                    }
                } else{
                    //#0.00
                    invoiceSplitrRuleDTO.setBizBillDiscountRate(new BigDecimal(itemValue));
                }
            }else if ("ruleStrategy".equals(fieldName)) {
                invoiceSplitrRuleDTO.setRuleStrategy(itemValue);
            }else if ("priceScale".equals(fieldName)) {
                try {
                    invoiceSplitrRuleDTO.setPriceScale(Integer.parseInt(itemValue));
                }catch (Exception e){
                    logger.error("解析拆票规则单价位数出现异常",e);
                    invoiceSplitrRuleDTO.setPriceScale(15);
                }

            } else if ("invoiceRemarkDistinctFlag".equals(fieldName)) {
                //跟前端约定的字段与中台拆票的字段有差异，但是项目已经封板了，前端也来不及再调整了，先这样吧
                //这个方法可以改为mastruct来转换,
                invoiceSplitrRuleDTO.setRemarkDuplicateFlag(Boolean.parseBoolean("".equals(itemValue) ? "true" : String.valueOf("1".equals(itemValue))));
            }else if("taxAmountAllowMax".equals(fieldName)){
                if(!CommonTools.isEmpty(itemValue) ) {
                    try {
                        invoiceSplitrRuleDTO.setTaxAmountAllowMax(new BigDecimal(itemValue));
                    }catch (Exception e){
                        logger.error("税额误差必须为数值且不大于1.27",e);
                    }
                }
            }
        }
        return invoiceSplitrRuleDTO;
    }
}
