package com.xforceplus.seller.config.client.parse.impl;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.xforceplus.phoenix.file.utils.CommonTools;
import com.xforceplus.seller.config.client.constant.PriceMethodTypeEnum;
import com.xforceplus.seller.config.client.model.MsConfigItemBean;
import com.xforceplus.seller.config.client.parse.ConfigParser;
import com.xforceplus.seller.config.client.parse.bean.InvoiceSplitrRuleDTO;
import io.vavr.Lazy;
import io.vavr.Tuple;
import io.vavr.Tuple2;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.regex.Pattern;

/**
 *
 * @Author zzw
 * @Description 拆票规则解析器
 * @Date
 * @return
 **/
@Component
public class InvoiceSplitParser implements ConfigParser {
    private static final Logger logger = LoggerFactory.getLogger(InvoiceSplitParser.class);

    @Override
    public InvoiceSplitrRuleDTO parse(List<MsConfigItemBean> cfgConfigItemEntities) {
        InvoiceSplitrRuleDTO invoiceSplitrRuleDTO = new InvoiceSplitrRuleDTO();
        for (MsConfigItemBean tmp : cfgConfigItemEntities) {
            invoiceSplitrRuleDTO.setRuldId(tmp.getConfigId());
            String fieldName = tmp.getConfigItemName();
            List<String>  itemValueList = tmp.getConfigItemValue();

            //当配置项值 设置了错误的格式，取出的集合是空则给一个默认值
            if (CollectionUtils.isEmpty(itemValueList)) {
                itemValueList = Lists.newArrayList("");
            }
            String itemValue = itemValueList.get(0);
            if ("invoiceLimit".equals(fieldName)) {
                if (CommonTools.isEmpty(itemValue)){
                    logger.info("开票限额为空。。。。。。");
                    invoiceSplitrRuleDTO.setInvoiceLimit(null);
                    continue;
                }else if (new BigDecimal(itemValue).compareTo(BigDecimal.ZERO)<=0){
                    throw  new RuntimeException("开票限额必须大于0");
                }
                invoiceSplitrRuleDTO.setInvoiceLimit(new BigDecimal(itemValue));
            }
            else if("priceMethod".equals(fieldName)) {
                if (!NumberUtils.isNumber(itemValue)) {
                    itemValue = "0";
                    logger.info("价格方式：0-不含税 1-含税 当前传入值不在范围内.使用默认价格方式：0-不含税");
                }
                PriceMethodTypeEnum priceMethodTypeEnum = PriceMethodTypeEnum.fromValue(Integer.parseInt(itemValue));
                invoiceSplitrRuleDTO.setPriceMethod(priceMethodTypeEnum);
            }
            else if ("mergeBySplitField".equals(fieldName)) {
                if (!NumberUtils.isNumber(itemValue) || StringUtils.isEmpty(itemValue)) {
                    itemValue = "0";
                }
                int mergeBySplitField = Integer.parseInt(itemValue);
                if (mergeBySplitField< 0 || mergeBySplitField > 1) {
                    mergeBySplitField = 0;
                }
                invoiceSplitrRuleDTO.setMergeBySplitField(mergeBySplitField != 0);
            }
            else if ("salesListMaxRow".equals(fieldName)) {
                //判断是否为整数
                if(null == itemValue || !Pattern.compile("^[\\d]*$").matcher(itemValue).matches()){
                    logger.info("销货清单最大行数只能为整数，传入值：{}",itemValue);
                    throw  new RuntimeException("销货清单最大行数只能为整数");
                }
                invoiceSplitrRuleDTO.setSalesListMaxRow(itemValue.equals("") ? 0: Integer.parseInt(itemValue));
            }
            else if ("integerOps".equals(fieldName)) {
                 invoiceSplitrRuleDTO.setIntegerOps(Boolean.parseBoolean(itemValue.equals("") ? "false" : itemValue));
            }
            else if ("invoiceItemMaxRow".equals(fieldName)) {
                if (CommonTools.isEmpty(itemValue)){
                    invoiceSplitrRuleDTO.setInvoiceItemMaxRow(null);
                    continue;
                }else if (new BigDecimal(itemValue).compareTo(BigDecimal.ZERO)<0){
                    throw  new RuntimeException("票面行数不能小于0");
                }
                invoiceSplitrRuleDTO.setInvoiceItemMaxRow(CommonTools.isEmpty(itemValue) ? 0: Integer.parseInt(itemValue));
            }
            else if ("customRemarkSize".equals(fieldName)) {
                if(!CommonTools.isEmpty(itemValue)){
                    if(itemValue.matches("^-?\\d+$")){
                        invoiceSplitrRuleDTO.setCustomRemarkSize(Integer.valueOf(itemValue));
                    }else{
                        throw  new RuntimeException("自定义备注长度必须为整数");
                    }
                }
            }
            else if ("priceScale".equals(fieldName)) {
                try {
                    invoiceSplitrRuleDTO.setPriceScale(Integer.parseInt(itemValue));
                }catch (Exception e){
                    logger.error("解析拆票规则单价位数出现异常",e);
                    invoiceSplitrRuleDTO.setPriceScale(15);
                }

            } else if ("invoiceRemarkDistinctFlag".equals(fieldName)) {
                //跟前端约定的字段与中台拆票的字段有差异，但是项目已经封板了，前端也来不及再调整了，先这样吧
                //这个方法可以改为mastruct来转换,
                invoiceSplitrRuleDTO.setRemarkDuplicateFlag(Boolean.parseBoolean("".equals(itemValue) ? "true" : String.valueOf("1".equals(itemValue))));
            }
            else if ("hideRemarkFieldName".equals(fieldName)) {
                if ("1".equals(itemValue)){
                    invoiceSplitrRuleDTO.setHideRemarkFieldName("true");
                }else {
                    invoiceSplitrRuleDTO.setHideRemarkFieldName("false");
                }
            } else if ("printItemSpecFlag".equals(fieldName)) {
                if ("1".equals(itemValue)){
                    invoiceSplitrRuleDTO.setPrintItemSpecFlag("false");
                }else {
                    invoiceSplitrRuleDTO.setPrintItemSpecFlag("true");
                }
            }else if ("outerToInnerDiscount".equals(fieldName)){
                if ("1".equals(itemValue)){
                    invoiceSplitrRuleDTO.setOuterToInnerDiscount("true");
                }else {
                    invoiceSplitrRuleDTO.setOuterToInnerDiscount("false");
                }
            }

            else if ("remarkKeyValueShowType".equals(fieldName)) {
                if(!CommonTools.isEmpty(itemValue) ){
                    invoiceSplitrRuleDTO.setRemarkKeyValueShowType(Integer.parseInt(itemValue));
                }
            } else if ("quantityLimitTuple".equals(fieldName)) {
                if (CollectionUtils.isNotEmpty(itemValueList)
                        && itemValueList.size() == 2
                        && itemValueList.stream().noneMatch(StringUtils::isBlank)) {
                    invoiceSplitrRuleDTO.setQuantityLimitUnit(itemValueList.get(0));
                    invoiceSplitrRuleDTO.setQuantityLimit(new BigDecimal(itemValueList.get(1)));
                }
            } else {

                if (this.stringValueConvert(invoiceSplitrRuleDTO, fieldName, itemValue)) {
                    continue;
                }

                if(this.stringValuesConvert(invoiceSplitrRuleDTO, fieldName, itemValueList)) {
                    continue;
                }

                if (this.bigDecimalValueConvert(invoiceSplitrRuleDTO, fieldName, itemValue)) {
                    continue;
                }

            }
        }

        //全电标识没有、默认税控
        if (StringUtils.isEmpty(invoiceSplitrRuleDTO.getTaxInvoiceSource())){
            invoiceSplitrRuleDTO.setTaxInvoiceSource("sk");
        }

        return invoiceSplitrRuleDTO;
    }


    /**
     * String值设置
     */
    private boolean stringValueConvert(InvoiceSplitrRuleDTO invoiceSplitrRuleDTO, String fieldName, String itemValue) {
        boolean  hitField = STRING_VALUE_CONVERT_MAP.containsKey(fieldName);
        if(hitField) {
            Optional.ofNullable(STRING_VALUE_CONVERT_MAP.get(fieldName)).ifPresent(consumer -> consumer.accept(invoiceSplitrRuleDTO, itemValue));
        }
        return hitField;
    }

    private static final Map<String, BiConsumer<InvoiceSplitrRuleDTO, String>> STRING_VALUE_CONVERT_MAP = Lazy.of(() -> {
        Map<String, BiConsumer<InvoiceSplitrRuleDTO, String>> map = Maps.newHashMap();

        map.put("saleListOption", InvoiceSplitrRuleDTO::setSaleListOption);
        map.put("discountMode", InvoiceSplitrRuleDTO::setDiscountMode);
        map.put("taxDeviceType", InvoiceSplitrRuleDTO::setTaxDeviceType);
        map.put("itemSort", InvoiceSplitrRuleDTO::setItemSort);
        map.put("unitPriceAmountOps", InvoiceSplitrRuleDTO::setUnitPriceAmountOps);
        map.put("amountSplitRule", InvoiceSplitrRuleDTO::setAmountSplitRule);
        map.put("invoiceType", InvoiceSplitrRuleDTO::setInvoiceType);
        map.put("ruleStrategy", InvoiceSplitrRuleDTO::setRuleStrategy);
        map.put("taxInvoiceSource", InvoiceSplitrRuleDTO::setTaxInvoiceSource);

        return map;
    }).get();


    /**
     * List<String>值设置
     */
    private boolean stringValuesConvert(InvoiceSplitrRuleDTO invoiceSplitrRuleDTO, String fieldName, List<String> itemValues) {
        boolean  hitField = STRING_VALUES_CONVERT_MAP.containsKey(fieldName);
        if(hitField) {
            if(!StringUtils.isEmpty(itemValues.get(0))) {
                Optional.ofNullable(STRING_VALUES_CONVERT_MAP.get(fieldName)).ifPresent(consumer -> consumer.accept(invoiceSplitrRuleDTO, itemValues));
            }
        }
        return hitField;
    }

    private static final Map<String, BiConsumer<InvoiceSplitrRuleDTO, List<String>>> STRING_VALUES_CONVERT_MAP = Lazy.of(() -> {
        Map<String, BiConsumer<InvoiceSplitrRuleDTO, List<String>>> map = Maps.newHashMap();
        map.put("splitFiledList", InvoiceSplitrRuleDTO::setSplitFiledList);
        map.put("invoiceRemarkFiledList", InvoiceSplitrRuleDTO::setInvoiceRemarkFiledList);
        map.put("invoiceItemRemarkFiledList", InvoiceSplitrRuleDTO::setInvoiceItemRemarkFiledList);
        map.put("goodsServicesNameFieldsList", InvoiceSplitrRuleDTO::setGoodsServicesNameFieldsList);
        map.put("itemSpecFieldsList", InvoiceSplitrRuleDTO::setItemSpecFieldsList);
        map.put("goodsServicesNameMainFieldsList", InvoiceSplitrRuleDTO::setGoodsServicesNameMainFieldsList);
        map.put("itemSpecMainFieldsList", InvoiceSplitrRuleDTO::setItemSpecMainFieldsList);

        return map;
    }).get();



    private boolean bigDecimalValueConvert(InvoiceSplitrRuleDTO invoiceSplitrRuleDTO, String fieldName, String itemValue) {
        boolean hitField = BIGDECIMAL_VALUE_CONVERT_DEFAULT_FALSE_MAP.containsKey(fieldName);
        if(hitField) {
            Optional.ofNullable(BIGDECIMAL_VALUE_CONVERT_DEFAULT_FALSE_MAP.get(fieldName)).ifPresent(consumer -> consumer._2.accept(invoiceSplitrRuleDTO, consumer._1.apply(itemValue)));
        }
        return hitField;
    }

    private static final Function<String, BigDecimal> DISCOUNT_RATE_LIMIT_TO_BIG_DECIMAL = itemValue -> {
        if (!NumberUtils.isNumber(itemValue) || StringUtils.isEmpty(itemValue)) {
            itemValue = "100";
        }
        BigDecimal discountRateLimit = new BigDecimal(itemValue);
        if (BigDecimal.ZERO.compareTo(discountRateLimit) > 0 || new BigDecimal("100").compareTo(discountRateLimit) < 0) {
            discountRateLimit = new BigDecimal("100");
        }
        return discountRateLimit;
    };

    private static final Function<String, BigDecimal> BIZ_BILL_DISCOUNT_RATE_TO_BIG_DECIMAL = itemValue -> {
        // 业务单折扣率, 页面传值会是携带 % 的格式
        if(!CommonTools.isEmpty(itemValue) ){
            // 存 #0.00 格式或者 %格式
            if(itemValue.contains("%")){
                // 洗掉%
                itemValue = StringUtils.replace(itemValue,"%","");
                // 转 #0.00
                return new BigDecimal(itemValue).divide(new BigDecimal("100"));
            }
        } else {
            //#0.00
            return new BigDecimal(itemValue);
        }
        return new BigDecimal("1");
    };

    private static final Function<String, BigDecimal> TAX_AMOUNT_ALLOW_MAX_TO_BIG_DECIMAL = itemValue -> {
        if(!CommonTools.isEmpty(itemValue) ) {
            try {
                return new BigDecimal(itemValue);
            }catch (Exception e){
                logger.error("税额误差必须为数值且不大于1.27",e);
            }
        }
        return null;
    };

    /**
     * BigDecimal 值设置
     */
    private static final Map<String,  Tuple2<Function<String, BigDecimal>,BiConsumer<InvoiceSplitrRuleDTO, BigDecimal>> > BIGDECIMAL_VALUE_CONVERT_DEFAULT_FALSE_MAP = Lazy.of(() -> {
        Map<String, Tuple2<Function<String, BigDecimal>,BiConsumer<InvoiceSplitrRuleDTO, BigDecimal>> > map = Maps.newHashMap();

        map.put("discountRateLimit", Tuple.of(DISCOUNT_RATE_LIMIT_TO_BIG_DECIMAL ,InvoiceSplitrRuleDTO::setDiscountRateLimit));
        map.put("bizBillDiscountRate", Tuple.of(BIZ_BILL_DISCOUNT_RATE_TO_BIG_DECIMAL ,InvoiceSplitrRuleDTO::setBizBillDiscountRate));
        map.put("taxAmountAllowMax", Tuple.of(TAX_AMOUNT_ALLOW_MAX_TO_BIG_DECIMAL ,InvoiceSplitrRuleDTO::setTaxAmountAllowMax));

        return map;
    }).get();

}
