package com.xforceplus.seller.config.client.parse.impl;

import com.google.common.collect.Lists;
import com.xforceplus.phoenix.file.utils.CommonTools;
import com.xforceplus.seller.config.client.constant.ValueEnum;
import com.xforceplus.seller.config.client.constant.smartmatch.SmartMatchAmountCheckRule;
import com.xforceplus.seller.config.client.constant.smartmatch.SmartMatchFieldUseKind;
import com.xforceplus.seller.config.client.constant.smartmatch.SmartMatchFindAlgorithm;
import com.xforceplus.seller.config.client.model.MsConfigItemBean;
import com.xforceplus.seller.config.client.parse.ConfigParser;
import com.xforceplus.seller.config.client.parse.bean.SmartMatchRuleDTO;
import com.xforceplus.xplatframework.utils.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 智能推荐规则解析
 */
@Component
@Slf4j
public class SmartMatchRuleParser implements ConfigParser {

    @Override
    public SmartMatchRuleDTO parse(List<MsConfigItemBean> cfgConfigItemEntities) {
        if (CollectionUtils.isEmpty(cfgConfigItemEntities)) {
            return null;
        }
        SmartMatchRuleDTO smartMatchRuleDTO = new SmartMatchRuleDTO();
        for (MsConfigItemBean msConfigItemBean : cfgConfigItemEntities) {
            smartMatchRuleDTO.setRuldId(msConfigItemBean.getConfigId());
            String fieldName = msConfigItemBean.getConfigItemName();
            if (StringUtils.isEmpty(fieldName)) {
                continue;
            }
            List<String> itemValueList = msConfigItemBean.getConfigItemValue();
            if (CollectionUtils.isEmpty(itemValueList)) {
                log.info("RedConfirmParser--itemValueList--parse--Merge配置项值设置错误:null");
                itemValueList = Lists.newArrayList("");
            }

            String value = itemValueList.get(0);
            if (CommonTools.isEmpty(value)) {
                continue;
            }
            switch (fieldName) {
                //matchCondition为0709迭代后的新属性，拆分为findRule和calScoreRule两个属性
                case "matchCondition": {
                    Map<SmartMatchFieldUseKind, List<SmartMatchRuleDTO.SmartMatchCondition>> smartMatchConditionMap;
                    try {
                        List<SmartMatchRuleDTO.SmartMatchCondition> smartMatchConditions = JsonUtils.writeJsonToListObject(value, SmartMatchRuleDTO.SmartMatchCondition.class);
                        smartMatchConditionMap = smartMatchConditions.stream().collect(Collectors.groupingBy(r -> r.getField().getUseKind()));
                    } catch (Exception e) {
                        log.error("解析匹配策略字段异常:{}", e.getMessage(), e);
                        throw new RuntimeException("解析打分规则字段异常");
                    }
                    smartMatchRuleDTO.setFindRule(smartMatchConditionMap.getOrDefault(SmartMatchFieldUseKind.FILTER, Lists.newArrayList()).stream().map(this::convertFindRuleItem).collect(Collectors.toList()));
                    smartMatchRuleDTO.setCalScoreRule(smartMatchConditionMap.getOrDefault(SmartMatchFieldUseKind.SCORE, Lists.newArrayList()).stream().map(this::convertCalScoreRuleItem).collect(Collectors.toList()));
                    break;
                }
                case "redReason": {
                    smartMatchRuleDTO.setRedReason(value);
                    break;
                }
                case "matchAlgorithm": {
                    smartMatchRuleDTO.setMatchAlgorithm(Integer.parseInt(value));
                    break;
                }
                case "amountCheckRule": {
                    smartMatchRuleDTO.setAmountCheckRule(
                            StringUtils.isBlank(value)
                                    ? SmartMatchAmountCheckRule.PRICING_AMOUNT_SAME.getValue()
                                    : ValueEnum.getEnumByValue(SmartMatchAmountCheckRule.class, Integer.parseInt(value)).orElse(SmartMatchAmountCheckRule.PRICING_AMOUNT_SAME).getValue()
                    );
                    break;
                }
                case "taxAmountMaxAdjust": {
                    smartMatchRuleDTO.setTaxAmountMaxAdjust(StringUtils.isBlank(value) ? BigDecimal.ZERO : new BigDecimal(value));
                    break;
                }
                case "findAlgorithm": {
                    smartMatchRuleDTO.setFindAlgorithm(
                            StringUtils.isBlank(value)
                                    ? SmartMatchFindAlgorithm.ITEM_NUM_LEAST.getValue()
                                    : ValueEnum.getEnumByValue(SmartMatchFindAlgorithm.class, Integer.parseInt(value)).orElse(SmartMatchFindAlgorithm.ITEM_NUM_LEAST).getValue()
                    );
                    break;
                }
                default:
                    break;
            }
        }

        return smartMatchRuleDTO;
    }

    /**
     * 匹配条件转换为过滤规则
     * @param smartMatchCondition 匹配条件
     * @return 过滤规则
     */
    private SmartMatchRuleDTO.FindRuleItem convertFindRuleItem(SmartMatchRuleDTO.SmartMatchCondition smartMatchCondition) {
        SmartMatchRuleDTO.FindRuleItem findRuleItem = new SmartMatchRuleDTO.FindRuleItem();
        findRuleItem.setField(smartMatchCondition.getField());
        findRuleItem.setOp(smartMatchCondition.getOp());
        findRuleItem.setType(smartMatchCondition.getType());
        findRuleItem.setValue(smartMatchCondition.getValue());
        findRuleItem.setValueList(smartMatchCondition.getValueList());
        findRuleItem.setTimeMode(smartMatchCondition.getTimeMode());
        findRuleItem.setFieldKind(smartMatchCondition.getField().getKind().getValue());
        return findRuleItem;
    }

    /**
     * 匹配条件转换为打分规则
     * @param smartMatchCondition 匹配条件
     * @return 打分规则
     */
    private SmartMatchRuleDTO.CalScoreRuleItem convertCalScoreRuleItem(SmartMatchRuleDTO.SmartMatchCondition smartMatchCondition) {
        SmartMatchRuleDTO.CalScoreRuleItem calScoreRuleItem = new SmartMatchRuleDTO.CalScoreRuleItem();
        calScoreRuleItem.setField(smartMatchCondition.getField());
        calScoreRuleItem.setType(smartMatchCondition.getType());
        calScoreRuleItem.setMatchType(smartMatchCondition.getOp().name());
        calScoreRuleItem.setTolerance(smartMatchCondition.getTolerance());
        calScoreRuleItem.setForce(smartMatchCondition.getForce());
        calScoreRuleItem.setValueMode(smartMatchCondition.getValueMode());
        calScoreRuleItem.setValue(smartMatchCondition.getValue());
        return calScoreRuleItem;
    }
}
