package com.xforceplus.seller.config.client.parse.impl;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.xforceplus.phoenix.file.utils.CommonTools;
import com.xforceplus.seller.config.client.model.MsConfigItemBean;
import com.xforceplus.seller.config.client.parse.ConfigParser;
import com.xforceplus.seller.config.client.parse.bean.SysInfoRuleDTO;
import io.vavr.Lazy;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * @Author zzw
 * @Description 拆票规则解析器
 * @Date
 * @return
 **/
@Component
public class SysInfoRuleParser implements ConfigParser {
    private static final Logger logger = LoggerFactory.getLogger(SysInfoRuleParser.class);

    @Override
    public SysInfoRuleDTO parse(List<MsConfigItemBean> cfgConfigItemEntities) {

        SysInfoRuleDTO sysInfoRuleDTO = new SysInfoRuleDTO();
        for (MsConfigItemBean msConfigItemBean : cfgConfigItemEntities) {
            sysInfoRuleDTO.setRuldId(msConfigItemBean.getConfigId());
            String fieldName = msConfigItemBean.getConfigItemName();
            List<String> itemValueList = msConfigItemBean.getConfigItemValue();
            if (CommonTools.isEmpty(fieldName)) {
                continue;
            }

            //当配置项值 设置了错误的格式，取出的集合是空则给一个默认值
            if (CollectionUtils.isEmpty(itemValueList)) {
                itemValueList = Lists.newArrayList("");
            }
            String itemValue = itemValueList.get(0);

            if ("uploadNagertiveMixBillFlag".equals(fieldName)) {
                //历史数据,可能没有这个配置项
                if (CommonTools.isEmpty(itemValue)) {
                    logger.info("没有找到系统配置项uploadNagertiveMixBillFlag");
                    sysInfoRuleDTO.setUploadNagertiveMixBillFlag(false);
                } else {
                    sysInfoRuleDTO.setUploadNagertiveMixBillFlag(Boolean.parseBoolean(itemValue));
                }
            } else if ("uploadBillAutoConfirm".equals(fieldName)) {
                //历史数据,可能没有这个配置项
                if (CommonTools.isEmpty(itemValue)) {
                    logger.info("uploadBillAutoConfirm");
                    sysInfoRuleDTO.setUploadBillAutoConfirm(true);
                } else {
                    sysInfoRuleDTO.setUploadBillAutoConfirm(Boolean.parseBoolean(itemValue));
                }
            } else if ("billTolerance".equals(fieldName)) {
                if (CommonTools.isEmpty(itemValue) || !Pattern.compile("^[0-9]\\d{0,13}(\\.\\d{0,4})?$").matcher(itemValue).matches()) {
                    logger.error("发票匹配业务单配置项值容差只能为2位小数金额数据，传入值：{}", itemValue);
                    itemValue = String.valueOf(0);
                }
                sysInfoRuleDTO.setBillTolerance(new BigDecimal(itemValue).setScale(2, 4));
            } else if ("bizBillDiscountRate".equals(fieldName)) {
                // 业务单折扣率
                if (!CommonTools.isEmpty(itemValue)) {
                    sysInfoRuleDTO.setBizBillDiscountRate(itemValue);
                } else {
                    sysInfoRuleDTO.setBizBillDiscountRate("1");
                }
            } else if ("invoiceMatchTolerance".equals(fieldName)) {
                if (CommonTools.isEmpty(itemValue) || !Pattern.compile("^[0-9]\\d{0,13}(\\.\\d{0,4})?$").matcher(itemValue).matches()) {
                    itemValue = String.valueOf(0);
                }
                sysInfoRuleDTO.setInvoiceMatchTolerance(new BigDecimal(itemValue).setScale(2, BigDecimal.ROUND_HALF_UP));
            } else if ("updateARBillAndCasm".equals(fieldName)) {
                if (StringUtils.isBlank(itemValue)) {
                    itemValue = "false";
                }
                sysInfoRuleDTO.setUpdateARBillAndCasm(itemValue);
            } else if ("updateARBillAndGoods".equals(fieldName)) {
                if (StringUtils.isBlank(itemValue)) {
                    itemValue = "false";
                }
                sysInfoRuleDTO.setUpdateARBillAndGoods(itemValue);
            } else if ("redNoCompareTolerance".equals(fieldName)) {
                if (CommonTools.isEmpty(itemValue) || !Pattern.compile("^[0-9]\\d{0,13}(\\.\\d{0,4})?$").matcher(itemValue).matches()) {
                    logger.error("红字信息比对容差只能为2位小数金额数据，传入值：{}", itemValue);
                    itemValue = String.valueOf(0);
                }
                sysInfoRuleDTO.setRedNoCompareTolerance(new BigDecimal(itemValue).setScale(2, 4));
            } else if ("billManualTag".equals(fieldName)) {
                sysInfoRuleDTO.setBillManualTag(NumberUtils.isCreatable(itemValue) ? Integer.parseInt(itemValue) : 2);
            } else if ("billManualTagRule".equals(fieldName)) {
                sysInfoRuleDTO.setBillManualTagRule(NumberUtils.isCreatable(itemValue) ? Integer.parseInt(itemValue) : 0);
            } else if (StringUtils.equals("recommendGoodsType", fieldName)) {
                sysInfoRuleDTO.setRecommendGoodsType(NumberUtils.isCreatable(itemValue) ? Integer.parseInt(itemValue) : 2);
            } else if ("modifyTaxRateFlag".equals(fieldName)) {
                sysInfoRuleDTO.setModifyTaxRateFlag(NumberUtils.isCreatable(itemValue) ? Integer.parseInt(itemValue) : 0);
            } else if (StringUtils.equals("qdSellerTitle", fieldName)) {
                sysInfoRuleDTO.setQdSellerTitle(NumberUtils.isCreatable(itemValue) ? Integer.parseInt(itemValue) : 2);
            } else if (StringUtils.equals("redBillAutoConfirmFlag", fieldName)) {
                sysInfoRuleDTO.setRedBillAutoConfirmFlag("true".equals(itemValue));
            } else if (StringUtils.equals("redBillMatchRedConfirmMatchConfig", fieldName)) {
                sysInfoRuleDTO.setRedBillMatchRedConfirmMatchConfig("true".equals(itemValue));
            } else if (StringUtils.equals("smallScaleTaxpayerChangeToCommonInvoice", fieldName)) {
                sysInfoRuleDTO.setSmallScaleTaxpayerChangeToCommonInvoice("true".equals(itemValue));
            } else if (StringUtils.equals("purchaserTitleMatchType", fieldName)) {
                sysInfoRuleDTO.setPurchaserTitleMatchType(NumberUtils.isCreatable(itemValue) ? Integer.parseInt(itemValue) : 0);
            } else {
                if (this.stringValueConvert(sysInfoRuleDTO, fieldName, itemValue)) {
                    continue;
                }
                if (this.stringValuesConvert(sysInfoRuleDTO, fieldName, itemValueList)) {
                    continue;
                }
                if (this.integerValueConvert(sysInfoRuleDTO, fieldName, itemValue)) {
                    continue;
                }
                if (this.integerValuesConvert(sysInfoRuleDTO, fieldName, itemValueList)) {
                    continue;
                }
                if (this.boolValueConvert(sysInfoRuleDTO, fieldName, itemValue)) {
                    continue;
                }
                if (this.boolValueConvertDefaultFalse(sysInfoRuleDTO, fieldName, itemValue)) {
                    continue;
                }
            }
        }
        return sysInfoRuleDTO;
    }

    private boolean stringValueConvert(SysInfoRuleDTO sysInfoRuleDTO, String fieldName, String itemValue) {
        boolean hitField = STRING_VALUE_CONVERT_MAP.containsKey(fieldName);
        if (hitField) {
            Optional.ofNullable(STRING_VALUE_CONVERT_MAP.get(fieldName)).ifPresent(consumer -> consumer.accept(sysInfoRuleDTO, itemValue));
        }
        return hitField;
    }

    /**
     * String值设置
     */
    private static final Map<String, BiConsumer<SysInfoRuleDTO, String>> STRING_VALUE_CONVERT_MAP = Lazy.of(() -> {
        Map<String, BiConsumer<SysInfoRuleDTO, String>> map = Maps.newHashMap();

        map.put("sendMailFlag", SysInfoRuleDTO::setSendMailFlag);
        map.put("taxInvoiceSource", SysInfoRuleDTO::setTaxInvoiceSource);

        map.put("redInvoiceReason", SysInfoRuleDTO::setRedInvoiceReason);
        map.put("customerNo", SysInfoRuleDTO::setCustomerNo);
        map.put("invoiceTitlePhoneRequired", SysInfoRuleDTO::setInvoiceTitlePhoneRequired);
        map.put("followWechat", SysInfoRuleDTO::setFollowWechat);
        map.put("orderSearchCondition", SysInfoRuleDTO::setOrderSearchCondition);
        map.put("orderSearchSource", SysInfoRuleDTO::setOrderSearchSource);
        map.put("invoiceTitleCustomFieldName", SysInfoRuleDTO::setInvoiceTitleCustomFieldName);
        map.put("invoiceTitleCustomFieldRequired", SysInfoRuleDTO::setInvoiceTitleCustomFieldRequired);
        map.put("canRepeatInvoice", SysInfoRuleDTO::setCanRepeatInvoice);
        map.put("autoMakeTerminal", SysInfoRuleDTO::setAutoMakeTerminal);
        map.put("invoiceFixedGoodsNo", SysInfoRuleDTO::setInvoiceFixedGoodsNo);
        map.put("goodsMatch", SysInfoRuleDTO::setGoodsMatch);
        map.put("bulkOrderQueryFlag", SysInfoRuleDTO::setBulkOrderQueryFlag);
        map.put("salesBillApplyInvoiceHint", SysInfoRuleDTO::setSalesBillApplyInvoiceHint);
        map.put("thirdPartyCode", SysInfoRuleDTO::setThirdPartyCode);


        map.put("validSameStore", SysInfoRuleDTO::setValidSameStore);
        map.put("taxInvoiceSourceForInterfaceRedFlush", SysInfoRuleDTO::setTaxInvoiceSourceForInterfaceRedFlush);

        return map;
    }).get();

    private boolean stringValuesConvert(SysInfoRuleDTO sysInfoRuleDTO, String fieldName, List<String> itemValues) {
        boolean hitField = STRING_VALUES_CONVERT_MAP.containsKey(fieldName);
        if (hitField) {
            if (!StringUtils.isEmpty(itemValues.get(0))) {
                Optional.ofNullable(STRING_VALUES_CONVERT_MAP.get(fieldName)).ifPresent(consumer -> consumer.accept(sysInfoRuleDTO, itemValues));
            }
        }
        return hitField;
    }

    /**
     * List<String>值设置
     */
    private static final Map<String, BiConsumer<SysInfoRuleDTO, List<String>>> STRING_VALUES_CONVERT_MAP = Lazy.of(() -> {
        Map<String, BiConsumer<SysInfoRuleDTO, List<String>>> map = Maps.newHashMap();
        map.put("uploadCheckByQd", SysInfoRuleDTO::setUploadCheckByQd);
        map.put("enableAdvancedValidation", SysInfoRuleDTO::setEnableAdvancedValidation);
        map.put("invoiceTitleApplyInvoiceType", SysInfoRuleDTO::setInvoiceTitleApplyInvoiceType);
        map.put("uploadBillAutoMake-electronic", SysInfoRuleDTO::setUploadBillAutoMakeElectronic);
        map.put("rejectApplyRepeatFlag", SysInfoRuleDTO::setRejectApplyRepeatFlag);
        map.put("sellerAndPurchaserControl", SysInfoRuleDTO::setSellerAndPurchaserControl);
        map.put("qdCashierAndInvoicerViewConfig", SysInfoRuleDTO::setQdCashierAndInvoicerViewConfig);
        return map;
    }).get();


    private boolean integerValueConvert(SysInfoRuleDTO sysInfoRuleDTO, String fieldName, String itemValue) {
        boolean hitField = INTEGER_VALUE_CONVERT_MAP.containsKey(fieldName);
        if (hitField) {
            if (!StringUtils.isEmpty(itemValue)) {
                Optional.ofNullable(INTEGER_VALUE_CONVERT_MAP.get(fieldName)).ifPresent(consumer -> consumer.accept(sysInfoRuleDTO, Integer.valueOf(itemValue)));
            }
        }
        return hitField;
    }

    /**
     * Integer值设置
     */
    private static final Map<String, BiConsumer<SysInfoRuleDTO, Integer>> INTEGER_VALUE_CONVERT_MAP = Lazy.of(() -> {
        Map<String, BiConsumer<SysInfoRuleDTO, Integer>> map = Maps.newHashMap();
        map.put("bill2PreviewInvoiceSkipFlag", SysInfoRuleDTO::setBill2PreviewInvoiceSkipFlag);
        map.put("orgTypeView", SysInfoRuleDTO::setOrgTypeView);
        return map;
    }).get();

    private boolean integerValuesConvert(SysInfoRuleDTO sysInfoRuleDTO, String fieldName, List<String> itemValues) {
        boolean hitField = INTEGER_VALUES_CONVERT_MAP.containsKey(fieldName);
        if (hitField) {
            if (!StringUtils.isEmpty(itemValues.get(0))) {
                Optional.ofNullable(INTEGER_VALUES_CONVERT_MAP.get(fieldName)).ifPresent(consumer -> consumer.accept(sysInfoRuleDTO, itemValues.stream().map(Integer::valueOf).collect(Collectors.toList())));
            }
        }
        return hitField;
    }

    /**
     * List<Integer>值设置
     */
    private static final Map<String, BiConsumer<SysInfoRuleDTO, List<Integer>>> INTEGER_VALUES_CONVERT_MAP = Lazy.of(() -> {
        Map<String, BiConsumer<SysInfoRuleDTO, List<Integer>>> map = Maps.newHashMap();
        map.put("elecInvoiceRedFlushReleaseFalg", SysInfoRuleDTO::setElecInvoiceRedFlushReleaseFalg);
        map.put("spePaperInoviceRedFlushReleaseFalg", SysInfoRuleDTO::setSpePaperInoviceRedFlushReleaseFalg);
        map.put("spePaperInoviceDrawftReleaseFlag", SysInfoRuleDTO::setSpePaperInoviceDrawftReleaseFlag);
        map.put("comPaperInoviceDrawftReleaseFlag", SysInfoRuleDTO::setComPaperInoviceDrawftReleaseFlag);
        map.put("comPaperInvoiceRedFlushReleaseFlag", SysInfoRuleDTO::setComPaperInvoiceRedFlushReleaseFlag);
        map.put("vehicleInvoiceRedFlushReleaseFlag", SysInfoRuleDTO::setVehicleInvoiceRedFlushReleaseFlag);
        map.put("speElecInvoiceRedFlushReleaseFlag", SysInfoRuleDTO::setSpeElecInvoiceRedFlushReleaseFlag);
        return map;
    }).get();

    /**
     * 布尔值boolean设置
     */
    private static final Map<String, BiConsumer<SysInfoRuleDTO, Boolean>> BOOL_VALUE_CONVERT_MAP = Lazy.of(() -> {
        Map<String, BiConsumer<SysInfoRuleDTO, Boolean>> map = Maps.newHashMap();
        map.put("checkRedBillFalg", SysInfoRuleDTO::setCheckRedBillFalg);
        map.put("checkRedBillOriginInvoiceFalg", SysInfoRuleDTO::setCheckRedBillOriginInvoiceFalg);
        map.put("checkARBillPurchaseInfoFalg", SysInfoRuleDTO::setCheckARBillPurchaseInfoFalg);
        map.put("checkAPBillSellerInfoFalg", SysInfoRuleDTO::setCheckAPBillSellerInfoFalg);
        map.put("autoUpdateTaxCodeFlag", SysInfoRuleDTO::setAutoUpdateTaxCodeFlag);
        map.put("comInvoiceNeedShortNameFlag", SysInfoRuleDTO::setComInvoiceNeedShortNameFlag);
        map.put("reffiendOilMixOpenFlag", SysInfoRuleDTO::setReffiendOilMixOpenFlag);
        map.put("checkTaxRateByConvertCodeFlag", SysInfoRuleDTO::setCheckTaxRateByConvertCodeFlag);
        map.put("prePayCardAutoOneGoodFlag", SysInfoRuleDTO::setPrePayCardAutoOneGoodFlag);
        map.put("comInvoiceAllowSameSellerOrPurcherFlg", SysInfoRuleDTO::setComInvoiceAllowSameSellerOrPurcherFlg);
        map.put("paperInvoiceMakeAutosendPdfFlag", SysInfoRuleDTO::setPaperInvoiceMakeAutosendPdfFlag);
        map.put("invoiceMakeRemindPrintBillBackGroundFlag", SysInfoRuleDTO::setInvoiceMakeRemindPrintBillBackGroundFlag);
        map.put("autoSplitPreCardFlag", SysInfoRuleDTO::setAutoSplitPreCardFlag);
        map.put("speInvoiceAllowSameSellerOrPurcharseFlag", SysInfoRuleDTO::setSpeInvoiceAllowSameSellerOrPurcharseFlag);
        map.put("invoiceLock", SysInfoRuleDTO::setInvoiceLock);
        map.put("elecInvoiceNeedShortNameFlag", SysInfoRuleDTO::setElecInvoiceNeedShortNameFlag);
        map.put("elecInvoiceMakeAutoSendEmailFlag", SysInfoRuleDTO::setElecInvoiceMakeAutoSendEmailFlag);
        map.put("speInvoiceNeedShortNameFlag", SysInfoRuleDTO::setSpeInvoiceNeedShortNameFlag);
        map.put("autoSplitDiscountFlag", SysInfoRuleDTO::setAutoSplitDiscountFlag);
        map.put("checkTerminalFailIssueRetry", SysInfoRuleDTO::setCheckTerminalFailIssueRetry);
        map.put("qdForceBackCalculateTaxAmount", SysInfoRuleDTO::setQdForceBackCalculateTaxAmount);
        map.put("mobileEmailFillingMethod", SysInfoRuleDTO::setMobileEmailFillingMethod);
        map.put("allowReissue", SysInfoRuleDTO::setAllowReissue);
        map.put("allowAddIssuer", SysInfoRuleDTO::setAllowAddIssuer);
        map.put("isAutoSelectIssuerFlag", SysInfoRuleDTO::setIsAutoSelectIssuerFlag);
        map.put("sendToAntFlag", SysInfoRuleDTO::setSendToAntFlag);
        map.put("reconFlag", SysInfoRuleDTO::setReconFlag);
        map.put("autoGenerateInvoiceGoods", SysInfoRuleDTO::setAutoGenerateInvoiceGoods);
        map.put("jdcSpecialTypeFlag", SysInfoRuleDTO::setJdcSpecialTypeFlag);
        return map;
    }).get();

    private boolean boolValueConvert(SysInfoRuleDTO sysInfoRuleDTO, String fieldName, String itemValue) {
        boolean hitField = BOOL_VALUE_CONVERT_MAP.containsKey(fieldName);
        if (hitField) {
            Optional.ofNullable(BOOL_VALUE_CONVERT_MAP.get(fieldName)).ifPresent(consumer -> consumer.accept(sysInfoRuleDTO, Boolean.parseBoolean(itemValue)));
        }
        return hitField;
    }

    /**
     * 布尔值设置，默认false
     */
    private static final Map<String, BiConsumer<SysInfoRuleDTO, Boolean>> BOOL_VALUE_CONVERT_DEFAULT_FALSE_MAP = Lazy.of(() -> {
        Map<String, BiConsumer<SysInfoRuleDTO, Boolean>> map = Maps.newHashMap();
        map.put("checkEIvoiceDisplayOthers", SysInfoRuleDTO::setCheckEIvoiceDisplayOthers);
        map.put("modelACombineSetZeroToNotIssue", SysInfoRuleDTO::setModelACombineSetZeroToNotIssue);
        return map;
    }).get();

    private boolean boolValueConvertDefaultFalse(SysInfoRuleDTO sysInfoRuleDTO, String fieldName, String itemValue) {
        boolean hitField = BOOL_VALUE_CONVERT_DEFAULT_FALSE_MAP.containsKey(fieldName);
        if (hitField) {
            Optional.ofNullable(BOOL_VALUE_CONVERT_DEFAULT_FALSE_MAP.get(fieldName)).ifPresent(consumer -> consumer.accept(sysInfoRuleDTO, "true".equals(itemValue)));
        }
        return hitField;
    }
}
