package com.xforceplus.taxware.architecture.g1.rocketmq.client;

import com.alibaba.fastjson.JSON;
import com.xforceplus.taxware.architecture.g1.core.mq.BaseMessageDTO;
import com.xforceplus.taxware.architecture.g1.core.utils.ExceptionUtil;
import com.xforceplus.taxware.architecture.g1.domain.log.LogContext;
import com.xforceplus.taxware.architecture.g1.domain.log.model.LogSender;
import com.xforceplus.taxware.architecture.g1.domain.log.model.impl.MqLogEvent;
import com.xforceplus.taxware.architecture.g1.rocketmq.client.exception.RmqConsumeException;
import com.xforceplus.taxware.architecture.g1.rocketmq.client.model.RmqMessageExt;
import com.xforceplus.taxware.architecture.g1.rocketmq.client.util.RmqUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.rocketmq.acl.common.AclClientRPCHook;
import org.apache.rocketmq.acl.common.SessionCredentials;
import org.apache.rocketmq.client.consumer.DefaultMQPushConsumer;
import org.apache.rocketmq.client.consumer.listener.ConsumeConcurrentlyContext;
import org.apache.rocketmq.client.consumer.listener.ConsumeConcurrentlyStatus;
import org.apache.rocketmq.client.consumer.listener.ConsumeReturnType;
import org.apache.rocketmq.client.consumer.listener.MessageListenerConcurrently;
import org.apache.rocketmq.client.consumer.rebalance.AllocateMessageQueueAveragely;
import org.apache.rocketmq.common.consumer.ConsumeFromWhere;
import org.apache.rocketmq.common.message.MessageExt;

import java.io.IOException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import org.apache.commons.io.IOUtils;

import static com.xforceplus.taxware.architecture.g1.rocketmq.client.Constants.VERSION_2;

@Slf4j
public class RmqConsumer {
    private static final int MAX_RECONSUME_TIMES = 4;
    private static final int CONSUME_THREAD_MIN = 4;
    private static final int CONSUME_THREAD_MAX = 8;

    private final DefaultMQPushConsumer defaultMQPushConsumer;

    private LogSender logSender;
    private String indexName;

    public RmqConsumer(final String consumerGroup, final String accessKey, final String secretKey) {
        final AclClientRPCHook aclClientRPCHook = new AclClientRPCHook(new SessionCredentials(accessKey, secretKey));
        this.defaultMQPushConsumer = new DefaultMQPushConsumer(consumerGroup, aclClientRPCHook, new AllocateMessageQueueAveragely());
        this.defaultMQPushConsumer.setMaxReconsumeTimes(MAX_RECONSUME_TIMES);
        this.defaultMQPushConsumer.setConsumeThreadMin(CONSUME_THREAD_MIN);
        this.defaultMQPushConsumer.setConsumeThreadMax(CONSUME_THREAD_MAX);
        this.defaultMQPushConsumer.setConsumeFromWhere(ConsumeFromWhere.CONSUME_FROM_TIMESTAMP);
    }

    public void start() {
        try {
            this.defaultMQPushConsumer.start();
        } catch (Exception e) {
            throw new RmqConsumeException(String.format("消费者启动异常，%s", Optional.ofNullable(e.getMessage()).orElse(e.getClass().getName())), e);
        }
    }

    public void setNameSrvAddr(final String nameSrvAddr) {
        this.defaultMQPushConsumer.setNamesrvAddr(nameSrvAddr);

    }

    public void subscribe(final String topic, final String subExpression) {
        try {
            this.defaultMQPushConsumer.subscribe(topic, subExpression);
        } catch (Exception e) {
            throw new RmqConsumeException(String.format("订阅异常，%s", Optional.ofNullable(e.getMessage()).orElse(e.getClass().getName())), e);
        }
    }

    public void registerMessageListener(final RmqMessageListenerConcurrently messageListener) {
        final String consumerGroup = this.defaultMQPushConsumer.getConsumerGroup();

        this.defaultMQPushConsumer.setPullInterval(messageListener.getPullInterval());

        this.defaultMQPushConsumer.registerMessageListener((MessageListenerConcurrently) (msgs, context) -> {
            LogContext.init();

            final MqLogEvent event = createEvent(msgs, context, consumerGroup);

            try {
                final List<RmqMessageExt> messageList = msgs.stream().map(msg -> {
                    final String bodyData = new String(msg.getBody(), StandardCharsets.UTF_8);
                    final BaseMessageDTO baseMessageDTO = JSON.parseObject(bodyData, BaseMessageDTO.class);
                    String body = baseMessageDTO.getBody();
                    final BaseMessageDTO.Meta meta = baseMessageDTO.getMeta();
                    if (StringUtils.equals(VERSION_2, meta.getVersion())) {
                        try {
                            body = IOUtils.toString(new URL(baseMessageDTO.getBody()), StandardCharsets.UTF_8);
                        } catch (IOException e) {
                            log.error("消费RMQ大报文异常", e);
                            throw new RmqConsumeException(String.format("消费异常，读取数据时，%s", Optional.ofNullable(e.getMessage()).orElse(e.getClass().getName())), e);
                        }
                    }
                    final RmqMessageExt result = new RmqMessageExt();
                    result.setKeys(msg.getKeys());
                    result.setTopic(msg.getTopic());
                    result.setTags(msg.getTags());
                    result.setDelayTimeLevel(msg.getDelayTimeLevel());
                    result.setBody(body);
                    result.setReconsumeTimes(msg.getReconsumeTimes());
                    result.putAllProperties(baseMessageDTO.getProperties());
                    result.setMsgId(msg.getMsgId());

                    return result;
                }).collect(Collectors.toList());

                RmqConsumer.this.defaultMQPushConsumer.setPullInterval(messageListener.getPullInterval());
                final ConsumeConcurrentlyStatus consumeStatus = messageListener.consumeMessage(messageList, context);

                event.setConsumeStatus(consumeStatus.name());

                return consumeStatus;
            } catch (Throwable e) {
                log.error("RmqConsumer接收消息时异常。{}", ExceptionUtil.toDesc(e));
                event.setThrowable(e);
                event.setConsumeStatus(ConsumeReturnType.EXCEPTION.name());
                throw e;
            } finally {
                event.getExt().putAll(LogContext.getAllPoint());

                if (logSender != null) {
                    logSender.send(event);
                }

                LogContext.clear();
            }
        });
    }

    public void setConsumeThreadMin(final int consumeThreadMin) {
        this.defaultMQPushConsumer.setConsumeThreadMin(consumeThreadMin);
    }

    public void setConsumeThreadMax(final int consumeThreadMax) {
        this.defaultMQPushConsumer.setConsumeThreadMax(consumeThreadMax);
    }

    public void setConsumeMessageBatchMaxSize(final int consumeMessageBatchMaxSize) {
        this.defaultMQPushConsumer.setConsumeMessageBatchMaxSize(consumeMessageBatchMaxSize);
    }

    public void setPullBatchSize(final int pullBatchSize) {
        this.defaultMQPushConsumer.setPullBatchSize(pullBatchSize);
    }

    public void setMaxReconsumeTimes(final int maxReconsumeTimes) {
        this.defaultMQPushConsumer.setMaxReconsumeTimes(maxReconsumeTimes);
    }

    public void setConsumeFromWhere(final ConsumeFromWhere consumeFromWhere) {
        this.defaultMQPushConsumer.setConsumeFromWhere(consumeFromWhere);
    }

    public void setLogSender(LogSender logSender) {
        this.logSender = logSender;
    }

    public void setIndexName(String indexName) {
        this.indexName = indexName;
    }

    public void shutdown() {
        this.defaultMQPushConsumer.shutdown();
    }

    private MqLogEvent createEvent(final List<MessageExt> messageList, final ConsumeConcurrentlyContext context, String consumerGroup) {
        final MqLogEvent event = new MqLogEvent();

        try {
            final MessageExt message = messageList.get(0);

            event.setMessageContent(new String(message.getBody(), StandardCharsets.UTF_8));
            event.setMessageContentSize(message.getBody().length);
            event.setMessageProperties(JSON.toJSONString(message.getProperties()));
            event.setReconsumeTimes(message.getReconsumeTimes());
            if (message.getDelayTimeLevel() > 0) {
                event.setDelayLevel(String.format("%s[%d]", RmqUtils.getDelayLevelName(message.getDelayTimeLevel()), message.getDelayTimeLevel()));
            }
            event.setTags(message.getTags());
            event.setKeys(message.getKeys());
            event.setService(this.indexName);
            event.setName(message.getTopic());
            event.setTraceId(LogContext.getTraceId());
            event.setType("RmqConsumer");
            event.setThreadName(Thread.currentThread().getName());
            event.setConsumerGroup(consumerGroup);

            return event;
        } catch (Exception e) {
            log.warn("RmqConsumer创建埋点事件对象异常", e);
            return event;
        }
    }
}
