package com.xforceplus.taxware.architecture.g1.rocketmq.client;

import com.alibaba.fastjson.JSON;
import com.xforceplus.taxware.architecture.g1.core.mq.BaseMessageDTO;
import com.xforceplus.taxware.architecture.g1.domain.log.LogContext;
import com.xforceplus.taxware.architecture.g1.domain.log.model.LogSender;
import com.xforceplus.taxware.architecture.g1.domain.log.model.impl.MqLogEvent;
import com.xforceplus.taxware.architecture.g1.rocketmq.client.exception.RmqProduceException;
import com.xforceplus.taxware.architecture.g1.rocketmq.client.model.RmqMessage;
import com.xforceplus.taxware.architecture.g1.rocketmq.client.util.RmqUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.rocketmq.acl.common.AclClientRPCHook;
import org.apache.rocketmq.acl.common.SessionCredentials;
import org.apache.rocketmq.client.exception.MQBrokerException;
import org.apache.rocketmq.client.producer.DefaultMQProducer;
import org.apache.rocketmq.client.producer.SendResult;
import org.apache.rocketmq.common.message.Message;
import org.apache.rocketmq.common.protocol.ResponseCode;

import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import static com.xforceplus.taxware.architecture.g1.rocketmq.client.Constants.VERSION_2;

@Slf4j
public class RmqProducer implements RmqProducerSender {

    private DefaultMQProducer defaultMQProducer;
    private LogSender logSender;
    private String indexName;
    private RmqSender rmqSender;

    public RmqProducer() {
    }

    public RmqProducer(final String producerGroup, final String accessKey, final String secretKey) {
        final AclClientRPCHook aclClientRPCHook = new AclClientRPCHook(new SessionCredentials(accessKey, secretKey));
        this.defaultMQProducer = new DefaultMQProducer(producerGroup, aclClientRPCHook);
        this.defaultMQProducer.setMaxMessageSize(24 * 1024 * 1024);
    }

    public void setDefaultMQProducer(final DefaultMQProducer defaultMQProducer) {
        this.defaultMQProducer = defaultMQProducer;
    }

    public void setNameSrvAddr(final String nameSrvAddr) {
        this.defaultMQProducer.setNamesrvAddr(nameSrvAddr);
    }

    @Override
    public SendResult send(final RmqMessage message) {
        MqLogEvent event = createEvent(message);

        try {

            final Message msg = translate(message);
            event = createEvent(msg);

            try {

                final SendResult sendResult = this.defaultMQProducer.send(msg);

                event.setSendStatus(sendResult.getSendStatus().name());

                return sendResult;

            } catch (Exception e) {

                if (e instanceof MQBrokerException) {

                    MQBrokerException mqBrokerException = (MQBrokerException) e;
                    // 出现broker busy或者system busy错误需要重试
                    if (ResponseCode.SYSTEM_BUSY == mqBrokerException.getResponseCode()) {
                        event.getExt().put("retrySend", "true");
                        // 等待2s后重试一次发送
                        Thread.sleep(2_000);

                        final SendResult sendResult = this.defaultMQProducer.send(msg);

                        event.setSendStatus(sendResult.getSendStatus().name());

                        return sendResult;
                    }
                }

                throw e;

            }

        } catch (Exception e) {
            log.error("RmqProducer发送消息时异常", e);
            event.setThrowable(e);
            throw new RmqProduceException(Optional.ofNullable(e.getMessage()).orElse(e.getClass().getSimpleName()), e);
        } finally {
            event.getExt().putAll(LogContext.getAllPoint());

            if (logSender != null) {
                logSender.send(event);
            }
        }
    }

    public SendResult send(final List<RmqMessage> messages) {
        final List<MqLogEvent> events = messages.stream().map(this::createEvent).collect(Collectors.toList());

        try {
            final List<Message> rmqMessages = messages.stream().map(this::translate).collect(Collectors.toList());
            final SendResult sendResult = this.defaultMQProducer.send(rmqMessages);

            for (MqLogEvent event : events) {
                event.setSendStatus(sendResult.getSendStatus().name());
            }

            return sendResult;
        } catch (Exception e) {
            log.error("RmqProducer发送消息时异常", e);
            for (MqLogEvent event : events) {
                event.setThrowable(e);
            }
            throw new RmqProduceException(Optional.ofNullable(e.getMessage()).orElse(e.getClass().getSimpleName()), e);
        } finally {
            for (MqLogEvent event : events) {
                event.getExt().putAll(LogContext.getAllPoint());

                if (logSender != null) {
                    logSender.send(event);
                }
            }
        }
    }

    public void start() {
        try {
            this.defaultMQProducer.start();
        } catch (Exception e) {
            throw new RmqProduceException(String.format("生产者启动异常，%s", Optional.ofNullable(e.getMessage()).orElse(e.getClass().getName())));
        }
    }

    public void setLogSender(final LogSender logSender) {
        this.logSender = logSender;
    }

    public void setIndexName(final String indexName) {
        this.indexName = indexName;
    }

    public void setRmqSender(RmqSender rmqSender) {
        this.rmqSender = rmqSender;
    }

    public void shutdown() {
        this.defaultMQProducer.shutdown();
    }

    private Message translate(final RmqMessage message) {
        final Message result = new Message();

        result.setTopic(message.getTopic());
        result.setTags(message.getTags());
        result.setDelayTimeLevel(message.getDelayTimeLevel());
        result.setKeys(message.getKeys());
        result.setBody(JSON.toJSONBytes(createBaseMessageDTO(message)));

        return result;
    }

    private BaseMessageDTO createBaseMessageDTO(final RmqMessage message) {

        // 报文大小超过1M转oss存储
        final int sizeLimit = 1024 * 1024;
        final int messageContentSize = message.getBody().getBytes(StandardCharsets.UTF_8).length;

        final BaseMessageDTO messageDTO = new BaseMessageDTO();
        final BaseMessageDTO.Meta meta = new BaseMessageDTO.Meta();
        if (messageContentSize > sizeLimit && rmqSender != null) {
            meta.setVersion(VERSION_2);
            final String bodyUrl = rmqSender.send(message.getBody());
            messageDTO.setBody(bodyUrl);
        } else {
            messageDTO.setBody(message.getBody());
        }

        messageDTO.setMeta(meta);
        messageDTO.setProperties(message.getProperties());

        return messageDTO;
    }

    private MqLogEvent createEvent(final RmqMessage message) {

        final MqLogEvent event = new MqLogEvent();

        try {

            final int messageContentSize = message.getBody().getBytes(StandardCharsets.UTF_8).length;
            event.setMessageContent(message.getBody());
            event.setMessageContentSize(messageContentSize);
            event.setMessageProperties(JSON.toJSONString(message.getProperties()));
            if (message.getDelayTimeLevel() > 0) {
                event.setDelayLevel(String.format("%s[%d]", RmqUtils.getDelayLevelName(message.getDelayTimeLevel()), message.getDelayTimeLevel()));
            }
            event.setTags(message.getTags());
            event.setKeys(message.getKeys());
            event.setService(this.indexName);
            event.setName(message.getTopic());
            event.setTraceId(LogContext.getTraceId());
            event.setType("RmqProducer");
            event.setThreadName(Thread.currentThread().getName());

            return event;
        } catch (Exception e) {
            log.warn("RmqProducer创建埋点事件对象异常", e);
            return event;
        }
    }

    private MqLogEvent createEvent(final Message message) {

        final MqLogEvent event = new MqLogEvent();

        try {

            event.setMessageContent(new String(message.getBody(), StandardCharsets.UTF_8));
            event.setMessageContentSize(message.getBody().length);
            event.setMessageProperties(JSON.toJSONString(message.getProperties()));
            if (message.getDelayTimeLevel() > 0) {
                event.setDelayLevel(String.format("%s[%d]", RmqUtils.getDelayLevelName(message.getDelayTimeLevel()), message.getDelayTimeLevel()));
            }
            event.setTags(message.getTags());
            event.setKeys(message.getKeys());
            event.setService(this.indexName);
            event.setName(message.getTopic());
            event.setTraceId(LogContext.getTraceId());
            event.setType("RmqProducer");
            event.setThreadName(Thread.currentThread().getName());

            return event;

        } catch (Exception e) {
            log.warn("RmqProducer创建埋点事件对象异常", e);
            return event;
        }
    }
}
