package com.xforceplus.taxware.architecture.g1.snowflake;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import redis.clients.jedis.DefaultJedisClientConfig;
import redis.clients.jedis.HostAndPort;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisClientConfig;
import redis.clients.jedis.params.SetParams;

import java.util.Optional;
import java.util.UUID;

public class SnowFlake {
    private static final Logger logger = LoggerFactory.getLogger(SnowFlake.class);

    private static final String KEY_PATTERN = "SnowFlake-%s-%d";
    private static final int MAX_SEQUENCE = 1024;
    private static final long TTL = 3_600 * 2;
    private static final long interval = 60_000 * 20;

    private final Option option;
    private final int workId;
    private final Generator generator;
    private final String cid = UUID.randomUUID().toString().toLowerCase();

    public SnowFlake(final Option option) {
        logger.info("SnowFlake开始启动, cid={}, option={}", cid, option.toString());

        this.option = option;

        this.workId = findWorkId();

        if (this.workId == 0) {
            final String errorMessage = String.format("SnowFlake的WorkId资源已耗尽! %s ", option.bizName);
            logger.error(errorMessage);
            throw new RuntimeException(errorMessage);
        }

        this.generator = new Generator(this.workId);

        logger.info("SnowFlake的WorkId保活心跳开启! {} {}", option.bizName, this.cid);

        new Thread(this::heartbeat).start();
    }

    /**
     * 获取下一个ID
     */
    public long nextId() {
        return this.generator.nextId();
    }

    /**
     * 心跳，保证WorkId的持续占有
     */
    private void heartbeat() {
        while (true) {
            try (final Jedis jedis = createJedis()) {
                final String key = String.format(KEY_PATTERN, option.bizName, this.workId);
                jedis.expire(key, TTL);

                logger.info("SnowFlake维持心跳[{}][{}]，下次心跳间隔[{}]", key, cid, interval);

                Thread.sleep(interval);
            } catch (Exception ex) {
                logger.error("SnowFlake心跳异常！" + Optional.ofNullable(ex.getMessage()).orElse(ex.getClass().getSimpleName()));
            }
        }
    }

    /**
     * 查找可用的WorkId
     *
     * @return work id (大于0是可用的ID，0为不可用ID)
     */
    private int findWorkId() {
        try (final Jedis jedis = createJedis()) {
            for (int i = 1; i <= MAX_SEQUENCE; i++) {
                final String key = String.format(KEY_PATTERN, option.bizName, i);

                final SetParams params = new SetParams();
                params.nx();
                params.ex(TTL);
                final String replyCode = jedis.set(key, this.cid, params);

                if ("ok".equalsIgnoreCase(replyCode)) {
                    return i;
                }
            }

            return 0;
        }
    }

    /**
     * 创建Jedis实例
     *
     * @return jedis
     */
    private Jedis createJedis() {

        // 构造Jedis的Config
        final DefaultJedisClientConfig.Builder jedisConfigBuilder = DefaultJedisClientConfig.builder();
        jedisConfigBuilder.database(option.database);
        if (option.user != null && option.user.trim().length() > 0) {
            jedisConfigBuilder.user(option.user);
        }
        if (option.password != null && option.password.trim().length() > 0) {
            jedisConfigBuilder.password(option.password);
        }
        final JedisClientConfig jedisClientConfig = jedisConfigBuilder.build();

        // 主机和端口
        final HostAndPort hp = new HostAndPort(option.host, option.port);

        // 返回jedis实例
        return new Jedis(hp, jedisClientConfig);
    }

    public static class Option {
        private String bizName = "Default";
        private String host;
        private int port;
        private int database = 1;
        private String user;
        private String password;

        public String getHost() {
            return host;
        }

        public void setHost(String host) {
            this.host = host;
        }

        public int getPort() {
            return port;
        }

        public void setPort(int port) {
            this.port = port;
        }

        public int getDatabase() {
            return database;
        }

        public void setDatabase(int database) {
            this.database = database;
        }

        public String getUser() {
            return user;
        }

        public void setUser(String user) {
            this.user = user;
        }

        public String getPassword() {
            return password;
        }

        public void setPassword(String password) {
            this.password = password;
        }

        public String getBizName() {
            return bizName;
        }

        public void setBizName(String bizName) {
            this.bizName = bizName;
        }

        @Override
        public String toString() {
            return "Option{"
                    + "bizName='" + bizName + '\''
                    + ", host='" + host + '\''
                    + ", port=" + port
                    + ", database=" + database
                    + ", user='" + user + '\''
                    + ", password='" + password + '\''
                    + '}';
        }
    }

    /**
     * ID 生产器
     */
    private static class Generator {

        /**
         * 起始的时间戳
         */
        private static final long START_TIMESTAMP = 1627747200000L;

        /**
         * 每一部分占用的位数
         */
        private static final long SEQUENCE_BIT = 12;   //序列号占用的位数
        private static final long MACHINE_BIT = 10;     //机器标识占用的位数
        private static final long DATA_CENTER_BIT = 0; //数据中心占用的位数

        /**
         * 每一部分的最大值
         */
        private static final long MAX_SEQUENCE = ~(-1L << SEQUENCE_BIT);
        private static final long MAX_MACHINE_NUM = ~(-1L << MACHINE_BIT);
        private static final long MAX_DATA_CENTER_NUM = ~(-1L << DATA_CENTER_BIT);

        /**
         * 每一部分向左的位移
         */
        private static final long MACHINE_LEFT = SEQUENCE_BIT;
        private static final long DATA_CENTER_LEFT = SEQUENCE_BIT + MACHINE_BIT;
        private static final long TIMESTAMP_LEFT = DATA_CENTER_LEFT + DATA_CENTER_BIT;

        private final long dataCenterId = 0;  //数据中心
        private final long machineId;     //机器标识
        private long sequence = 0L; //序列号
        private long lastTimeStamp = -1L;  //上一次时间戳

        private long getNextMill() {
            long mill = getNewTimeStamp();
            while (mill <= lastTimeStamp) {
                mill = getNewTimeStamp();
            }
            return mill;
        }

        private long getNewTimeStamp() {
            return System.currentTimeMillis();
        }

        /**
         * 根据指定的数据中心ID和机器标志ID生成指定的序列号
         *
         * @param machineId 机器标志ID
         */
        public Generator(long machineId) {
            if (machineId > MAX_MACHINE_NUM || machineId < 0) {
                throw new IllegalArgumentException("MachineId can't be greater than MAX_MACHINE_NUM or less than 0！");
            }

            this.machineId = machineId;
        }

        /**
         * 产生下一个ID
         */
        public synchronized long nextId() {
            long currTimeStamp = getNewTimeStamp();
            if (currTimeStamp < lastTimeStamp) {
                throw new RuntimeException("Clock moved backwards.  Refusing to generate id");
            }

            if (currTimeStamp == lastTimeStamp) {
                //相同毫秒内，序列号自增
                sequence = (sequence + 1) & MAX_SEQUENCE;
                //同一毫秒的序列数已经达到最大
                if (sequence == 0L) {
                    currTimeStamp = getNextMill();
                }
            } else {
                //不同毫秒内，序列号置为0
                sequence = 0L;
            }

            lastTimeStamp = currTimeStamp;

            return (currTimeStamp - START_TIMESTAMP) << TIMESTAMP_LEFT //时间戳部分
                    | dataCenterId << DATA_CENTER_LEFT       //数据中心部分
                    | machineId << MACHINE_LEFT             //机器标识部分
                    | sequence;                             //序列号部分
        }
    }
}
