package com.xforceplus.taxware.leqi.kernel.contract.model.util;

import com.xforceplus.taxware.leqi.kernel.contract.model.exception.Sm4DecryptException;
import com.xforceplus.taxware.leqi.kernel.contract.model.exception.Sm4EncryptException;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.binary.Hex;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.jetbrains.annotations.NotNull;

import javax.crypto.Cipher;
import javax.crypto.spec.SecretKeySpec;
import java.security.Security;

import static java.nio.charset.StandardCharsets.UTF_8;

/**
 * 国密SM4
 *
 * @author Bobo
 * @create 2023/3/20 15:34
 * @since 1.0.0
 */
public class SM4Utils {

    public static final String ALGORITHM_NAME = "SM4";
    // 加密算法/分组加密模式/分组填充方式
    // PKCS5Padding-以8个字节为一组进行分组加密
    // 定义分组加密模式使用：PKCS5Padding
    public static final String ALGORITHM_NAME_ECB_PADDING = "SM4/ECB/PKCS5Padding";

    static {
        Security.addProvider(new BouncyCastleProvider());
    }

    /**
     * 加密
     *
     * @param secretKey 密钥
     * @param content   待加密内容
     * @return
     */
    public static String encrypt(@NotNull String secretKey, @NotNull String content) {

        try {

            final var cipher = generateCipher(Cipher.ENCRYPT_MODE, secretKey);

            final var encodeContent = Base64.encodeBase64(cipher.doFinal(content.getBytes(UTF_8)));

            return new String(encodeContent, UTF_8);

        } catch (Exception e) {
            throw new Sm4EncryptException(e);
        }

    }

    /**
     * 解密
     *
     * @param secretKey 密钥
     * @param content   待解密内容
     * @return
     */
    public static String decrypt(@NotNull String secretKey, @NotNull String content) {

        try {

            final var cipher = generateCipher(Cipher.DECRYPT_MODE, secretKey);

            return new String(cipher.doFinal(Base64.decodeBase64(content)), UTF_8);

        } catch (Exception e) {
            throw new Sm4DecryptException(e);
        }

    }

    private static Cipher generateCipher(int mode, String secretKey) throws Exception {
        final var cipher = Cipher.getInstance(ALGORITHM_NAME_ECB_PADDING, BouncyCastleProvider.PROVIDER_NAME);
        final var sm4Key = new SecretKeySpec(Hex.decodeHex(secretKey.toCharArray()), ALGORITHM_NAME);
        cipher.init(mode, sm4Key);
        return cipher;
    }
}
