package com.xforceplus.taxware.contract.allelectric.message;

import lombok.Data;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;

/**
 * 发票开具结果回调消息
 * <p>
 * https://xforceplus.yuque.com/dew9bm/yagacb/buwgsn
 * 优惠政策枚举值详见 ： https://xforceplus.yuque.com/dew9bm/yagacb/qwzueume97ouom15
 */
@Data
public class InvoiceResultMessage {

    @Data
    public static class RequestProperties {

        /**
         * 消息总线规定的业务字段
         */
        private String businessNo;

        /**
         * 调用方标志
         */
        private String fromSystem;

        /**
         * 租户id
         */
        private String tenantId;
    }

    /**
     * 1 = 成功。其他返回具体的错误码
     **/
    private String code = "1";

    /**
     * 错误信息
     **/
    private String message;

    /**
     * 结果
     **/
    private Result result;

    @Data
    public static class Result {

        /**
         * 请求返回的流水号
         */
        private String taskId;

        /**
         * 预制发票id
         **/
        private String pid;

        /**
         * 发票号码 [20]
         **/
        private String invoiceNo;

        /**
         * 开票日期时间(yyyyMMddHHmmss) 14位
         */
        private String dateIssued;

        /**
         * 纸票发票代码
         * 仅纸票开具结果返回
         */
        private String paperInvoiceCode;

        /**
         * 纸票发票号码
         * 仅纸票开具结果返回
         */
        private String paperInvoiceNo;

        /**
         * 纸票密文区
         * 仅纸票开具结果返回
         */
        private String ciperText;

        /**
         * 发票类型
         * qc:全电电普
         * qs:全电电专
         **/
        private String invoiceType;

        /**
         * 剩余开具额度（18，2）
         */
        private BigDecimal creditLimit;

        /**
         * 特定要素类型代码
         * (枚举值详见
         * https://xforceplus.yuque.com/dew9bm/yagacb/qwzueume97ouom15
         * 一、全电特定业务枚举)
         **/
        private String invoiceStyleType;

        /**
         * 是否乐企标识（不传默认为false，如果传为true，走乐企通道）
         */
        private Boolean isNaturalSystemFlag;

        /**
         * 二维码地址（200）
         */
        private String qrCodeUrl;

        /**
         * 开票控制项
         */
        private InvoiceControl control;

        @Data
        public static class InvoiceControl {

            /**
             * 差额开票枚举项
             * 空：不适用差额征税
             * difference：差额征税差额开票
             * full：差额征税全额开票
             */
            private String taxDifferenceFlag;

            /**
             * 是否以客戶端返回的全票面为准返回的发票数据控制项
             * （true：客户端返回的全票面 false：取预制发票数据）
             * 当值为true时 明细的含税金额由税件反算
             */
            private Boolean clientReturnFlag;
        }

        /**
         * 销方信息
         **/
        private SellerInfo seller;

        @Data
        public static class SellerInfo {

            /**
             * 销方名称 [180]
             **/
            private String sellerName;

            /**
             * 销方税号 [20]
             **/
            private String sellerTaxNo;

            /**
             * 销方地址 [180]
             **/
            private String sellerAddress;

            /**
             * 销方电话 [20]
             **/
            private String sellerTel;

            /**
             * 销方开户行 [80]
             **/
            private String sellerBankName;

            /**
             * 销方开户行账号 [40]
             **/
            private String sellerBankAccount;
        }

        /**
         * 购方信息
         **/
        private BuyerInfo buyer;

        @Data
        public static class BuyerInfo {

            /**
             * 购方名称 [180]
             **/
            private String buyerName;

            /**
             * 购方税号 [20]
             **/
            private String buyerTaxNo;

            /**
             * 购方地址 [180]
             **/
            private String buyerAddress;

            /**
             * 购方电话 [20]
             **/
            private String buyerTel;

            /**
             * 购方开户行 [80]
             **/
            private String buyerBankName;

            /**
             * 购方开户行账号 [40]
             **/
            private String buyerBankAccount;

            /**
             * 是否开票给自然人标识
             */
            private Boolean isNatureFlag = false;
        }

        /**
         * 操作人信息
         **/
        private OperatorInfo operator;

        @Data
        public static class OperatorInfo {

            /**
             * 开票人
             **/
            private String issuer;
        }

        /**
         * 发票金额信息
         **/
        private InvoiceAmountInfo invoiceAmount;

        @Data
        public static class InvoiceAmountInfo {

            /**
             * 合计金额（18位，2位小数）
             **/
            private BigDecimal amountWithoutTax;

            /**
             * 价税合计（18位，2位小数）
             **/
            private BigDecimal amountWithTax;

            /**
             * 合计税额（18位，2位小数）
             **/
            private BigDecimal taxAmount;
        }

        /**
         * 红字信息
         **/
        private RedLetterInfo redLetter;

        @Data
        public static class RedLetterInfo {

            /**
             * 原发票号码(本期不包含) [20]
             **/
            private String originalInvoiceNo;

            /**
             * 红字确认单编号(本期不包含) [32]
             **/
            private String redLetterNumber;
        }

        /**
         * 备注 [200]
         **/
        private String remark;

        /**
         * 价格方式(1-含税,0-不含税，默认: 不含税)
         * 本期不包含含税开具
         */
        private String pricingMethod;

        /**
         * 明细列表
         **/
        private List<Details> details;

        @Data
        public static class Details {

            /**
             * 明细pid
             **/
            private String pid;

            /**
             * 明细折扣类型（0:普通明细;1:折扣明细;2:被折扣明细）
             */
            private String discountType;

            /**
             * 明细金额信息
             **/
            private DetailAmountInfo detailAmount;

            @Data
            public static class DetailAmountInfo {

                /**
                 * 数量（16位）（单位非空时非空）
                 **/
                private BigDecimal quantity;

                /**
                 * 不含税单价（21位）（单位非空时非空）
                 **/
                private BigDecimal unitPrice;

                /**
                 * 不含税金额（18位，2位小数）
                 **/
                private BigDecimal amountWithoutTax;

                /**
                 * 税额（18位，2位小数）
                 **/
                private BigDecimal taxAmount;

                /**
                 * 含税金额（18位，2位小数）
                 * 当控制项：clientReturnFlag 为true时 含税金额由税件反算
                 **/
                private BigDecimal amountWithTax;
            }

            /**
             * 商品信息
             **/
            private ItemInfo item;

            @Data
            public static class ItemInfo {

                /**
                 * 商品名称 [100]
                 **/
                private String itemName;

                /**
                 * 规格型号 [40]
                 **/
                private String specifications;

                /**
                 * 单位 [20]
                 **/
                private String unit;
            }

            /**
             * 税务信息
             **/
            private TaxInfo tax;

            @Data
            public static class TaxInfo {

                /**
                 * 税率
                 **/
                private BigDecimal taxRate;

                /**
                 * 税编 [19]
                 **/
                private String goodsTaxNo;

                /**
                 * 税收优惠政策类型代码（枚举值详见：
                 * https://xforceplus.yuque.com/dew9bm/yagacb/qwzueume97ouom15
                 * 二、全电优惠政策类型枚举）
                 **/
                private String taxIncentivesType;

                /**
                 * 全电即征即退补充枚举 (当taxIncentivesType为11-18时需要传入)
                 * (枚举值详见:
                 * https://xforceplus.yuque.com/dew9bm/yagacb/qwzueume97ouom15
                 * 三、全电即征即退类型枚举)
                 */
                private String vatRefundType;
            }
        }

        /**
         * 特定要素扩展字段
         */
        private List<Map<String, Object>> extra;
    }
}
