package com.xforceplus.taxware.chestnut.check.model.base;

import com.xforceplus.taxware.chestnut.check.model.common.Alias;
import lombok.Data;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.validator.constraints.Length;
import org.hibernate.validator.constraints.ScriptAssert;

import javax.validation.constraints.Digits;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import javax.validation.constraints.Pattern;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;

/**
 * taxware-chestnut-contract
 *
 * @Author 👽 Rayder
 * @Date 📅 2023/2/24 16:58
 * @Motto: Tools change the world. 🛠
 * @Copyright: xforceplus 🇨🇳
 * @Description:
 */
@ScriptAssert.List(
        {
                @ScriptAssert(
                        lang = "javascript",
                        script = "com.xforceplus.taxware.chestnut.check.model.base.BaseDetail.isValidUnionInfo(_this)",
                        message = "数量，单价和单位有一项是非空，则另外两项也要非空"
                ),
        }
)
@Data
public class BaseDetail {

    /**
     * 蓝字发票明细序号（非必须）
     * 红票必传
     */
    @Alias("蓝字发票明细序号")
    @Digits(integer = 8, fraction = 0)
    private Integer originalRowNum;

    /**
     * 序号
     * 8位，从1开始。
     */
    @Alias("序号")
    @NotNull
    @Digits(integer = 8, fraction = 0)
    private Integer rowNum;

    /**
     * 商品和服务税收分类合并编码
     */
    @Alias("商品和服务税收分类合并编码")
    @NotEmpty
    @Length(max = 19)
    private String goodsTaxNo;

    /**
     * 项目名称
     * 格式必须是“*简称*商品名称 todo-rayder 是否校验这个格式？-另外需要确认关系：项目名称 = 商品服务简称+货物或应税劳务、服务名称
     */
    @Alias("项目名称")
    @NotEmpty
    @Length(max = 600)
    @Pattern(regexp = "\\*.+?\\*.+")
    private String itemName;

    /**
     * 规格型号（非必须）
     */
    @Alias("规格型号")
    @Length(max = 150)
    private String specifications;

    /**
     * 单位（非必须）
     * 如“数量”“单价”栏次非空，则本栏次必须非空
     */
    @Alias("单位")
    @Length(max = 300)
    private String unit;

    /**
     * 单价（非必须）
     * 如“数量”“单位”栏次非空，则本栏次必须非空
     * 文档上还有一个说法，选填（如“单位”非空，则必填）todo-rayder 以哪个为准？待验证
     */
    @Alias("单价")
    @Length(max = 25)
    private String unitPrice;

    /**
     * 数量（非必须）
     * 如“单价”“单价”栏次非空，则本栏次必须非空
     * 文档上还有一个说法，选填（如“单位”非空，则必填）todo-rayder 以哪个为准？待验证
     */
    @Alias("数量")
    @Length(max = 25)
    private String quantity;

    /**
     * 金额
     */
    @Alias("金额")
    @NotNull
    @Digits(integer = 18, fraction = 2)
    private BigDecimal amountWithoutTax;

    /**
     * 税率
     * 必须在可用税率信息中存在。//todo-rayder 可用税率业务校验
     */
    @Alias("税率")
    @NotNull
    @Digits(integer = 16, fraction = 6)
    private BigDecimal taxRate;

    /**
     * 税额
     */
    @Alias("税额")
    @NotNull
    @Digits(integer = 18, fraction = 2)
    private BigDecimal taxAmount;

    /**
     * 数量，单价和单位有一项是非空，则另外两项也要非空"
     */
    public static boolean isValidUnionInfo(BaseDetail detail) {
        if (detail.getUnit() != null && detail.getUnit().trim().length() > 0
                || detail.getUnitPrice() != null && detail.getUnitPrice().trim().length() > 0
                || detail.getQuantity() != null && detail.getQuantity().trim().length() > 0) {
            return detail.getUnit() != null && detail.getUnit().trim().length() > 0
                    && detail.getUnitPrice() != null && detail.getUnitPrice().trim().length() > 0
                    && detail.getQuantity() != null && detail.getQuantity().trim().length() > 0;
        }
        return true;

    }

    public String getItemShortName() {
        if (StringUtils.isBlank(this.itemName)) {
            return null;
        }

        // 根据标准的商品名称（*AA*BBB） 拆分出商品名称和商品税目名称
        final var split = Arrays.asList(StringUtils.split(this.itemName, "*"));
        final var spmcsm = new ArrayList<>(split);

        if (spmcsm.size() < 2 && !this.itemName.startsWith("*")) {
            return null;
        }

        return spmcsm.get(0);
    }
}
