package com.xforceplus.taxware.chestnut.check.model.validator.redletter;

import cn.hutool.core.util.StrUtil;
import com.xforceplus.taxware.architecture.g1.domain.exception.TXWR000002Exception;
import com.xforceplus.taxware.chestnut.check.model.base.InvoiceStatusInfo;
import com.xforceplus.taxware.chestnut.check.model.common.InvoiceStatusProvider;
import com.xforceplus.taxware.chestnut.check.model.common.RedInvoiceHistoryProvider;
import com.xforceplus.taxware.chestnut.check.model.constant.ApplyReasonEnum;
import com.xforceplus.taxware.chestnut.check.model.validator.invoice.InvoiceBaseValidator;

import java.math.BigDecimal;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * @author lv
 * @Date 2023/2/16
 * @Descrption 申请红字确认单时：红字确认单信息与原蓝字发票信息进行比对
 *
 * 3.2.3.2数据校验
 * 含税金额=税额+金额，不能有误差。
 * 红票开具金额<= 对应蓝字发票开具金额。
 * 3.2.3.3业务逻辑校验
 * 红字发票的商品金额税额不为正数。
 * 对应全电蓝字发票带折扣行时，将折扣金额、税额分摊反映在对应的项目名称行上，再和红字发票进行比对。
 * 没有红字发票类型字段。蓝字增值税专用发票只能用增值税专用发票冲红；蓝字普通发票只能用普通发票进行冲红。
 * 对应全电蓝字发票开具过红字发票时，系统对冲红的“对应蓝字发票明细序号”行累计数量、累计金额和累计税额进行监控。当累计开具红字发票的数量以及负数金额或者税额绝对值之和超过原全电蓝字发票时，不允许其继续开具红字发票。
 * 当蓝字发票对应的“增值税用途标签”为空、“消费税用途标签”为“未勾选库存”、“入账状态标签”为“未入账”或“已入账撤销”时，红字信息只允许销售方发起（除非蓝字发票为收购发票）且只能进行全额冲红。
 * “冲红原因”为“开票有误”时，必须全额冲红，并且明细单价、金额、数量必须和蓝字发票保持一致。
 * 当蓝字发票对应的“增值税优惠用途标签”为“待农产品全额加计扣除”或“已用于农产品全额加计扣除”的，必须全额红冲；
 *        todo-rayder “待农产品部分加计扣除”或“已用于农产品部分加计扣除”的，第一次红冲只能对未加计部分全额冲红或对这张蓝票全额红冲，第二次红冲仅允许对剩余部分（即已加计部分）全额红冲。
 *   “红冲原因”为“销货退回”时，只允许修改数量，不允许修改单价。
 *   “红冲原因”为“服务终止”时，允许修改总金额和数量，不允许修改单价。
 *   “红冲原因”为“销售折让”时，不能修改单价和数量。
 *   商品服务编码为以1、2开头的冲红原因不允许选择“服务中止”，商品服务编码为以3开头的冲红原因不允许选择“销售退回”，兼营销售则不对红字原因进行控制。
 */
public class RedLetterVsOriginalInvoiceValidator {

    /**
     * 默认业务校验前提，合规性校验和数据校验均已通过
     */
    public static void validate(final RedLetterValidator redLetter,
                                final InvoiceBaseValidator invoiceBase,
                                final RedInvoiceHistoryProvider redInvoiceHistoryProvider,
                                final InvoiceStatusProvider invoiceStatusProvider) {

        BaseBusinessCheck(redLetter, invoiceBase);

        // 记录原蓝票明细索引
        Map<Integer, InvoiceBaseValidator.InvoiceDetail> indexInfo = new HashMap();
        invoiceBase.getInvoiceDetailList().forEach(detail -> indexInfo.put(detail.getRowNum(), detail));

        // 记录商品服务编码混合的情况
        Map<String, Boolean> goodsTaxNoStartWithMixedCase = new HashMap();

        for (var detail : redLetter.getDetails()) {
            InvoiceBaseValidator.InvoiceDetail originalDetail = indexInfo.get(detail.getOriginalRowNum());

            handleDiscountCase(indexInfo, detail, originalDetail);

            checkNegativeAmount(detail);

            checkModifyInfoWithApplyReason(redLetter, detail, originalDetail);

            //记录商品编码第一位混合的情况
            String goodsTaxNoStartWithValue = detail.getGoodsTaxNo().substring(0, 1);
            if (!goodsTaxNoStartWithMixedCase.containsKey(goodsTaxNoStartWithValue)) {
                goodsTaxNoStartWithMixedCase.put(goodsTaxNoStartWithValue, true);
            }
        }

        // 校验税收分类编码和红冲原因匹配关系
        checkGoodsTaxNoWithApplyReason(redLetter, goodsTaxNoStartWithMixedCase);

        // 历史累计开具红票的校验
        checkHistoryRedInvoiceDetailLimit(redLetter, redInvoiceHistoryProvider, indexInfo);

        //校验发票勾选状态
        checkInvoiceCheckStatus(redLetter, invoiceBase, invoiceStatusProvider);

    }

    /**
     * 基础信息业务校验
     *
     * @param redLetter
     * @param invoiceBase
     */
    private static void BaseBusinessCheck(RedLetterValidator redLetter, InvoiceBaseValidator invoiceBase) {
        var originalAmountInfo = redLetter.getOriginalInvoiceInfo().getInvoiceAmountInfo();

        //todo-rayder 红字确认单没有含税金额？
//        if (originalAmountInfo.getAmountWithTax().compareTo(originalAmountInfo.getTaxAmount().add(originalAmountInfo.getAmountWithoutTax())) != 0) {
//            throw new TXWR000002Exception("原蓝票含税金额不等于税额加金额之和");
//        }

        if (redLetter.getReverseAmountWithoutTax().abs().compareTo(invoiceBase.getInvoiceAmountInfo().getAmountWithoutTax()) > 0) {
            throw new TXWR000002Exception("红票开具金额不能大于对应蓝字发票开具金额");
        }

        // 发票类型校验
        if (
                !invoiceBase.getInvoiceBaseInfo().getInvoiceType().equals(redLetter.getOriginalInvoiceInfo().getOriginalInvoiceType())
        ) {
            throw new TXWR000002Exception("发票类型不匹配");
        }
    }


    /**
     * 折扣分摊处理，明细比较前必须先处理
     *
     * @param indexInfo
     * @param detail
     * @param originalDetail
     */
    private static void handleDiscountCase(Map<Integer, InvoiceBaseValidator.InvoiceDetail> indexInfo, RedLetterValidator.Detail detail, InvoiceBaseValidator.InvoiceDetail originalDetail) {
        if ("02".equals(originalDetail.getDiscountType())) {
            InvoiceBaseValidator.InvoiceDetail discountDetail = indexInfo.get(detail.getOriginalRowNum() + 1);
            if (!"01".equals(discountDetail.getDiscountType())) {
                throw new TXWR000002Exception(String.format("第%s行明细对应原蓝票折扣信息标志不匹配", detail.getRowNum()));
            }
            if (discountDetail.getTaxAmount().compareTo(new BigDecimal(0)) > 0
                    || discountDetail.getAmountWithoutTax().compareTo(new BigDecimal(0)) > 0
                    || !StrUtil.isEmpty(discountDetail.getQuantity()) && new BigDecimal(discountDetail.getQuantity()).compareTo(new BigDecimal(0)) > 0
            ) {
                throw new TXWR000002Exception(String.format("第%s行明细对应原蓝票折扣行的数量、金额或税额不能大于0", detail.getRowNum()));
            }
            originalDetail.setTaxAmount(originalDetail.getTaxAmount().add(discountDetail.getTaxAmount()));
            originalDetail.setAmountWithoutTax(originalDetail.getAmountWithoutTax().add(discountDetail.getAmountWithoutTax()));
        }
    }

    /**
     * 校验是否是负数金额税额
     *
     * @param detail
     */
    private static void checkNegativeAmount(RedLetterValidator.Detail detail) {
        if (detail.getAmountWithoutTax().compareTo(BigDecimal.ZERO) > 0) {
            throw new TXWR000002Exception(String.format("第%s行明细金额不能为正数", detail.getRowNum()));
        }
        if (detail.getTaxAmount().compareTo(BigDecimal.ZERO) > 0) {
            throw new TXWR000002Exception(String.format("第%s行明细税额不能为正数", detail.getRowNum()));
        }
    }

    /**
     * 校验红冲的明细与红冲原因的匹配关系
     *
     * @param redLetter
     * @param detail
     * @param originalDetail
     */
    private static void checkModifyInfoWithApplyReason(RedLetterValidator redLetter, RedLetterValidator.Detail detail, InvoiceBaseValidator.InvoiceDetail originalDetail) {
        if (
                !Objects.equals(originalDetail.getUnit(), detail.getUnit())
                        || !Objects.equals(originalDetail.getGoodsTaxNo(), detail.getGoodsTaxNo())
                        || !Objects.equals(originalDetail.getItemName(), detail.getItemName())
                        || originalDetail.getTaxRate().compareTo(detail.getTaxRate()) != 0
        ) {
            throw new TXWR000002Exception(String.format("第%s行明细和原蓝票明细商品内容存在不一致的情况", detail.getRowNum()));
        }

        // 数量，单价为空的情况,则不校验数量和单价
        Boolean isNeedCheckQuantityOrUnitPrice = true;
        if (StrUtil.isEmpty(detail.getQuantity())
                && StrUtil.isEmpty(originalDetail.getQuantity())
                && StrUtil.isEmpty(detail.getUnitPrice())
                && StrUtil.isEmpty(originalDetail.getUnitPrice())
        ) {
            isNeedCheckQuantityOrUnitPrice = false;
        } else if (
                StrUtil.isEmpty(detail.getQuantity())
                        || StrUtil.isEmpty(originalDetail.getQuantity())
                        || StrUtil.isEmpty(detail.getUnitPrice())
                        || StrUtil.isEmpty(originalDetail.getUnitPrice())
        ) {
            throw new TXWR000002Exception(String.format("第%s行明细和原蓝票数量和单价存在为空不一致情况", detail.getRowNum()));
        }

        // 准备比较条件
        Boolean isMatchQuantity = isNeedCheckQuantityOrUnitPrice
                && new BigDecimal(originalDetail.getQuantity()).compareTo(new BigDecimal(detail.getQuantity()).abs()) == 0
                || !isNeedCheckQuantityOrUnitPrice;
        Boolean isMatchUnitPrice = isNeedCheckQuantityOrUnitPrice
                && new BigDecimal(originalDetail.getUnitPrice()).compareTo(new BigDecimal(detail.getUnitPrice()).abs()) == 0
                || !isNeedCheckQuantityOrUnitPrice;
        Boolean isMatchUnit = isNeedCheckQuantityOrUnitPrice
                && Objects.equals(originalDetail.getUnit(), detail.getUnit())
                || !isNeedCheckQuantityOrUnitPrice;
        Boolean isMatchTaxAmount = originalDetail.getTaxAmount().compareTo(detail.getTaxAmount().abs()) == 0;
        Boolean isMatchAmountWithoutTax = originalDetail.getAmountWithoutTax().compareTo(detail.getAmountWithoutTax().abs()) == 0;


        // 金额，数量正负匹配问题
        if (ApplyReasonEnum.WRONG.getValue().equals(redLetter.getApplyReason())) {
            if (!isMatchUnitPrice
                    || !isMatchQuantity
                    || !isMatchUnit
                    || !isMatchTaxAmount
                    || !isMatchAmountWithoutTax) {
                throw new TXWR000002Exception(String.format("开具有误时，第%s行明细和原蓝票金额、数量和单价存在不一致情况", detail.getRowNum()));
            }
        } else if (ApplyReasonEnum.RETURN.getValue().equals(redLetter.getApplyReason())) {
            if (!isMatchUnitPrice
                    || !isMatchUnit) {
                throw new TXWR000002Exception(String.format("销货退回时，第%s行明细只允许修改数量", detail.getRowNum()));
            }

        } else if (ApplyReasonEnum.BREAK.getValue().equals(redLetter.getApplyReason())) {
            if (!isMatchUnitPrice
                    || !isMatchUnit) {
                throw new TXWR000002Exception(String.format("服务中止时，第%s行明细只允许修改金额、数量", detail.getRowNum()));
            }
        } else if (ApplyReasonEnum.DISCOUNT.getValue().equals(redLetter.getApplyReason())) {
            if (!isMatchUnitPrice
                    || !isMatchQuantity
                    || !isMatchUnit) {
                throw new TXWR000002Exception(String.format("销售折让时，第%s行明细不能修改单价和数量", detail.getRowNum()));
            }
        }
    }

    /**
     * 校验商品服务编码与红冲原因的匹配关系
     *
     * @param redLetter
     * @param goodsTaxNoStartWithMixedCase
     */
    private static void checkGoodsTaxNoWithApplyReason(RedLetterValidator redLetter, Map<String, Boolean> goodsTaxNoStartWithMixedCase) {
        boolean isGoods = goodsTaxNoStartWithMixedCase.containsKey("1")
                || goodsTaxNoStartWithMixedCase.containsKey("2");
        boolean isService = goodsTaxNoStartWithMixedCase.containsKey("3");
        
        // 如果商品编码出现了混合多个的情况，则认为是兼营销售，不校验红冲原因
        if (isGoods && isService) {
            // 兼营销售不校验原因
            return;
        }

        // 货物明细
        if (isGoods) {
            if (ApplyReasonEnum.BREAK.getValue().equals(redLetter.getApplyReason())) {
                throw new TXWR000002Exception("商品编码是1或2打头，红冲原因不允许选择服务中止");
            }
        }

        // 服务明细
        if (isService) {
            if (ApplyReasonEnum.RETURN.getValue().equals(redLetter.getApplyReason())) {
                throw new TXWR000002Exception("商品编码是3打头，红冲原因不允许选择销售退回");
            }
        }
    }


    private static void checkHistoryRedInvoiceDetailLimit(RedLetterValidator redLetter, RedInvoiceHistoryProvider redInvoiceHistoryProvider, Map<Integer, InvoiceBaseValidator.InvoiceDetail> indexInfo) {
        if (redInvoiceHistoryProvider == null) {
            return;
        }
        List<InvoiceBaseValidator> redInvoiceHistoryList = redInvoiceHistoryProvider.provide();

        Map<Integer, BigDecimal> detailTotalAmountWithoutTax = new HashMap<>();
        Map<Integer, BigDecimal> detailTotalTaxAmount = new HashMap<>();
        Map<Integer, BigDecimal> detailTotalQuantity = new HashMap<>();
        redInvoiceHistoryList.forEach(redInvoiceHistory -> {
            redInvoiceHistory.getInvoiceDetailList().forEach(detail -> {
                if (detail.getOriginalRowNum() == null) {
                    throw new TXWR000002Exception(String.format("查询到的历史已开红票明细对应原蓝票明细数据缺失"));
                }
                if (detailTotalAmountWithoutTax.containsKey(detail.getOriginalRowNum())) {
                    detailTotalAmountWithoutTax.get(detail.getOriginalRowNum()).add(detail.getAmountWithoutTax());
                } else {
                    detailTotalAmountWithoutTax.put(detail.getOriginalRowNum(), detail.getAmountWithoutTax());
                }

                if (detailTotalTaxAmount.containsKey(detail.getOriginalRowNum())) {
                    detailTotalTaxAmount.get(detail.getOriginalRowNum()).add(detail.getTaxAmount());
                } else {
                    detailTotalTaxAmount.put(detail.getOriginalRowNum(), detail.getTaxAmount());
                }

                if (!StrUtil.isEmpty(detail.getQuantity()) && detailTotalQuantity.containsKey(detail.getOriginalRowNum())) {
                    detailTotalQuantity.get(detail.getOriginalRowNum()).add(new BigDecimal(detail.getQuantity()));
                } else {
                    detailTotalQuantity.put(detail.getOriginalRowNum(), new BigDecimal(detail.getQuantity()));
                }

            });
        });
        redLetter.getDetails().forEach(redLetterDetail -> {
            InvoiceBaseValidator.InvoiceDetail originalDetail = indexInfo.get(redLetterDetail.getOriginalRowNum());
            if (detailTotalAmountWithoutTax.get(redLetterDetail.getRowNum()) != null) {
                if (detailTotalAmountWithoutTax.get(redLetterDetail.getRowNum()).add(redLetterDetail.getAmountWithoutTax()).add(originalDetail.getAmountWithoutTax()).compareTo(new BigDecimal(0)) < 0) {
                    throw new TXWR000002Exception(String.format("第%s行明细红冲金额超过原蓝票可红冲剩余额度", redLetterDetail.getRowNum()));
                }
            }
            if (detailTotalTaxAmount.get(redLetterDetail.getRowNum()) != null) {
                if (detailTotalTaxAmount.get(redLetterDetail.getRowNum()).add(redLetterDetail.getTaxAmount()).add(originalDetail.getTaxAmount()).compareTo(new BigDecimal(0)) < 0) {
                    throw new TXWR000002Exception(String.format("第%s行明细红冲税额超过原蓝票可红冲剩余额度", redLetterDetail.getRowNum()));
                }
            }
            if (detailTotalQuantity.get(redLetterDetail.getRowNum()) != null && !StrUtil.isEmpty(redLetterDetail.getQuantity())) {
                if (detailTotalQuantity.get(redLetterDetail.getRowNum()).add(new BigDecimal(redLetterDetail.getQuantity())).add(new BigDecimal(originalDetail.getQuantity())).compareTo(new BigDecimal(0)) < 0) {
                    throw new TXWR000002Exception(String.format("第%s行明细红冲数量超过原蓝票可红冲剩余额度", redLetterDetail.getRowNum()));
                }
            }
        });
    }

    private static void checkInvoiceCheckStatus(RedLetterValidator redLetter, InvoiceBaseValidator invoiceBase, InvoiceStatusProvider invoiceStatusProvider) {
        if (invoiceStatusProvider == null) {
            return;
        }

        Boolean isMatchTaxAmount = invoiceBase.getInvoiceAmountInfo().getTaxAmount().compareTo(redLetter.getReverseTaxAmount().abs()) == 0;
        Boolean isMatchAmountWithoutTax = invoiceBase.getInvoiceAmountInfo().getAmountWithoutTax().compareTo(redLetter.getReverseAmountWithoutTax().abs()) == 0;

        // 全额判断条件
        Boolean isFull = isMatchAmountWithoutTax && isMatchTaxAmount;
        InvoiceStatusInfo invoiceStatusInfo = invoiceStatusProvider.provide();
        if (
                (StrUtil.isEmpty(invoiceStatusInfo.getVatUsageLabel())
                        || "0".equals(invoiceStatusInfo.getVatUsageLabel()))
                        && ("01".equals(invoiceStatusInfo.getEntryStatus())
                        || "03".equals(invoiceStatusInfo.getEntryStatus()))
        ) {
            // 收购发票条件
            Boolean isRecycling = "16".equals(invoiceBase.getInvoiceBaseInfo().getInvoiceStyleType()) || "17".equals(invoiceBase.getInvoiceBaseInfo().getInvoiceStyleType());
            if (!isRecycling && !"0".equals(redLetter.getApplyIdentity())) {
                throw new TXWR000002Exception("原蓝票未入账且未勾选，只能销方发起");
            }
            if (!isFull) {
                throw new TXWR000002Exception("原蓝票未入账且未勾选，只能全额红冲");
            }
        }

        if ("12".equals(invoiceStatusInfo.getVatUsageLabel())) {
            if (!isFull) {
                throw new TXWR000002Exception("原蓝票是用于农产品全额加计扣除，只能全额红冲");
            }
        }
    }
}
