package com.xforceplus.taxware.chestnut.check.model.validator.tax;

import com.xforceplus.taxware.architecture.g1.domain.exception.TXWR000004Exception;
import com.xforceplus.taxware.chestnut.check.model.base.TaxCode;
import com.xforceplus.taxware.chestnut.contract.model.constant.enums.TaxCodeStatusEnum;
import org.apache.commons.lang3.StringUtils;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.ZoneId;
import java.util.Date;

/**
 * @Auther lv
 * @Date 2023/2/24
 * 失败情况：
 * 1。税编记录不存在
 * 2。税编不是叶子节点：sfhzx：Y
 * 3。不在有效期内：小于qyrq启用日期 || 大于tyrq停用日期
 * 4。简称是否跟查询到的税编一致
 * todo 甄孟莹 5。税率范围应在多税率范围内：TXWRECMI0014-发票明细有 1 行不合法。第 2 行【满口香】，税编是 1030110040000000000：税率不合法!请使用如下税率:0.09；
 */
public class TaxCodeValidator {
    public static void validate(String itemName, TaxCode taxCode) {
        if (null == taxCode) {
            throw new TXWR000004Exception("税编不存在，禁止开票");
        }

        if (TaxCodeStatusEnum.TAX_RATE_STATUS_Y.getStatus().equals(taxCode.getSummarizeItemFlag())) {
            throw new TXWR000004Exception("该税编是汇总项，禁止开票");
        }

        if (!StringUtils.equals(taxCode.getItemShortName(), itemName)) {
            throw new TXWR000004Exception("税编有误，禁止开票");
        }
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
        Date now = Date.from(LocalDate.now().atStartOfDay(ZoneId.systemDefault()).toInstant());

        try {
            Date startDate = format.parse(taxCode.getStartDate());
            Date endDate = format.parse(taxCode.getEndDate());

            // 启用日期不可为空
            if (null == startDate || startDate.compareTo(now) > 0) {
                throw new TXWR000004Exception("税编不在有效期范围内，禁止开票");
            }

            // 停用日期可为空
            if (null != endDate && endDate.compareTo(now) < 0) {
                throw new TXWR000004Exception("税编不在有效期范围内，禁止开票");
            }
        } catch (ParseException e) {
            throw new TXWR000004Exception("税编有误，禁止开票");
        }
    }
}
