package com.xforceplus.taxware.chestnut.check.model.validator.invoice;

import com.alibaba.fastjson.annotation.JSONField;
import com.xforceplus.taxware.chestnut.check.model.base.*;
import com.xforceplus.taxware.chestnut.check.model.common.Alias;
import com.xforceplus.taxware.chestnut.check.model.common.ValidateResult;
import com.xforceplus.taxware.chestnut.check.model.util.BasicValidator;
import lombok.Data;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.validator.constraints.Length;
import org.hibernate.validator.constraints.ScriptAssert;

import javax.validation.constraints.Digits;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import javax.validation.constraints.Pattern;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.List;

/**
 * @author zhenmengying
 * @date 2023/1/18
 * @describe: 全电发票上传
 */
@ScriptAssert.List(
        {
                @ScriptAssert(
                        lang = "javascript",
                        script = "com.xforceplus.taxware.chestnut.check.model.validator.invoice.InvoiceBaseValidator.isValidNotEmptyRedLetterInfo(_this)",
                        message = "标志为红票，对应红字确认单编号和原蓝票发票号码均不能为空"
                )
        }
)
@Data
public class InvoiceBaseValidator {

    private BaseInvoiceInfo invoiceBaseInfo;

    private SellerInfo sellerInfo;

    private BuyerInfo buyerInfo;

    private InvoiceAmountInfo invoiceAmountInfo;

    private OperatorInfo operatorInfo;

    private RedLetterInfo redLetterInfo;

    private List<InvoiceDetail> invoiceDetailList;


    public static boolean isValidNotEmptyRedLetterInfo(InvoiceBaseValidator invoiceInfo) {
        final var invoiceBaseInfo = invoiceInfo.getInvoiceBaseInfo();
        final var redInfo = invoiceInfo.getRedLetterInfo();
        if ("1".equals(invoiceBaseInfo.getInvoiceFlag()) && redInfo != null) {
            return redInfo.getRedLetterNumber() != null && redInfo.getRedLetterNumber().trim().length() > 0
                    && redInfo.getOriginalInvoiceNo() != null && redInfo.getOriginalInvoiceNo().trim().length() > 0;
        }
        return true;
    }

    /**
     * 校验蓝票明细
     * 数量，单价和单位有一项是非空，则另外两项也要非空"
     */
    /*public ValidateResult validBlueDetailUnionInfo(String invoiceFlag, InvoiceDetail detail) {

        if (Objects.equals(invoiceFlag, "0")
                && (detail.getUnitPrice() != null && detail.getUnitPrice().trim().length() > 0
                || detail.getQuantity() != null && detail.getQuantity().trim().length() > 0)) {
            final var isValid = detail.getUnitPrice() != null && detail.getUnitPrice().trim().length() > 0
                    && detail.getQuantity() != null && detail.getQuantity().trim().length() > 0;
            if (!isValid) {
                return ValidateResult.fail(String.format("第%d行数量，单价有一项是非空，则另外一项也要非空", detail.getRowNum()));
            }
        }

        return ValidateResult.success();
    }*/

    public ValidateResult validate() {

        // 先做主信息合规校验，不通过直接返回，不进行数据校验
        var result = BasicValidator.validate(this);
        result = BasicValidator.mergeValidateResult(result, BasicValidator.validate(this.getInvoiceBaseInfo()));
        result = BasicValidator.mergeValidateResult(result, BasicValidator.validate(this.getSellerInfo()));
        result = BasicValidator.mergeValidateResult(result, BasicValidator.validate(this.getBuyerInfo()));
        result = BasicValidator.mergeValidateResult(result, BasicValidator.validate(this.getInvoiceAmountInfo()));
        result = BasicValidator.mergeValidateResult(result, BasicValidator.validate(this.getOperatorInfo()));
        result = BasicValidator.mergeValidateResult(result, BasicValidator.validate(this.getRedLetterInfo()));

        final var detailList = this.getInvoiceDetailList();
        if (detailList == null || detailList.size() == 0) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("明细行数据不能为空"));
            return result;
        }

        // 先做明细信息合规校验，不通过直接返回，不再进行数据校验
        for (var detail : detailList) {
            final var validateResult = BasicValidator.validate(detail, detail.getRowNum() + "");
            result = BasicValidator.mergeValidateResult(result, validateResult);
            // result = BasicValidator.mergeValidateResult(result, this.validBlueDetailUnionInfo(this.getInvoiceBaseInfo().getInvoiceFlag(), detail));
        }

        if (result.isSuccess()) {
            result = BasicValidator.mergeValidateResult(result, this.validateData());
        }

        return result;
    }

    /**
     * 数据校验
     * 汇总金额校验：|所有商品行金额之和-合计金额|<=0.01。
     * 汇总税额校验：| 金额 * 税率（第1行）+......+金额 * 税率（第N行）-税额合计|<=1.27。
     * 折扣行校验：折扣行的商品名称应与原发票行一致。
     */
    private ValidateResult validateData() {
        var result = ValidateResult.success();

        // 校验蓝票的红票信息
        if ("0".equals(invoiceBaseInfo.getInvoiceFlag())) {
            if (StringUtils.isNotEmpty(this.redLetterInfo.getRedLetterNumber())
                    || StringUtils.isNotEmpty(this.redLetterInfo.getOriginalInvoiceNo())
                    || StringUtils.isNotEmpty(this.redLetterInfo.getOriginalDateIssued())
            ) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("蓝字发票对应的红字发票相关参数应为空"));
            }
        }

        final var invoiceTotalAmount = this.getInvoiceAmountInfo().getAmountWithTax();
        final var invoiceTotalTaxAmount = this.getInvoiceAmountInfo().getTaxAmount();
        if (invoiceTotalAmount.compareTo(invoiceTotalTaxAmount.add(this.getInvoiceAmountInfo().getAmountWithoutTax())) != 0) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("票面价税合计不等于金额加税额"));
        }
        // 红票不进行明细校验
        if ("1".equals(invoiceBaseInfo.getInvoiceFlag())) {
            return result;
        }
        var detailTotalAmount = BigDecimal.valueOf(0);
        var detailTotalTaxAmount = BigDecimal.valueOf(0);
        final var detailList = this.getInvoiceDetailList();
        for (int i = 0; i < detailList.size(); i++) {
            final var detail = detailList.get(i);
            final var validateDataResult = detail.validateData();
            result = BasicValidator.mergeValidateResult(result, validateDataResult);

            // 开具红票 对应蓝字发票明细序号不能为空
            if ("1".equals(this.getInvoiceBaseInfo().getInvoiceFlag())) {
                if (detail.getOriginalRowNum() == null) {
                    result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%s行对应蓝字发票明细序号不能为空", i + 1)));
                    return result;
                }
            }

            if ((i + 1) != detail.getRowNum()) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("明细行序号不是从1开始或不连续或重复"));
                return result;
            }
            // 折扣行校验
            if (detail.getRowNum() == 1) {
                if ("02".equals(detail.getDiscountType())) {
                    if (detailList.size() < 2) {
                        result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("第一行是被折扣行,明细行数据不能少于两行"));
                        return result;
                    }
                    if (!detailList.get(1).getDiscountType().equals("01")) {
                        result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("被折扣行的下一行必须为折扣行"));
                    }
                } else if ("01".equals(detail.getDiscountType())) {
                    result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("第一行不能为折扣行"));
                }
            }
            if ("01".equals(detail.getDiscountType())
                    && !detailList.get(i - 1).getItemName().equals(detail.getItemName())) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%s行折扣行商品名称和原发票行不一致", i + 1)));
            }
            detailTotalAmount = detailTotalAmount.add(detail.getAmountWithTax());
            detailTotalTaxAmount = detailTotalTaxAmount.add(detail.getAmountWithoutTax().multiply(detail.getTaxRate()));
        }

        if (detailTotalAmount.subtract(invoiceTotalAmount).abs().setScale(2, RoundingMode.HALF_UP).compareTo(BigDecimal.valueOf(0.01)) > 0) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("明细行合计金额超过最大误差0.01元"));
        }

        if (detailTotalTaxAmount.subtract(invoiceTotalTaxAmount).abs().setScale(2, RoundingMode.HALF_UP).compareTo(BigDecimal.valueOf(1.27)) > 0) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("明细行合计税额超过最大误差1.27元"));
        }

        return result;
    }

    @Data
    public static class InvoiceDetail extends BaseDetail {

        /**
         * 含税金额
         */
        @Alias("含税金额")
        @NotNull
        @Digits(integer = 18, fraction = 2)
        @JSONField(name = "hsje")
        private BigDecimal amountWithTax;

        /**
         * 扣除额（非必须）
         */
        @Alias("扣除额")
        @Digits(integer = 18, fraction = 2)
        @JSONField(name = "kce")
        private BigDecimal deduction;


        /**
         * 发票行性质
         * 00：正常行
         * 01：折扣行
         * 02：被折扣行
         */
        @Alias("发票行性质")
        @NotEmpty
        @Length(max = 2)
        @Pattern(regexp = "0[0-2]")
        @JSONField(name = "fphxz")
        private String discountType;

        /**
         * 优惠政策标识（非必须）
         * 01：简易征收
         * 02：稀土产品
         * 03：免税
         * 04：不征税
         * 05：先征后退
         * 06：100%先征后退
         * 07：50%先征后退
         * 08：按3%简易征收
         * 09：按5%简易征收
         * 10：按5%简易征收减按1.5%计征
         * 11：即征即退30%
         * 12：即征即退50%
         * 13：即征即退70%
         * 14：即征即退100%
         * 15：超税负3%即征即退
         * 16：超税负8%即征即退
         * 17：超税负12%即征即退
         * 18：超税负6%即征即退
         */
        @Alias("优惠政策标识")
        @Pattern(regexp = "^$|0[1-9]|1[0-8]")
        @JSONField(name = "yhzcbs")
        private String taxIncentivesType;

        /**
         * 数据校验
         */
        ValidateResult validateData() {
            var result = ValidateResult.success();
            result = BasicValidator.mergeValidateResult(result, isValidAmount());
            result = BasicValidator.mergeValidateResult(result, isValidTaxAmount());

            return result;
        }

        // 解答关于红冲原因选择销售折让部分红冲场景的咨询问题，已反馈企业冲红原因选择“销售折让”时， 其中单价和数量保持和原蓝字发票一致，不能修改；金额填写需要冲销的金额；此时不校验单价*数量= 金额  。

        /**
         * 单行金额校验：|单价*数量-金额|<=0.01。
         */
        private ValidateResult isValidAmount() {
            if (StringUtils.isBlank(this.getUnitPrice()) || StringUtils.isBlank(this.getQuantity())) {
                return ValidateResult.success();
            }
            final var unitPrice = new BigDecimal(this.getUnitPrice());
            final var quantity = new BigDecimal(this.getQuantity());
            if (unitPrice.multiply(quantity).subtract(this.getAmountWithoutTax()).abs().setScale(2, RoundingMode.HALF_UP)
                    .compareTo(BigDecimal.valueOf(0.01)) > 0) {
                return ValidateResult.fail("单行金额超过最大误差0.01");
            }

            return ValidateResult.success();
        }

        /**
         * 单行税额校验：| 金额 * 税率 - 税额 |<=0.06。
         */
        private ValidateResult isValidTaxAmount() {
            if (this.getAmountWithoutTax().multiply(this.getTaxRate()).subtract(this.getTaxAmount()).abs()
                    .compareTo(BigDecimal.valueOf(0.06)) > 0) {
                return ValidateResult.fail("单行税额超过最大误差0.06");
            }

            return ValidateResult.success();
        }
    }

    @Data
    public static class Additional {

        /**
         * 附加要素名称（非必须）
         */
        @Alias("附加要素名称")
        @Length(max = 200)
        @JSONField(name = "fjysmc")
        private String additionalElementName;

        /**
         * 附加要素类型（非必须）
         */
        @Alias("附加要素类型")
        @Length(max = 200)
        @JSONField(name = "fjyslx")
        private String additionalElementType;

        /**
         * 附加要素值（非必须）
         */
        @Alias("附加要素值")
        @Length(max = 200)
        @JSONField(name = "fjysz")
        private String additionalElementValue;
    }
}
