package com.xforceplus.taxware.chestnut.check.model.validator.tax;

import cn.hutool.core.util.StrUtil;
import com.xforceplus.taxware.architecture.g1.domain.exception.TXWR000004Exception;
import com.xforceplus.taxware.chestnut.check.model.base.BaseDetail;
import com.xforceplus.taxware.chestnut.check.model.base.TaxRate;
import com.xforceplus.taxware.chestnut.contract.model.constant.enums.TaxRateStatusEnum;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.ZoneId;
import java.util.Date;
import java.util.List;

/**
 * @Auther lv
 * @Date 2023/2/24
 * 失败情况：
 * 1。税率记录不存在
 * 2。cszt参数状态1：停用
 * 3。不在有效期内：小于yxqq有效期起||大于yxqz有效期止
 * 4。大于tyrq停用日期
 */
public class TaxRateValidator {
    public static void validate(List<BaseDetail> baseDetails,List<TaxRate> taxRateList) {

        baseDetails.stream().forEach(baseDetail -> {

            final var taxRate = taxRateList.stream().filter(x -> baseDetail.getTaxRate().compareTo(x.getTaxRate()) == 0).findAny().orElseThrow(() -> new TXWR000004Exception(String.format("税率[%s]记录不存在，禁止开票", baseDetail.getTaxRate())));
            System.out.println(String.format("validator rate -> %s", taxRate));

            String paraFlag = taxRate.getParaFlag();
            if (!TaxRateStatusEnum.TAX_RATE_STATUS_0.getStatus().equals(paraFlag)) {
                throw new TXWR000004Exception(String.format("税率[%s]当前状态为已停用，禁止开票", taxRate.getTaxRate()));
            }

            SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
            Date now = Date.from(LocalDate.now().atStartOfDay(ZoneId.systemDefault()).toInstant());

            try {
                if (StrUtil.isBlank(taxRate.getStartDate())) {
                    throw new TXWR000004Exception(String.format("税率[%s]启用日期为空，禁止开票", taxRate.getTaxRate()));
                }

                Date startDate = format.parse(taxRate.getStartDate());
                // 启用日期不可为空
                if (null == startDate || startDate.compareTo(now) > 0) {
                    throw new TXWR000004Exception(String.format("税率[%s]启用日期为[%s]不在有效期范围内，禁止开票", taxRate.getTaxRate(), taxRate.getStartDate()));
                }

                if (StrUtil.isNotBlank(taxRate.getStopDate())) {
                    Date stopDate = format.parse(taxRate.getStopDate());
                    // 有效期止可为空
                    if (null != stopDate && stopDate.compareTo(now) < 0) {
                        throw new TXWR000004Exception(String.format("税率[%s]停用日期为[%s]不在有效期范围内，禁止开票", taxRate.getTaxRate(), taxRate.getStopDate()));
                    }
                }

                if (StrUtil.isNotBlank(taxRate.getEndDate())) {
                    Date endDate = format.parse(taxRate.getEndDate());
                    // 停用日期可为空
                    if (null != endDate && endDate.compareTo(now) < 0) {
                        throw new TXWR000004Exception(String.format("税率[%s]结束日期为[%s]不在有效期范围内，禁止开票", taxRate.getTaxRate(), taxRate.getEndDate()));
                    }
                }
            } catch (ParseException e) {
                throw new TXWR000004Exception(String.format("税率[%s]时间有误，禁止开票", taxRate.getTaxRate()));
            }
        });
    }
}
