package com.xforceplus.taxware.chestnut.check.model.validator.invoice;

import com.xforceplus.taxware.chestnut.check.model.common.Alias;
import com.xforceplus.taxware.chestnut.check.model.common.InvoiceStyleTypeGoodsTaxNoProvider;
import com.xforceplus.taxware.chestnut.check.model.common.ValidateResult;
import com.xforceplus.taxware.chestnut.check.model.constant.AreaUnitEnum;
import com.xforceplus.taxware.chestnut.check.model.util.BasicValidator;
import lombok.Data;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.validator.constraints.Length;

import javax.validation.constraints.Digits;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.math.BigDecimal;
import java.util.Arrays;
import java.util.List;

/**
 * @author Thj
 * @createdAt: 2024/1/11 17:43
 * @Description:
 */
@Data
public class RealEstateSalesBaseValidator {
    private static final List<String> realEstatePlaceKeyWord = Arrays.asList("街", "路", "村", "乡", "镇", "道", "巷", "号");

    @Alias("不动产销售明细列表")
    private List<RealEstateSalesDetail> realEstateSalesDetailList;

    /**
     * 不动产销售
     */
    private RealEstateSalesInfo realEstateSalesInfo;

    /**
     * 能力编码列表
     */
    private List<String> capabilityCodeList;

    @Data
    public static class RealEstateSalesInfo {

        /**
         * 房屋产权证书/不动产权证号码
         */
        @Alias("房屋产权证书/不动产权证号码")
        @Length(max = 40)
        private String realEstateNo;

        /**
         * 不动产坐落地址
         */
        @Alias("不动产坐落地址")
        private String realEstatePlace;

        /**
         * 跨地市标志
         */
        private Boolean crossCitySign;

        /**
         * 面积单位
         */
        private AreaUnitEnum areaUnit;

        /**
         * 不动产单元代码/网签合同备案编码
         */
        @Alias("不动产单元代码/网签合同备案编码")
        @Length(max = 28)
        private String realEstateCode;

        /**
         * 土地增值税项目编号
         */
        @Alias("土地增值税项目编号")
        @Length(max = 16)
        private String landVatItemNo;

        /**
         * 核定计税价格
         */
        @Alias("核定计税价格")
        @Digits(integer = 18, fraction = 2)
        private BigDecimal taxablePrice;

        /**
         * 实际成交含税金额
         */
        @Alias("实际成交含税金额")
        @Digits(integer = 18, fraction = 2)
        private BigDecimal transactionPrice;
    }

    @Data
    public static class RealEstateSalesDetail {

        /**
         * 序号
         * 8位，从1开始。
         */
        @Alias("序号")
        @NotNull
        @Digits(integer = 8, fraction = 0)
        private Integer rowNum;

        @Alias("不动产销售税编")
        @NotEmpty
        @Length(max = 19)
        private String goodsTaxNo;
    }

    public ValidateResult validate() {

        // 先做主信息合规校验，不通过直接返回，不进行数据校验
        var result = BasicValidator.validate(this);
        result = BasicValidator.mergeValidateResult(result, BasicValidator.validate(this.getRealEstateSalesInfo()));
        result = BasicValidator.mergeValidateResult(result, BasicValidator.validate(this.getRealEstateSalesDetailList()));

        if (realEstateSalesDetailList == null || realEstateSalesDetailList.isEmpty()) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("不动产销售明细列表明细不能为空"));
            return result;
        }

        if (realEstateSalesDetailList.size() != 1) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("不动产销售发票明细只能有一行"));
            return result;
        }

        if (realEstateSalesInfo == null) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("不动产销售信息不能为空"));
            return result;
        }

        if (StringUtils.isBlank(realEstateSalesInfo.getRealEstatePlace()) || realEstatePlaceKeyWord.stream().noneMatch(x -> realEstateSalesInfo.getRealEstatePlace().contains(x))) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("不动产地址必须包含街、路、村、乡、镇、道、巷、号等任意一个关键词。"));
            return result;
        }

        if (realEstateSalesInfo.getRealEstatePlace().length() > 120) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("不动产地址总长度不能超过120"));
            return result;
        }

        if (realEstateSalesInfo.getCrossCitySign() == null) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("跨地市标志不能为空"));
            return result;
        }

        if (realEstateSalesInfo.getAreaUnit() == null) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("面积单位不能为空"));
            return result;
        }

        if (realEstateSalesInfo.getTaxablePrice() != null && realEstateSalesInfo.getTransactionPrice() == null) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("实际成交含税金额若按核定计税价格征税的，实际成交含税金额必填。"));
            return result;
        }

        // 能力编码校验
        if (CollectionUtils.isEmpty(this.getCapabilityCodeList()) || !this.getCapabilityCodeList().contains("05")) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("当前开票使用的不动产销售能力和拥有的乐企开票能力不匹配，请去香蕉运维页面配置相关能力"));
            return result;
        }

        return result;
    }

    public ValidateResult validateRealEstateSalesData(InvoiceStyleTypeGoodsTaxNoProvider provider) {
        var result = ValidateResult.success();

        final var realEstateSalesDetail = this.getRealEstateSalesDetailList().get(0);

        final var realEstateSalesGoodTaxNoList = provider.provideGoodsTaxNo();
        if (!realEstateSalesGoodTaxNoList.contains(realEstateSalesDetail.getGoodsTaxNo())) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("开具发票时，使用的商品编码应为税收分类编码中不动产销售类商品编码，请检查"));
            return result;
        }

        return result;
    }
}
