package com.xforceplus.taxware.chestnut.check.model.base;

import com.xforceplus.taxware.chestnut.check.model.common.Alias;
import com.xforceplus.taxware.chestnut.check.model.util.NumberUtil;
import lombok.Data;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.validator.constraints.Length;
import org.hibernate.validator.constraints.ScriptAssert;

import javax.validation.constraints.Digits;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import javax.validation.constraints.Pattern;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Objects;

/**
 * taxware-chestnut-contract
 *
 * @Author 👽 Rayder
 * @Date 📅 2023/2/24 16:58
 * @Motto: Tools change the world. 🛠
 * @Copyright: xforceplus 🇨🇳
 * @Description:
 */
@ScriptAssert.List(
        {
                @ScriptAssert(
                        lang = "javascript",
                        script = "com.xforceplus.taxware.chestnut.check.model.base.BaseDetail.isValidUnionInfo(_this)",
                        message = "数量、单价必须同时为空或同时不为空，请检查"
                ),
        }
)
@Data
public class BaseDetail {

    /**
     * 蓝字发票明细序号（非必须）
     * 红票必传
     */
    @Alias("蓝字发票明细序号")
    @Digits(integer = 8, fraction = 0)
    private Integer originalRowNum;

    /**
     * 序号
     * 8位，从1开始。
     */
    @Alias("序号")
    @NotNull
    @Digits(integer = 8, fraction = 0)
    private Integer rowNum;

    /**
     * 商品和服务税收分类合并编码
     */
    @Alias("商品和服务税收分类合并编码")
    @NotEmpty
    @Length(max = 19)
    private String goodsTaxNo;

    /**
     * 项目名称
     */
    @Alias("项目名称")
    @NotEmpty
    @Length(max = 600)
    @Pattern(regexp = "\\*.+?\\*.+")
    private String itemName;

    /**
     * 规格型号（非必须）
     */
    @Alias("规格型号")
    @Length(max = 150)
    private String specifications;

    /**
     * 单位（非必须）
     * 如“数量”“单价”栏次非空，则本栏次必须非空
     */
    @Alias("单位")
    @Length(max = 300)
    private String unit;

    /**
     * 单价（非必须）
     * 如“数量”“单位”栏次非空，则本栏次必须非空
     * 文档上还有一个说法，选填（如“单位”非空，则必填）
     */
    @Alias("单价")
    @Length(max = 25)
    private String unitPrice;

    /**
     * 数量（非必须）
     * 如“单价”“单价”栏次非空，则本栏次必须非空
     * 文档上还有一个说法，选填（如“单位”非空，则必填）
     */
    @Alias("数量")
    @Length(max = 25)
    private String quantity;

    /**
     * 金额
     */
    @Alias("金额")
    @NotNull
    @Digits(integer = 18, fraction = 2)
    private BigDecimal amountWithoutTax;

    /**
     * 税率
     * 必须在可用税率信息中存在。
     */
    @Alias("税率")
    @NotNull
    @Digits(integer = 16, fraction = 6)
    private BigDecimal taxRate;

    /**
     * 税额
     */
    @Alias("税额")
    @NotNull
    @Digits(integer = 18, fraction = 2)
    private BigDecimal taxAmount;

    /**
     * 税收优惠政策类型代码
     */
    private String taxIncentivesType;

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        BaseDetail that = (BaseDetail) o;
        if (!Objects.equals(originalRowNum, that.originalRowNum)) {
            return false;
        }
        if (!Objects.equals(rowNum, that.rowNum)) {
            return false;
        }
        if (!Objects.equals(goodsTaxNo, that.goodsTaxNo)) {
            return false;
        }
        if (!Objects.equals(itemName, that.itemName)) {
            return false;
        }
        if (!Objects.equals(unit, that.unit)) {
            return false;
        }

        // 强化校验 先判断单价数量是否为数值 如果是 按数值判断 非数值走强制判断
        if (NumberUtil.isNumber(unitPrice) && NumberUtil.isNumber(that.unitPrice)) {
            if (new BigDecimal(unitPrice).compareTo(new BigDecimal(that.unitPrice)) != 0) {
                return false;
            }
        } else if (!Objects.equals(unitPrice, that.unitPrice)) {
            return false;
        }
        if (NumberUtil.isNumber(quantity) && NumberUtil.isNumber(that.quantity)) {
            if (new BigDecimal(quantity).compareTo(new BigDecimal(that.quantity)) != 0) {
                return false;
            }
        } else if (!Objects.equals(quantity, that.quantity)) {
            return false;
        }

        if (!Objects.equals(specifications, that.specifications)) {
            return false;
        }
        if (amountWithoutTax.compareTo(that.amountWithoutTax) != 0) {
            return false;
        }
        if (taxRate.compareTo(that.taxRate) != 0) {
            return false;
        }

        return taxAmount.compareTo(that.taxAmount) == 0;
    }

    @Override
    public int hashCode() {
        return Objects.hash(originalRowNum, rowNum, goodsTaxNo, itemName, specifications, unit, unitPrice, quantity, amountWithoutTax, taxRate, taxAmount);
    }

    /**
     * 数量、单价必须同时为空或同时不为空
     */
    public static boolean isValidUnionInfo(BaseDetail detail) {
        if (detail.getUnitPrice() != null && detail.getUnitPrice().trim().length() > 0
                || detail.getQuantity() != null && detail.getQuantity().trim().length() > 0) {
            return detail.getUnitPrice() != null && detail.getUnitPrice().trim().length() > 0
                    && detail.getQuantity() != null && detail.getQuantity().trim().length() > 0;
        }

        return true;
    }

    public String getItemShortName() {
        if (StringUtils.isBlank(this.itemName)) {
            return null;
        }

        // 根据标准的商品名称（*AA*BBB） 拆分出商品名称和商品税目名称
        final var split = Arrays.asList(StringUtils.split(this.itemName, "*"));
        final var spmcsm = new ArrayList<>(split);

        if (spmcsm.size() < 2 && !this.itemName.startsWith("*")) {
            return null;
        }

        return spmcsm.get(0);
    }
}
