package com.xforceplus.taxware.chestnut.check.model.validator.invoice;

import com.xforceplus.taxware.chestnut.check.model.common.Alias;
import com.xforceplus.taxware.chestnut.check.model.common.InvoiceStyleTypeGoodsTaxNoProvider;
import com.xforceplus.taxware.chestnut.check.model.common.ValidateResult;
import com.xforceplus.taxware.chestnut.check.model.util.BasicValidator;
import lombok.Data;
import org.apache.commons.collections4.CollectionUtils;
import org.hibernate.validator.constraints.Length;

import javax.validation.constraints.Digits;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.List;
import java.util.Objects;

/**
 * @author Thj
 * @createdAt: 2024/2/2 18:09
 * @Description:
 */
@Data
public class FarmProducePurchaseBaseValidator {

    private FarmProducePurchaseInfo farmProducePurchaseInfo;
    private List<FarmProducePurchaseDetail> farmProducePurchaseDetailList;

    /**
     * 能力编码列表
     */
    private List<String> capabilityCodeList;

    @Data
    public static class FarmProducePurchaseInfo {

        private String sellerTaxCode;

        private String purchaserTaxCode;

        private String invoiceType;
    }

    @Data
    public static class FarmProducePurchaseDetail {

        /**
         * 序号
         * 8位，从1开始。
         */
        @Alias("序号")
        @NotNull
        @Digits(integer = 8, fraction = 0)
        private Integer rowNum;

        @Alias("农产品收购税编")
        @NotEmpty
        @Length(max = 19)
        private String goodsTaxNo;
    }

    public ValidateResult validate() {
        // 先做主信息合规校验，不通过直接返回，不进行数据校验
        var result = ValidateResult.success();

        if (farmProducePurchaseDetailList == null || farmProducePurchaseDetailList.isEmpty()) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("农产品收购明细列表明细不能为空"));
            return result;
        }

        if (farmProducePurchaseInfo == null) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("农产品收购信息不能为空"));
            return result;
        }

        if (!Objects.equals(farmProducePurchaseInfo.getInvoiceType(), "02")) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("农产品收购发票，只允许开具全电电子普通发票"));
            return result;
        }

        // 能力编码校验
        if (CollectionUtils.isEmpty(this.getCapabilityCodeList()) || !this.getCapabilityCodeList().contains("16")) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("当前开票使用的农产品收购能力和拥有的乐企开票能力不匹配，请去香蕉运维页面配置相关能力"));
            return result;
        }

        return result;
    }

    public ValidateResult validateFarmProducePurchaseData(InvoiceStyleTypeGoodsTaxNoProvider provider) {
        var result = ValidateResult.success();

        final var farmProducePurchaseGoodsTaxNoList = provider.provideGoodsTaxNo();

        for (FarmProducePurchaseDetail farmProducePurchaseDetail : this.farmProducePurchaseDetailList) {
            if (!farmProducePurchaseGoodsTaxNoList.contains(farmProducePurchaseDetail.getGoodsTaxNo())) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d行明细，开具发票时，使用的商品编码应为税收分类编码中农产品收购类商品编码，请检查", farmProducePurchaseDetail.getRowNum())));
                return result;
            }

            if (!farmProducePurchaseDetail.getGoodsTaxNo().startsWith("101")) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d行明细，开具发票时，使用的商品编码应为“农产品”类商品和服务税收分类编码,“农产品”类商品和服务税收分类编码是以“101”开头的商品编码。", farmProducePurchaseDetail.getRowNum())));
                return result;
            }
        }

        return result;
    }
}
