package com.xforceplus.taxware.chestnut.check.model.validator.invoice;

import com.xforceplus.taxware.architecture.g1.core.utils.ExceptionUtil;
import com.xforceplus.taxware.architecture.g1.domain.exception.TXWR000001Exception;
import com.xforceplus.taxware.chestnut.check.model.base.OilTaxCode;
import com.xforceplus.taxware.chestnut.check.model.common.Alias;
import com.xforceplus.taxware.chestnut.check.model.common.CPYGoodsTaxNoProvider;
import com.xforceplus.taxware.chestnut.check.model.common.ValidateResult;
import com.xforceplus.taxware.chestnut.check.model.util.BasicValidator;
import lombok.Data;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.validator.constraints.Length;

import javax.validation.constraints.Digits;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import javax.validation.constraints.Pattern;
import java.math.BigDecimal;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author Thj
 * @createdAt: 2023/10/23 17:07
 * @Description: 成品油相关校验
 */
@Data
public class CPYBaseValidator {

    @Alias("成品油明细列表")
    private List<CPYDetail> cpyDetailList;

    private List<String> capabilityCodeList;

    /**
     * 本次开具发票数
     */
    private Integer invoiceNumber;

    @Data
    public static class CPYDetail {

        /**
         * 序号
         * 8位，从1开始。
         */
        @Alias("序号")
        @NotNull
        @Digits(integer = 8, fraction = 0)
        private Integer rowNum;

        @Alias("成品油税编")
        @NotEmpty
        @Length(max = 19)
        private String goodsTaxNo;

        @Alias("成品油数量")
        @NotNull
        @Digits(integer = 18, fraction = 2)
        private BigDecimal quantity;

        @Alias("成品油单价")
        @NotNull
        @Digits(integer = 18, fraction = 2)
        private BigDecimal unitPrice;

        @Alias("成品油单位")
        @NotEmpty
        @Pattern(regexp = "升|吨")
        private String unit;
    }

    public ValidateResult validate() {

        // 先做主信息合规校验，不通过直接返回，不进行数据校验
        var result = BasicValidator.validate(this);

        if (cpyDetailList == null || cpyDetailList.size() == 0) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("成品油明细列表明细不能为空"));
            return result;
        }

        if (cpyDetailList.size() > 2000) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("成品油明细列表明细不能大于2000行"));
            return result;
        }

        // 乐企成品油开票 在预占之前增加能力编码校验
        if (CollectionUtils.isEmpty(this.getCapabilityCodeList()) || !this.getCapabilityCodeList().contains("01")) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("当前开票使用的成品油能力和拥有的乐企开票能力不匹配，请去香蕉运维页面配置相关能力"));
            return result;
        }

        // 先做明细信息合规校验，不通过直接返回，不再进行数据校验
        for (var detail : cpyDetailList) {
            final var validateResult = BasicValidator.validate(detail, detail.getRowNum() + "");
            result = BasicValidator.mergeValidateResult(result, validateResult);
        }

        return result;
    }

    /**
     * 成品油纳税人开具商品和服务税收分类编码为：
     * 1070101020100000000航空煤油成品油发票、
     * 1070101040200000000燃料油定点直供、
     * 1070101050200000000石脑油定点直供发票
     * 均须单独开具，不得相互混开或与其他发票混开。
     * 开票商品为燃料油时，其“单位”栏必须选择“吨”且不能为空。
     * 成品油企业只能使用已授权的成品油编码开具成品油发票。
     *
     * @param provider
     * @return
     */
    public ValidateResult validateCpyData(CPYGoodsTaxNoProvider provider) {
        var result = ValidateResult.success();

        final var cpyDetailList = this.getCpyDetailList();

        final var goodsTaxNoList = cpyDetailList.stream().map(CPYDetail::getGoodsTaxNo).distinct().collect(Collectors.toList());
        if (goodsTaxNoList.contains("1070101020100000000")
                || goodsTaxNoList.contains("1070101040200000000")
                || goodsTaxNoList.contains("1070101050200000000")
        ) {
            if (cpyDetailList.size() > 1) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("成品油纳税人开具税编明细含有不得相互混开的税编，请检查。"));
                return result;
            }

            if (invoiceNumber > 1) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("本次批量开具的成品油发票明细中含有不得与其他发票混开的税编，请检查。"));
                return result;
            }
        }

        if (provider == null || provider.provide() == null) {
            return result;
        }

        final var oilTaxCodeList = provider.provide();
        final var oilGoodsTaxNoList = oilTaxCodeList.stream().map(OilTaxCode::getGoodsTaxNo).distinct().collect(Collectors.toList());
        try {
            cpyDetailList.forEach(cpyDetail -> {
                if (!oilGoodsTaxNoList.contains(cpyDetail.getGoodsTaxNo())) {
                    throw new TXWR000001Exception(String.format("第%d行明细,含有非已授权的成品油编码，请检查", cpyDetail.getRowNum()));
                }

                final var oilTaxNo = oilTaxCodeList.stream()
                        .filter(oilGoodsTaxNo -> StringUtils.equals(oilGoodsTaxNo.getGoodsTaxNo(), cpyDetail.getGoodsTaxNo()))
                        .findAny().orElseThrow(() -> new TXWR000001Exception("含有非已授权的成品油编码，请检查"));

                if (StringUtils.equals(oilTaxNo.getLevyItemName(), "燃料油") && !StringUtils.equals(cpyDetail.getUnit(), "吨")) {
                    throw new TXWR000001Exception(String.format("第%d行明细,开票商品为燃料油时，其“单位”栏必须选择“吨”且不能为空。", cpyDetail.getRowNum()));
                }
            });
        } catch (Throwable e) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(ExceptionUtil.toDesc(e)));
        }

        return result;
    }
}
