package com.xforceplus.taxware.chestnut.check.model.validator.taxpayer;

import com.xforceplus.taxware.architecture.g1.domain.exception.TXWR000004Exception;
import com.xforceplus.taxware.chestnut.check.model.base.TaxpayerInfo;
import com.xforceplus.taxware.chestnut.check.model.common.RealEstateLeaseTaxPayerInfoProvider;
import com.xforceplus.taxware.chestnut.contract.model.constant.enums.SellerTaxpayerStatusEnum;
import org.apache.commons.collections4.CollectionUtils;

import javax.annotation.Nullable;
import java.util.Objects;

/**
 * @Auther lv
 * @Date 2023/2/13
 * @Descrption 纳税人基础信息校验：
 * 	未查询到纳税人基本信息，本地无纳税人基本信息。
 * 	纳税人信息中“纳税人状态”不是“03（正常）”、“09（报验）”中的任意一个。
 */
public class TaxpayerInfoValidator {

    /**
     * 	未查询到纳税人基本信息，本地无纳税人基本信息。
     * 	纳税人信息中“纳税人状态”不是“03（正常）”、“09（报验）”中的任意一个。
     *
     * @param taxpayerInfo
     */
    public static void validate(TaxpayerInfo taxpayerInfo, @Nullable String invoiceStyleTypeCode) {

        if (taxpayerInfo == null) {
            throw new TXWR000004Exception("未查询到纳税人基本信息，禁止开票");
        }

        /**
         * 开票方纳税人状态
         * 01：受理
         * 02：筹建期
         * 03：正常
         * 04：停业
         * 05：非正常
         * 06：清算
         * 07：注销
         * 08：非正常户注销
         * 09：报验
         * 10：核销报验
         * 11：中断缴费
         * 12：恢复缴费
         * 13：简易注销无异议
         * 99：其他
         */
        String sellerTaxpayerStatus = taxpayerInfo.getSellerTaxpayerStatus();
        if (!SellerTaxpayerStatusEnum.SELLER_TAXPAYER_STATUS_03.getStatus().equals(sellerTaxpayerStatus) && !SellerTaxpayerStatusEnum.SELLER_TAXPAYER_STATUS_09.getStatus().equals(sellerTaxpayerStatus)) {
            throw new TXWR000004Exception(String.format("该纳税人状态为[%s]，期望状态为[%s]，禁止开票", SellerTaxpayerStatusEnum.fromStatus(sellerTaxpayerStatus).getStatusDesc(), "正常或报验"));
        }

        // 乐企成品油开票 增加纳税人基本信息成品油资质校验（生产企业、经销企业）
        if (Objects.equals("01", invoiceStyleTypeCode)
                && (CollectionUtils.isEmpty(taxpayerInfo.getIndustryNaturePropertiesList()) ||
                (!taxpayerInfo.getIndustryNaturePropertiesList().contains("01") && !taxpayerInfo.getIndustryNaturePropertiesList().contains("02")))) {
            throw new TXWR000004Exception("非成品油生产企业、成品油经销企业，禁止开具成品油发票");
        }

        // 若开票方非农产品收购企业，禁止开具农产品收购发票。
        if (Objects.equals("16", invoiceStyleTypeCode)
                && (CollectionUtils.isEmpty(taxpayerInfo.getIndustryNaturePropertiesList()) ||
                !taxpayerInfo.getIndustryNaturePropertiesList().contains("36"))) {
            throw new TXWR000004Exception("若开票方非农产品收购企业，禁止开具农产品收购发票");
        }
    }

    /**
     * 当不动产租赁发票明细中的税收分类编码为“3040502020102000000个人出租住房”时，纳税人必须为个体工商户或自然人。
     * 个体工商户指：【登记注册类型】为个体工商户（“410个体工商户”、“411内资个体”、“412港澳台个体”、“413外资个体”）。
     * 自然人指：【登记注册类型】（“430个人”、“431内资个人”、“432港澳台个人”、“433外资个人”）。
     * 税编校验条件前置
     */
    public static void validateRealEstateLease(TaxpayerInfo taxpayerInfo, RealEstateLeaseTaxPayerInfoProvider provider) {
        final var provideTaxPayerList = provider.provideTaxPayerList();
        if (provideTaxPayerList.stream().noneMatch(x -> x.equals(taxpayerInfo.getRegisterType()))) {
            throw new TXWR000004Exception("当不动产租赁发票明细中的税收分类编码为“3040502020102000000个人出租住房”时，纳税人必须为个体工商户或自然人。\n" +
                    "个体工商户指：【登记注册类型】为个体工商户（“410个体工商户”、“411内资个体”、“412港澳台个体”、“413外资个体”）。\n" +
                    "自然人指：【登记注册类型】（“430个人”、“431内资个人”、“432港澳台个人”、“433外资个人”）。");
        }
    }

    /**
     * 开具发票时，纳税人基础信息中的企业性质不为“31（房地产开发企业）”的，禁止使用前七位为5010101/5010201的编码。
     * 税编校验条件前置
     */
    public static void validateRealEstateSales(TaxpayerInfo taxpayerInfo) {

        if (CollectionUtils.isEmpty(taxpayerInfo.getIndustryNaturePropertiesList()) ||
                !taxpayerInfo.getIndustryNaturePropertiesList().contains("31")) {
            throw new TXWR000004Exception("开具不动产销售发票时，纳税人基础信息中的企业性质不为“31（房地产开发企业）”的，禁止使用前七位为5010101/5010201的编码。");
        }
    }
}
