package com.xforceplus.taxware.chestnut.check.model.validator.invoice;

import com.alibaba.fastjson.annotation.JSONField;
import com.xforceplus.taxware.chestnut.check.model.common.Alias;
import com.xforceplus.taxware.chestnut.check.model.common.InvoiceStyleTypeGoodsTaxNoProvider;
import com.xforceplus.taxware.chestnut.check.model.common.ValidateResult;
import com.xforceplus.taxware.chestnut.check.model.constant.AreaUnitEnum;
import com.xforceplus.taxware.chestnut.check.model.constant.TaxIncentivesTypeEnum;
import com.xforceplus.taxware.chestnut.check.model.util.BasicValidator;
import lombok.Data;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.validator.constraints.Length;

import javax.validation.constraints.Digits;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Objects;

/**
 * @author Thj
 * @createdAt: 2023/12/18 11:17
 * @Description:
 */
@Data
public class RealEstateLeaseBaseValidator {
    private static final List<String> realEstatePlaceKeyWord = Arrays.asList("街", "路", "村", "乡", "镇", "道", "巷", "号");

    @Alias("不动产租赁明细列表")
    private List<RealEstateLeaseDetail> realEstateLeaseDetails;

    /**
     * 不动产租赁
     */
    private RealEstateLeaseInfo realEstateLeaseInfo;

    /**
     * 能力编码列表
     */
    private List<String> capabilityCodeList;

    @Data
    public static class RealEstateLeaseInfo {

        /**
         * 房屋产权证书/不动产权证号码
         */
        @Alias("产权证书/不动产权证号码")
        @Length(max = 40)
        private String realEstateNo;

        /**
         * 不动产坐落地址
         */
        @Alias("不动产坐落省份")
        private String realEstateProvince;

        @Alias("不动产坐落市")
        private String realEstateCity;

        @Alias("不动产坐落详细地址")
        private String realEstatePlace;

        /**
         * 租赁期起
         */
        private Date leaseTermStart;

        /**
         * 租赁期止
         */
        private Date leaseTermEnd;

        /**
         * 跨地市标志
         */
        private Boolean crossCitySign;

        /**
         * 面积单位
         */
        private AreaUnitEnum areaUnit;
    }

    @Data
    public static class RealEstateLeaseDetail {

        /**
         * 序号
         * 8位，从1开始。
         */
        @Alias("序号")
        @NotNull
        @Digits(integer = 8, fraction = 0)
        private Integer rowNum;

        @Alias("不动产租赁税编")
        @NotEmpty
        @Length(max = 19)
        private String goodsTaxNo;

        /**
         * 税收优惠政策类型代码
         */
        private TaxIncentivesTypeEnum taxIncentivesType;

        /**
         * 金额
         */
        @Alias("金额")
        @NotNull
        @Digits(integer = 18, fraction = 2)
        private BigDecimal amountWithoutTax;

        /**
         * 税额
         */
        @Alias("税额")
        @NotNull
        @Digits(integer = 18, fraction = 2)
        private BigDecimal taxAmount;

        /**
         * 含税金额
         */
        @Alias("含税金额")
        @NotNull
        @Digits(integer = 18, fraction = 2)
        @JSONField(name = "hsje")
        private BigDecimal amountWithTax;
    }

    public ValidateResult validate() {

        // 先做主信息合规校验，不通过直接返回，不进行数据校验
        var result = BasicValidator.validate(this);

        if (realEstateLeaseDetails == null || realEstateLeaseDetails.isEmpty()) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("不动产租赁明细列表明细不能为空"));
            return result;
        }

        if (realEstateLeaseDetails.size() != 1) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("不动产租赁发票明细只能有一行"));
            return result;
        }

        if (realEstateLeaseInfo == null) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("不动产租赁信息不能为空"));
            return result;
        }

        if (realEstateLeaseInfo.getCrossCitySign() == null) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("跨地市标志不能为空"));
            return result;
        }

        if (realEstateLeaseInfo.getAreaUnit() == null) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("面积单位不能为空"));
            return result;
        }

        if (realEstateLeaseInfo.getLeaseTermStart() == null || realEstateLeaseInfo.getLeaseTermEnd() == null) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("起止日期不能为空，且格式应为yyyyMMdd"));
            return result;
        }

        if (StringUtils.isBlank(realEstateLeaseInfo.getRealEstatePlace()) || realEstatePlaceKeyWord.stream().noneMatch(x -> realEstateLeaseInfo.getRealEstatePlace().contains(x))) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("不动产地址必须包含街、路、村、乡、镇、道、巷、号等任意一个关键词。"));
            return result;
        }

        if (realEstateLeaseInfo.getRealEstatePlace().length() > 80) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("不动产租赁发票开具：不动产坐落地址（详细地址）总长度不能超过80"));
            return result;
        }

        // 能力编码校验
        if (CollectionUtils.isEmpty(this.getCapabilityCodeList()) || !this.getCapabilityCodeList().contains("06")) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("当前开票使用的不动产租赁能力和拥有的乐企开票能力不匹配，请去香蕉运维页面配置相关能力"));
            return result;
        }

        // 先做明细信息合规校验，不通过直接返回，不再进行数据校验
        for (var detail : realEstateLeaseDetails) {
            final var validateResult = BasicValidator.validate(detail, detail.getRowNum() + "");
            result = BasicValidator.mergeValidateResult(result, validateResult);
        }

        return result;
    }

    public ValidateResult validateRealEstateLeaseData(InvoiceStyleTypeGoodsTaxNoProvider provider) {
        var result = ValidateResult.success();

        final var realEstateLeaseDetail = this.getRealEstateLeaseDetails().get(0);

        final var realEstateLeaseGoodTaxNoList = provider.provideGoodsTaxNo();
        if (!realEstateLeaseGoodTaxNoList.contains(realEstateLeaseDetail.getGoodsTaxNo())) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("开具发票时，使用的商品编码应为税收分类编码中不动产租赁类商品编码，请检查"));
            return result;
        }

        if (Objects.equals(realEstateLeaseDetail.getGoodsTaxNo(), "3040502020102000000")) {
            if (realEstateLeaseDetail.getTaxIncentivesType() != TaxIncentivesTypeEnum.FIVE_PERCENT_SIMPLE_EXPROPRIATION_REDUCE_ONE_POINT_FIVE_PERCENT) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("个体工商户或自然人在使用“3040502020102000000个人出租住房”开具不动产租赁发票时，只能按照5%减按1.5%计算应纳税额"));
                return result;
            }

            final var amountWithTax = realEstateLeaseDetail.getAmountWithTax();
            BigDecimal onePlusFivePercent = new BigDecimal("1.05");
            BigDecimal onePointFivePercent = new BigDecimal("0.015");

            final var calculateTaxAmount = amountWithTax.multiply(onePointFivePercent).divide(onePlusFivePercent, BigDecimal.ROUND_HALF_UP);
            final var calculateAmountWithoutTax = amountWithTax.subtract(calculateTaxAmount);

            final var judgeNum = new BigDecimal("0.01");
            if (realEstateLeaseDetail.getTaxAmount().subtract(calculateTaxAmount).abs().compareTo(judgeNum) > 0) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("应纳税额计算方式为：税额=含税销售额/（1+5%）*1.5%，请检查税额是否正确"));
                return result;
            }
            if (realEstateLeaseDetail.getAmountWithoutTax().subtract(calculateAmountWithoutTax).abs().compareTo(judgeNum) > 0) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("应纳税额计算方式为：税额=含税销售额/（1+5%）*1.5%，不含税金额=含税销售额-税额。请检查不含税金额是否正确"));
                return result;
            }
        }

        return result;
    }
}
