package com.xforceplus.taxware.chestnut.check.model.validator.invoice;

import com.xforceplus.taxware.chestnut.check.model.common.Alias;
import com.xforceplus.taxware.chestnut.check.model.common.InvoiceStyleTypeGoodsTaxNoProvider;
import com.xforceplus.taxware.chestnut.check.model.common.ValidateResult;
import com.xforceplus.taxware.chestnut.check.model.constant.TaxIncentivesTypeEnum;
import com.xforceplus.taxware.chestnut.check.model.util.BasicValidator;
import lombok.Data;
import org.apache.commons.collections4.CollectionUtils;
import org.hibernate.validator.constraints.Length;

import javax.validation.constraints.Digits;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.math.BigDecimal;
import java.util.List;

/**
 * @author Thj
 * @createdAt: 2024/3/11 14:05
 * @Description:
 */
@Data
public class SelfFarmProduceSalesBaseValidator {
    private List<SelfFarmProduceSalesDetail> selfFarmProduceSalesDetailList;

    /**
     * 能力编码列表
     */
    private List<String> capabilityCodeList;

    @Data
    public static class SelfFarmProduceSalesDetail {

        /**
         * 序号
         * 8位，从1开始。
         */
        @Alias("序号")
        @NotNull
        @Digits(integer = 8, fraction = 0)
        private Integer rowNum;

        @Alias("自产农产品销售税编")
        @NotEmpty
        @Length(max = 19)
        private String goodsTaxNo;

        /**
         * 税收优惠政策类型代码
         */
        private TaxIncentivesTypeEnum taxIncentivesType;

        @Alias("明细税率")
        @NotNull
        @Digits(integer = 18, fraction = 2)
        private BigDecimal taxRate;

        @Alias("明细税额")
        @NotNull
        @Digits(integer = 18, fraction = 2)
        private BigDecimal taxAmount;
    }

    public ValidateResult validate() {
        // 先做主信息合规校验，不通过直接返回，不进行数据校验
        var result = ValidateResult.success();

        if (selfFarmProduceSalesDetailList == null || selfFarmProduceSalesDetailList.isEmpty()) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("自产农产品销售明细列表明细不能为空"));
            return result;
        }

        // 能力编码校验
        if (CollectionUtils.isEmpty(this.getCapabilityCodeList()) || !this.getCapabilityCodeList().contains("12")) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("当前开票使用的自产农产品销售能力和拥有的乐企开票能力不匹配，请去香蕉运维页面配置相关能力"));
            return result;
        }

        // 先做明细信息合规校验，不通过直接返回，不再进行数据校验
        for (var detail : selfFarmProduceSalesDetailList) {
            final var validateResult = BasicValidator.validate(detail, detail.getRowNum() + "");
            result = BasicValidator.mergeValidateResult(result, validateResult);
        }

        return result;
    }

    public ValidateResult validateSelfFarmProduceSalesData(InvoiceStyleTypeGoodsTaxNoProvider provider) {
        var result = ValidateResult.success();

        final var selfFarmProduceSalesGoodsTaxNoList = provider.provideGoodsTaxNo();

        for (SelfFarmProduceSalesDetail selfFarmProduceSalesDetail : this.selfFarmProduceSalesDetailList) {
            if (!selfFarmProduceSalesGoodsTaxNoList.contains(selfFarmProduceSalesDetail.getGoodsTaxNo())) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d行明细，开具自产农产品销售发票时，只能使用特定要素标签归类为“11（农产品）” 的税收分类编码，请检查", selfFarmProduceSalesDetail.getRowNum())));
                return result;
            }

            if (!selfFarmProduceSalesDetail.getGoodsTaxNo().startsWith("101")) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d行明细，开具发票时，使用的商品编码应为“农产品”类商品和服务税收分类编码,“农产品”类商品和服务税收分类编码是以“101”开头的商品编码。", selfFarmProduceSalesDetail.getRowNum())));
                return result;
            }

            if (selfFarmProduceSalesDetail.getTaxIncentivesType() != TaxIncentivesTypeEnum.DUTY_FREE) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d行明细，自产农产品销售发票：必须使用“免税”优惠政策", selfFarmProduceSalesDetail.getRowNum())));
                return result;
            }

            if (selfFarmProduceSalesDetail.getTaxRate().compareTo(BigDecimal.ZERO) != 0) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d行明细，自产农产品销售发票：增值税税率/征收率固定为0", selfFarmProduceSalesDetail.getRowNum())));
                return result;
            }

            if (selfFarmProduceSalesDetail.getTaxAmount().compareTo(BigDecimal.ZERO) != 0) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d行明细，自产农产品销售发票：增值税税额固定为0", selfFarmProduceSalesDetail.getRowNum())));
                return result;
            }
        }

        return result;
    }
}
