package com.xforceplus.taxware.chestnut.check.model.validator.invoice;

import com.xforceplus.taxware.chestnut.check.model.common.Alias;
import com.xforceplus.taxware.chestnut.check.model.common.InvoiceStyleTypeGoodsTaxNoProvider;
import com.xforceplus.taxware.chestnut.check.model.common.ValidateResult;
import com.xforceplus.taxware.chestnut.check.model.util.BasicValidator;
import com.xforceplus.taxware.chestnut.contract.model.constant.enums.CapabilityCodeEnum;
import lombok.Data;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.validator.constraints.Length;

import javax.validation.constraints.Digits;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.List;

/**
 * @author Thj
 * @createdAt: 2024/2/20 16:57
 * @Description:
 */
@Data
public class BuildingServiceValidator {

    @Alias("建筑服务明细列表")
    private List<BuildingServiceDetail> buildingServiceDetailList;

    /**
     * 能力编码列表
     */
    private List<String> capabilityCodeList;

    private BuildingServiceInfo buildingServiceInfo;

    @Data
    public static class BuildingServiceInfo {
        /**
         * 建筑服务发生地
         */
        private String place;

        /**
         * 建筑项目名称
         */
        private String entryName;

        /**
         * 跨地市标志
         */
        private Boolean crossCitySign;

        /**
         * 跨区域涉税事项报验管理编号
         */
        private String crossRegionalTaxManageNo;
    }

    @Data
    public static class BuildingServiceDetail {

        /**
         * 序号
         * 8位，从1开始。
         */
        @Alias("序号")
        @NotNull
        @Digits(integer = 8, fraction = 0)
        private Integer rowNum;

        @Alias("建筑服务税编")
        @NotEmpty
        @Length(max = 19)
        private String goodsTaxNo;
    }

    public ValidateResult validate() {
        // 先做主信息合规校验，不通过直接返回，不进行数据校验
        var result = ValidateResult.success();
        if (buildingServiceDetailList == null || buildingServiceDetailList.isEmpty()) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("建筑服务明细列表明细不能为空"));
            return result;
        }

        if (buildingServiceDetailList.size() != 1) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("建筑服务发票明细只能有一行，传入行数【%s】", buildingServiceDetailList.size())));
            return result;
        }

        if (buildingServiceInfo == null) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("建筑服务信息不能为空"));
            return result;
        }

        if (StringUtils.isBlank(buildingServiceInfo.getPlace())) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("建筑服务发生地不能为空"));
            return result;
        }

        if (buildingServiceInfo.getPlace().length() > 120) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("建筑服务发生地总长度不能超过120，传入值为【%s】，长度为【%s】",
                    buildingServiceInfo.getPlace(), buildingServiceInfo.getPlace().length())));
            return result;
        }

        if (StringUtils.isBlank(buildingServiceInfo.getEntryName())) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("建筑服务建筑项目名称不能为空"));
            return result;
        }

        if (buildingServiceInfo.getEntryName().length() > 80) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("建筑服务建筑项目名称总长度不能超过80，传入值为【%s】，长度为【%s】",
                    buildingServiceInfo.getEntryName(), buildingServiceInfo.getEntryName().length())));
            return result;
        }

        if (buildingServiceInfo.getCrossCitySign() == null) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("跨地（市）标志不能为空"));
            return result;
        }

        if (buildingServiceInfo.getCrossCitySign() && StringUtils.isEmpty(buildingServiceInfo.getCrossRegionalTaxManageNo())) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("跨地（市）标志为“是”时，跨区域涉税事项报验管理编号必填"));
            return result;
        }

        if (!buildingServiceInfo.getCrossCitySign() && StringUtils.isNotEmpty(buildingServiceInfo.getCrossRegionalTaxManageNo())) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("跨地（市）标志为“否”时，跨区域涉税事项报验管理编号必须为空，传入值为【%s】", buildingServiceInfo.getCrossRegionalTaxManageNo())));
            return result;
        }

        if (buildingServiceInfo.getCrossRegionalTaxManageNo() != null && buildingServiceInfo.getCrossRegionalTaxManageNo().length() > 50) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("跨区域涉税事项报验管理编号长度不能超过50，传入值为【%s】，长度为【%s】",
                    buildingServiceInfo.getCrossRegionalTaxManageNo(), buildingServiceInfo.getCrossRegionalTaxManageNo().length())));
            return result;
        }

        // 能力编码校验
        if (CollectionUtils.isEmpty(this.getCapabilityCodeList()) || !this.getCapabilityCodeList().contains("03")) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("请前往电子税局“乐企数字开放平台”检查所使用的[%s]是否已被授权邀请", CapabilityCodeEnum.fromCode("03").toDesc())));
            return result;
        }

        return result;
    }

    public ValidateResult validateBuildingServiceData(InvoiceStyleTypeGoodsTaxNoProvider provider) {
        var result = ValidateResult.success();

        final var buildingServiceDetail = this.getBuildingServiceDetailList().get(0);

        final var buildingServiceGoodTaxNoList = provider.provideGoodsTaxNo();
        if (!buildingServiceGoodTaxNoList.contains(buildingServiceDetail.getGoodsTaxNo())) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("开具发票时，使用的商品编码应为税收分类编码中建筑服务类商品编码，请检查。传入值为【%s】", buildingServiceDetail.getGoodsTaxNo())));
            return result;
        }

        return result;
    }
}
