package com.xforceplus.taxware.chestnut.check.model.validator.invoice;

import com.xforceplus.taxware.chestnut.check.model.common.Alias;
import com.xforceplus.taxware.chestnut.check.model.common.InvoiceStyleTypeGoodsTaxNoProvider;
import com.xforceplus.taxware.chestnut.check.model.common.ValidateResult;
import com.xforceplus.taxware.chestnut.check.model.util.BasicValidator;
import com.xforceplus.taxware.chestnut.check.model.util.CommonUtil;
import com.xforceplus.taxware.chestnut.contract.model.constant.enums.CapabilityCodeEnum;
import lombok.Data;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.validator.constraints.Length;

import javax.validation.constraints.Digits;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.List;

/**
 * @author lv
 * @createdAt: 2025/08/25 15:05
 * @Description:
 */
@Data
public class JewelryBaseValidator {
    private List<String> goodsTaxNoList = List.of("1060509010000000000","1060509020100000000","1060509020200000000");
    /**
     * 销方税号
     */
    private String sellerTaxNo;
    /**
     * 销方是否试点企业，默认false
     */
    private boolean sellerJewelryPilotFlag = false;

    /**
     * 销方是否金银首饰资质企业，默认false
     */
    private boolean sellerJewelryQualificationFlag = false;
    /**
     * 购方税号
     */
    private String buyerTaxNo;
    /**
     * 购方名称
     */
    private String buyerName;

    /**
     * 购方是否自然人，默认false
     */
    private boolean buyerNatureFlag = false;
    /**
     * 购方是否试点企业，默认false
     */
    private boolean buyerJewelryPilotFlag = false;

    /**
     * 销方是否金银首饰资质企业，默认false
     */
    private boolean buyerJewelryQualificationFlag = false;

    /**
     * 是否红票
     */
    private boolean isRed;

    /**
     * 明细
     */
    private List<Detail> detailList;

    /**
     * 能力编码列表
     */
    private List<String> capabilityCodeList;

    /**
     * 特殊业务类型
     */
    private String invoiceStyleType;

    @Data
    public static class Detail {

        /**
         * 序号
         * 8位，从1开始。
         */
        @Alias("序号")
        @NotNull
        @Digits(integer = 8, fraction = 0)
        private Integer rowNum;

        @Alias("金银首饰税编")
        @NotEmpty
        @Length(max = 19)
        private String goodsTaxNo;
    }

    public ValidateResult validate() {
        // 主信息合规校验，不通过直接返回，不进行数据校验
        var result = ValidateResult.success();

        // 校验是否应走特殊业务开具
        if (detailList == null || detailList.isEmpty()) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("金银首饰明细列表明细不能为空"));
            return result;
        }

        // 能力编码校验
        if (CollectionUtils.isEmpty(this.getCapabilityCodeList()) || (!this.getCapabilityCodeList().contains("26") && !this.getCapabilityCodeList().contains("27"))) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("请前往电子税局“乐企数字开放平台”检查所使用的[%s]是否已被授权邀请", CapabilityCodeEnum.fromCode("26").toDesc())));
            return result;
        }

        // 明细信息合规校验，不通过直接返回，不再进行数据校验
        for (var detail : detailList) {
            final var validateResult = BasicValidator.validate(detail, detail.getRowNum() + "");
            result = BasicValidator.mergeValidateResult(result, validateResult);
            if (!this.isRed() && !goodsTaxNoList.contains(detail.getGoodsTaxNo())) {
                result = BasicValidator.mergeValidateResult(result,
                        ValidateResult.fail(String.format("第%d行明细，开具金银首饰发票时，只能使用税收分类编码【1060509010000000000,1060509020100000000,1060509020200000000】，请检查。传入值为【%s】",
                                detail.getRowNum(), detail.getGoodsTaxNo())));
                return result;
            }
        }

        return result;
    }

    /**
     * 校验资质
     */
    public ValidateResult validateJewelryCapabilityCode() {
        final var result = ValidateResult.success();
        /*
         * 校验是否应该走特殊业务
         *（1）若销售方和购买方均为金银首饰经营企业，在企业选择开具发票时弹窗提示：“购销双方均为金银首饰纳税人，请确认该笔交易为批发或零售业务？”，如选择“批发”，则视为金银首饰批发业务，生成版式文件左上角应展示“金银首饰批发”，发票上传时特定要素填写26：金银首饰批发；
         * 如选择“零售”，则视为金银首饰零售业务，继续弹窗提示：“该笔交易为金银首饰零售业务，请按规定进行消费税申报。”，发票上传时金银首饰交易类型填写27：金银首饰零售。
         *（2）若销售方为金银首饰归类纳税人，购买方非金银首饰归类登记纳税人，视为金银首饰零售业务，开具发票时弹窗提示：“请及时进行金银首饰消费税税（费）种认定并按规定进行消费税申报。”，发票上传时金银首饰交易类型应当填写27：金银首饰零售。
         *（3）若销售方非金银首饰归类纳税人，购买方为金银首饰归类登记纳税人，视为金银首饰零售业务，开具发票时弹窗提示：“请及时进行金银首饰消费税税（费）种认定并按规定进行消费税申报。”，发票上传时金银首饰交易类型应当填写27：金银首饰零售。
         *（4）若销售方和购买方均非金银首饰归类纳税人（包含购买方为自然人），视为零售业务，开具发票时弹窗提示：“请及时进行金银首饰消费税税（费）种认定并按规定进行消费税申报。”，发票上传时金银首饰交易类型应当填写27：金银首饰零售。
        *● 深圳试点期间，销售方纳税人必须是深圳罗湖区企业（区县级税务机关代码为14403030000），如果受票方为企业则必须也是甚至是罗湖区企业（可调用查询购方是否为试点地区企业的接口确认购方是否为深圳市罗湖区企业），如果受票方是个人（购买方自然人标志传Y或购买方纳税人识别号为空），则不限制购买方地区。
        * 销方资质 购方资质 特殊业务 前提条件
        *是 是 26：金银首饰批发/27：金银首饰零售 销购方均为试点省份企业
        *是 否 27：金银首饰零售 销购方均为试点省份企业/销方为试点省份企业&购方非企业
        *否 是 27：金银首饰零售 销购方均为试点省份企业
        *否 否 27：金银首饰零售 销购方均为试点省份企业/销方为试点省份企业&购方非企业
        * 其他均为非特殊业务
         */
        if (this.isSellerJewelryPilotFlag() && this.isBuyerJewelryPilotFlag()) {
            if (this.isSellerJewelryQualificationFlag() && this.isBuyerJewelryQualificationFlag()) {
                if ("26".equals(this.invoiceStyleType) || "27".equals(this.invoiceStyleType)) {
                    return result;
                }

                return BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("需开具金银首饰特殊业务发票，特殊业务代码传入值为空%s，应为：【26-金银首饰批发或27-金银首饰零售】",
                        CommonUtil.cleanBlankString(this.getInvoiceStyleType()))));
            }
            if ((this.isSellerJewelryQualificationFlag() && !this.isBuyerJewelryQualificationFlag())
                    || (!this.isSellerJewelryQualificationFlag() && this.isBuyerJewelryQualificationFlag())
                    || (!this.isSellerJewelryQualificationFlag() && !this.isBuyerJewelryQualificationFlag())) {
                if ("27".equals(this.invoiceStyleType)) {
                    return result;
                }
                return BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("金银首饰特殊业务代码不合法。传入特殊业务代码：【%s】，应为：【27-金银首饰零售】",
                        CommonUtil.cleanBlankString(this.getInvoiceStyleType()))));
            }
        }

        if (this.isSellerJewelryPilotFlag() && this.isBuyerNatureFlag()) {
            if ("27".equals(this.invoiceStyleType)) {
                return result;
            }
            return BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("金银首饰特殊业务代码不合法。传入特殊业务代码：【%s】，应为：【27-金银首饰零售】",
                    CommonUtil.cleanBlankString(this.getInvoiceStyleType()))));
        }

        // 都应该是非特殊业务
        if (StringUtils.isEmpty(this.invoiceStyleType)) {
            return result;
        }

        return BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("不能开具金银首饰特殊业务发票，请开具非特殊业务的发票。传入特殊业务代码：【%s】，应为空",
                CommonUtil.cleanBlankString(this.getInvoiceStyleType()))));
    }
}
