package com.xforceplus.taxware.chestnut.check.model.validator.invoice;

import com.alibaba.fastjson.annotation.JSONField;
import com.xforceplus.taxware.chestnut.check.model.common.Alias;
import com.xforceplus.taxware.chestnut.check.model.common.InvoiceStyleTypeGoodsTaxNoProvider;
import com.xforceplus.taxware.chestnut.check.model.common.RealEstateLeaseParkingFeeItemNameProvider;
import com.xforceplus.taxware.chestnut.check.model.common.ValidateResult;
import com.xforceplus.taxware.chestnut.check.model.constant.AreaUnitEnum;
import com.xforceplus.taxware.chestnut.check.model.constant.TaxIncentivesTypeEnum;
import com.xforceplus.taxware.chestnut.check.model.util.BasicValidator;
import com.xforceplus.taxware.chestnut.check.model.util.CommonUtil;
import com.xforceplus.taxware.chestnut.contract.model.constant.enums.CapabilityCodeEnum;
import lombok.Data;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.validator.constraints.Length;

import javax.validation.constraints.Digits;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Objects;

import static com.xforceplus.taxware.chestnut.check.model.constant.AreaUnitEnum.PINGFANGMI;

/**
 * @author Thj
 * @createdAt: 2023/12/18 11:17
 * @Description:
 */
@Data
public class RealEstateLeaseBaseValidator {
    private static final List<String> realEstatePlaceKeyWord = Arrays.asList("街", "路", "村", "乡", "镇", "道", "巷", "号");

    @Alias("不动产租赁明细列表")
    private List<RealEstateLeaseDetail> realEstateLeaseDetails;

    /**
     * 不动产租赁
     */
    private List<RealEstateLeaseInfo> realEstateLeaseInfoList;

    /**
     * 能力编码列表
     */
    private List<String> capabilityCodeList;

    @Data
    public static class RealEstateLeaseInfo {

        private Integer rowNum;

        /**
         * 房屋产权证书/不动产权证号码
         */
        @Alias("产权证书/不动产权证号码")
        @Length(max = 40)
        private String realEstateNo;

        /**
         * 不动产坐落地址
         */
        @Alias("不动产坐落省份")
        private String realEstateProvince;

        @Alias("不动产坐落市")
        private String realEstateCity;

        @Alias("不动产坐落详细地址")
        private String realEstatePlace;

        /**
         * 租赁期起
         */
        private Date leaseTermStart;
        private String leaseTermStartStr;

        /**
         * 租赁期止
         */
        private Date leaseTermEnd;
        private String leaseTermEndStr;

        /**
         * 跨地市标志
         */
        private Boolean crossCitySign;

        /**
         * 面积单位
         */
        private AreaUnitEnum areaUnit;

        /**
         * 车牌号
         */
        private List<String> vehicleNo;
    }

    @Data
    public static class RealEstateLeaseDetail {

        /**
         * 序号
         * 8位，从1开始。
         */
        @Alias("序号")
        @NotNull
        @Digits(integer = 8, fraction = 0)
        private Integer rowNum;

        @Alias("不动产租赁税编")
        @NotEmpty
        @Length(max = 19)
        private String goodsTaxNo;

        @Alias("项目名称")
        @NotEmpty
        private String itemSimpleName;

        /**
         * 税收优惠政策类型代码
         */
        private TaxIncentivesTypeEnum taxIncentivesType;

        /**
         * 金额
         */
        @Alias("金额")
        @NotNull
        @Digits(integer = 18, fraction = 2)
        private BigDecimal amountWithoutTax;

        /**
         * 税额
         */
        @Alias("税额")
        @NotNull
        @Digits(integer = 18, fraction = 2)
        private BigDecimal taxAmount;

        /**
         * 含税金额
         */
        @Alias("含税金额")
        @NotNull
        @Digits(integer = 18, fraction = 2)
        @JSONField(name = "hsje")
        private BigDecimal amountWithTax;
    }

    public ValidateResult validate() {

        // 先做主信息合规校验，不通过直接返回，不进行数据校验
        var result = BasicValidator.validate(this);
        result = BasicValidator.mergeValidateResult(result, BasicValidator.validate(this.getRealEstateLeaseInfoList()));
        result = BasicValidator.mergeValidateResult(result, BasicValidator.validate(this.getRealEstateLeaseDetails()));

        if (realEstateLeaseDetails == null || realEstateLeaseDetails.isEmpty()) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("不动产租赁明细列表明细不能为空"));
            return result;
        }

        if (realEstateLeaseInfoList == null || realEstateLeaseInfoList.isEmpty()) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("不动产租赁信息列表不能为空"));
            return result;
        }

        if (realEstateLeaseInfoList.size() != realEstateLeaseDetails.size()) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("不动产租赁发票明细行（被折扣行与折扣行视为1行）数量需要与特定要素信息数量保持一致。发票明细行数为【%s】，特定要素信息行数为【%s】",
                    realEstateLeaseDetails.size(), realEstateLeaseInfoList.size())));
            return result;
        }

        for (RealEstateLeaseInfo realEstateLeaseInfo : realEstateLeaseInfoList) {

            if (realEstateLeaseInfo.getCrossCitySign() == null) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d行跨地市标志不能为空", realEstateLeaseInfo.getRowNum())));
                return result;
            }

            if (realEstateLeaseInfo.getAreaUnit() == null) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d行面积单位不能为空", realEstateLeaseInfo.getRowNum())));
                return result;
            }

            if (realEstateLeaseInfo.getLeaseTermStart() == null || realEstateLeaseInfo.getLeaseTermEnd() == null) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d行起止日期不能为空", realEstateLeaseInfo.getRowNum())));
                return result;
            }

            if (StringUtils.isBlank(realEstateLeaseInfo.getRealEstatePlace()) || realEstatePlaceKeyWord.stream().noneMatch(x -> realEstateLeaseInfo.getRealEstatePlace().contains(x))) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d行不动产地址必须包含街、路、村、乡、镇、道、巷、号等任意一个关键词。传入值为【%s】",
                        realEstateLeaseInfo.getRowNum(), realEstateLeaseInfo.getRealEstatePlace())));
                return result;
            }

            if (realEstateLeaseInfo.getRealEstatePlace().length() > 80) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d行不动产租赁发票开具：不动产坐落地址（详细地址）总长度不能超过80，传入值为【%s】，长度【%s】", realEstateLeaseInfo.getRowNum(),
                        realEstateLeaseInfo.getRealEstatePlace(), realEstateLeaseInfo.getRealEstatePlace().length())));
                return result;
            }
        }

        // 能力编码校验
        if (CollectionUtils.isEmpty(this.getCapabilityCodeList()) || !this.getCapabilityCodeList().contains("06")) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("请前往电子税局“乐企数字开放平台”检查所使用的[%s]是否已被授权邀请", CapabilityCodeEnum.fromCode("06").toDesc())));
            return result;
        }

        // 先做明细信息合规校验，不通过直接返回，不再进行数据校验
        for (var detail : realEstateLeaseDetails) {
            final var validateResult = BasicValidator.validate(detail, detail.getRowNum() + "");
            result = BasicValidator.mergeValidateResult(result, validateResult);
        }

        return result;
    }

    public ValidateResult validateRealEstateLeaseData(InvoiceStyleTypeGoodsTaxNoProvider provider) {
        var result = ValidateResult.success();
        for (int i = 0; i < this.getRealEstateLeaseDetails().size(); i++) {

            final var realEstateLeaseDetail = this.getRealEstateLeaseDetails().get(i);

            final var realEstateLeaseGoodTaxNoList = provider.provideGoodsTaxNo();
            if (!realEstateLeaseGoodTaxNoList.contains(realEstateLeaseDetail.getGoodsTaxNo())) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d行开具发票时，使用的商品编码应为税收分类编码中不动产租赁类商品编码，请检查。传入值为【%s】",
                        realEstateLeaseDetail.getRowNum(), realEstateLeaseDetail.getGoodsTaxNo())));
                return result;
            }

            if (Objects.equals(realEstateLeaseDetail.getGoodsTaxNo(), "3040502020102000000")) {
                if (realEstateLeaseDetail.getTaxIncentivesType() != TaxIncentivesTypeEnum.FIVE_PERCENT_SIMPLE_EXPROPRIATION_REDUCE_ONE_POINT_FIVE_PERCENT) {
                    result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d行个体工商户或自然人在使用“3040502020102000000个人出租住房税编”开具不动产租赁发票时，只能按照5%%减按1.5%%计算应纳税额，传入值为【%s】",
                            realEstateLeaseDetail.getRowNum(), realEstateLeaseDetail.getTaxIncentivesType())));
                    return result;
                }

                final var amountWithTax = realEstateLeaseDetail.getAmountWithTax();
                BigDecimal onePlusFivePercent = new BigDecimal("1.05");
                BigDecimal onePointFivePercent = new BigDecimal("0.015");

                final var calculateTaxAmount = amountWithTax.multiply(onePointFivePercent).divide(onePlusFivePercent, BigDecimal.ROUND_HALF_UP);
                final var calculateAmountWithoutTax = amountWithTax.subtract(calculateTaxAmount);

                final var judgeNum = new BigDecimal("0.01");
                if (realEstateLeaseDetail.getTaxAmount().subtract(calculateTaxAmount).abs().compareTo(judgeNum) > 0) {
                    result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d行应纳税额计算方式为：税额=含税销售额*1.5%%/（1+5%%），请检查税额是否正确，传入值为【%s!=%s*1.5%%/（1+5%%）】",
                            realEstateLeaseDetail.getRowNum(), realEstateLeaseDetail.getTaxAmount(), amountWithTax)));
                    return result;
                }
                if (realEstateLeaseDetail.getAmountWithoutTax().subtract(calculateAmountWithoutTax).abs().compareTo(judgeNum) > 0) {
                    result = BasicValidator.mergeValidateResult(result,
                            ValidateResult.fail(String.format("第%d行应纳税额计算方式为：税额=含税销售额*1.5%%/（1+5%%），不含税金额=含税销售额-税额。请检查不含税金额是否正确，传入值为【%s!=%s-（%s*1.5%%/（1+5%%））】",
                            realEstateLeaseDetail.getRowNum(),realEstateLeaseDetail.getAmountWithoutTax(), amountWithTax, amountWithTax)));
                    return result;
                }
            }
        }

        return result;
    }

    public ValidateResult validateParkingFee(RealEstateLeaseParkingFeeItemNameProvider provider) {
        var result = ValidateResult.success();

        for (int i = 0; i < this.getRealEstateLeaseInfoList().size(); i++) {
            final var realEstateLeaseInfo = this.getRealEstateLeaseInfoList().get(i);
            final var realEstateLeaseDetail = this.getRealEstateLeaseDetails().get(i);
            var leaseTermStartStr = realEstateLeaseInfo.getLeaseTermStartStr();
            final var leaseTermEndStr = realEstateLeaseInfo.getLeaseTermEndStr();

            // 停车费业务
            if (Objects.equals(realEstateLeaseDetail.getGoodsTaxNo(), "3040502020200000000")) {
                // 1、单位必须为“平方米” ；
                // 2、项目名称必须为“车辆停放服务” ；
                // 3、车牌号必填， 最多支持填写 3 个车牌号； 除该场景车牌号必须为空；
                // 4、租赁期起止格式必须为“ yyyy-MM-dd HH:mm（ 租赁期起） ” +“ 空格” +“yyyy-MM-dd HH:mm（租赁期止） 。
                if (realEstateLeaseInfo.getAreaUnit() != PINGFANGMI) {
                    result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d行不动产租赁停车费业务，面积单位必须为“平方米”，传入值为【%s】",
                            realEstateLeaseDetail.getRowNum(), realEstateLeaseInfo.getAreaUnit())));
                    return result;
                }

                if (!Objects.equals(realEstateLeaseDetail.getItemSimpleName(), "车辆停放服务")) {
                    result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d行不动产租赁停车费业务，项目名称必须为“车辆停放服务”，传入值为【%s】",
                            realEstateLeaseDetail.getRowNum(), realEstateLeaseDetail.getItemSimpleName())));
                }

                if (CollectionUtils.isNotEmpty(realEstateLeaseInfo.getVehicleNo())) {

                    if (realEstateLeaseInfo.getVehicleNo().size() > 3) {
                        result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d行不动产租赁停车费业务，车牌号最多支持3个车牌号，传入值为【%s】",
                                realEstateLeaseDetail.getRowNum(), realEstateLeaseInfo.getVehicleNo())));
                    }

                    if (realEstateLeaseInfo.getVehicleNo().stream().anyMatch(vehicleNo -> vehicleNo.length() > 20)) {
                        result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d行不动产租赁停车费业务，每个车牌号长度不能超过20，传入值为【%s】",
                                realEstateLeaseDetail.getRowNum(), realEstateLeaseInfo.getVehicleNo())));
                    }

                }

                // 到天不拦截，后续自动补充0
                if (!StringUtils.isBlank(leaseTermStartStr) && leaseTermStartStr.length() == 8) {
                    return result;
                }

                if (StringUtils.isBlank(leaseTermStartStr) || leaseTermStartStr.length() != 12
                        || StringUtils.isBlank(leaseTermEndStr) || leaseTermEndStr.length() != 12) {
                    result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d行不动产租赁停车费业务，起止日期不能为空，且格式应为yyyyMMddHHmm，传入值为【%s】",
                            realEstateLeaseDetail.getRowNum(), CommonUtil.cleanBlankString(leaseTermStartStr))));
                    return result;
                }
            }

            // 非停车费业务
            else {
                if (CollectionUtils.isNotEmpty(realEstateLeaseInfo.getVehicleNo())) {
                    result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d行不动产租赁非停车费业务，车牌号必须为空，传入值为【%s】",
                            realEstateLeaseDetail.getRowNum(), realEstateLeaseInfo.getVehicleNo())));
                    return result;
                }

                if (StringUtils.isBlank(leaseTermStartStr) || leaseTermStartStr.length() != 8
                        || StringUtils.isBlank(leaseTermEndStr) || leaseTermEndStr.length() != 8) {
                    result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d行不动产租赁非停车费业务，起止日期不能为空，且格式应为yyyyMMdd，传入值为【%s】",
                            realEstateLeaseDetail.getRowNum(), CommonUtil.cleanBlankString(leaseTermStartStr))));
                    return result;
                }
            }
        }

        return result;
    }
}
