package com.xforceplus.taxware.chestnut.check.model.validator.invoice;

import com.google.common.collect.Lists;
import com.xforceplus.taxware.chestnut.check.model.common.Alias;
import com.xforceplus.taxware.chestnut.check.model.common.InvoiceStyleTypeGoodsTaxNoProvider;
import com.xforceplus.taxware.chestnut.check.model.common.ValidateResult;
import com.xforceplus.taxware.chestnut.check.model.constant.AreaUnitEnum;
import com.xforceplus.taxware.chestnut.check.model.util.BasicValidator;
import com.xforceplus.taxware.chestnut.contract.model.constant.enums.CapabilityCodeEnum;
import lombok.Data;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.validator.constraints.Length;

import javax.validation.constraints.Digits;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.math.BigDecimal;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author Thj
 * @createdAt: 2024/1/11 17:43
 * @Description:
 */
@Data
public class RealEstateSalesBaseValidator {
    private static final List<String> realEstatePlaceKeyWord = Arrays.asList("街", "路", "村", "乡", "镇", "道", "巷", "号");

    @Alias("不动产销售明细列表")
    private List<RealEstateSalesDetail> realEstateSalesDetailList;

    /**
     * 不动产销售
     */
    private List<RealEstateSalesInfo> realEstateSalesInfoList;

    /**
     * 能力编码列表
     */
    private List<String> capabilityCodeList;

    @Data
    public static class RealEstateSalesInfo {

        private Integer rowNum;

        /**
         * 房屋产权证书/不动产权证号码
         */
        @Alias("房屋产权证书/不动产权证号码")
        @Length(max = 40)
        private String realEstateNo;

        /**
         * 不动产坐落地址
         */
        @Alias("不动产坐落省份")
        private String realEstateProvince;

        @Alias("不动产坐落市")
        private String realEstateCity;

        @Alias("不动产坐落详细地址")
        private String realEstatePlace;

        /**
         * 跨地市标志
         */
        private Boolean crossCitySign;

        /**
         * 面积单位
         */
        private AreaUnitEnum areaUnit;

        /**
         * 不动产单元代码/网签合同备案编码
         */
        @Alias("不动产单元代码/网签合同备案编码")
        @Length(max = 28)
        private String realEstateCode;

        /**
         * 土地增值税项目编号
         */
        @Alias("土地增值税项目编号")
        @Length(max = 16)
        private String landVatItemNo;

        /**
         * 核定计税价格
         */
        @Alias("核定计税价格")
        @Digits(integer = 18, fraction = 2)
        private BigDecimal taxablePrice;

        /**
         * 实际成交含税金额
         */
        @Alias("实际成交含税金额")
        @Digits(integer = 18, fraction = 2)
        private BigDecimal transactionPrice;
    }

    @Data
    public static class RealEstateSalesDetail {

        /**
         * 序号
         * 8位，从1开始。
         */
        @Alias("序号")
        @NotNull
        @Digits(integer = 8, fraction = 0)
        private Integer rowNum;

        @Alias("不动产销售税编")
        @NotEmpty
        @Length(max = 19)
        private String goodsTaxNo;
    }

    public ValidateResult validate() {

        // 先做主信息合规校验，不通过直接返回，不进行数据校验
        var result = BasicValidator.validate(this);
        result = BasicValidator.mergeValidateResult(result, BasicValidator.validate(this.getRealEstateSalesInfoList()));
        result = BasicValidator.mergeValidateResult(result, BasicValidator.validate(this.getRealEstateSalesDetailList()));

        if (realEstateSalesDetailList == null || realEstateSalesDetailList.isEmpty()) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("不动产销售明细列表明细不能为空"));
            return result;
        }

        if (realEstateSalesInfoList == null || realEstateSalesInfoList.isEmpty()) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("不动产销售信息列表不能为空"));
            return result;
        }

        if (realEstateSalesInfoList.size() != realEstateSalesDetailList.size()) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("不动产销售发票明细行（被折扣行与折扣行视为1行）数量需要与特定要素信息数量保持一致。发票明细行数为【%s】，特定要素信息行数为【%s】",
                    realEstateSalesDetailList.size(), realEstateSalesInfoList.size())));
            return result;
        }

        for (RealEstateSalesInfo realEstateSalesInfo : realEstateSalesInfoList) {
            if (StringUtils.isBlank(realEstateSalesInfo.getRealEstatePlace()) || realEstatePlaceKeyWord.stream().noneMatch(x -> realEstateSalesInfo.getRealEstatePlace().contains(x))) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d行不动产销售特定要素列表，不动产地址必须包含街、路、村、乡、镇、道、巷、号等任意一个关键词。传入值为【%s】",
                        realEstateSalesInfo.getRowNum(), realEstateSalesInfo)));
                return result;
            }
            final List<String> specialProviceList = Lists.newArrayList("台湾省", "香港特别行政区", "澳门特别行政区");
            if (specialProviceList.stream().noneMatch(specialProvice -> Objects.equals(realEstateSalesInfo.getRealEstateProvince(), specialProvice))) {
                if (StringUtils.isBlank(realEstateSalesInfo.getRealEstateCity())) {
                    result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d行不动产销售特定要素列表， 当“不动产坐落地址（省）”为香港特别行政区、澳门特别行政区、台湾省时，“不动产坐落地址（市）”可为空，其他情况必填。",
                            realEstateSalesInfo.getRowNum())));
                    return result;
                }
            }

            if (realEstateSalesInfo.getRealEstatePlace().length() > 80) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d行不动产销售特定要素列表，不动产坐落地址（详细地址）总长度不能超过80，传入值为【%s】，长度为【%s】",
                        realEstateSalesInfo.getRowNum(), realEstateSalesInfo.getRealEstatePlace(), realEstateSalesInfo.getRealEstatePlace().length())));
                return result;
            }

            if (realEstateSalesInfo.getCrossCitySign() == null) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d行不动产销售特定要素列表，跨地市标志不能为空", realEstateSalesInfo.getRowNum())));
                return result;
            }

            if (realEstateSalesInfo.getAreaUnit() == null) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d行不动产销售特定要素列表，面积单位不能为空", realEstateSalesInfo.getRowNum())));
                return result;
            }

            if (realEstateSalesInfo.getTaxablePrice() != null && realEstateSalesInfo.getTransactionPrice() == null) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d行不动产销售特定要素列表，实际成交含税金额若按核定计税价格征税的，实际成交含税金额必填。", realEstateSalesInfo.getRowNum())));
                return result;
            }
        }
        final var provinceList = realEstateSalesInfoList.stream().map(RealEstateSalesInfo::getRealEstateProvince).distinct().collect(Collectors.toList());
        if (provinceList.size() > 1) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("开具多行发票明细时，每行发票明细“不动产坐落地址（省）[%s]”必须相同。", StringUtils.join(provinceList, ","))));
            return result;

        }

        // 能力编码校验
        if (CollectionUtils.isEmpty(this.getCapabilityCodeList()) || !this.getCapabilityCodeList().contains("05")) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("请前往电子税局“乐企数字开放平台”检查所使用的[%s]是否已被授权邀请", CapabilityCodeEnum.fromCode("05").toDesc())));
            return result;
        }

        return result;
    }

    public ValidateResult validateRealEstateSalesData(InvoiceStyleTypeGoodsTaxNoProvider provider) {
        var result = ValidateResult.success();

        final var realEstateSalesDetail = this.getRealEstateSalesDetailList().get(0);

        final var realEstateSalesGoodTaxNoList = provider.provideGoodsTaxNo();
        if (!realEstateSalesGoodTaxNoList.contains(realEstateSalesDetail.getGoodsTaxNo())) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("开具发票时，使用的商品编码应为税收分类编码中不动产销售类商品编码，请检查，传入值为【%s】", realEstateSalesDetail.getGoodsTaxNo())));
            return result;
        }

        return result;
    }
}
