package com.xforceplus.taxware.chestnut.check.model.validator.invoice;

import cn.hutool.core.bean.BeanUtil;
import com.xforceplus.taxware.architecture.g1.domain.exception.TXWR000002Exception;
import com.xforceplus.taxware.chestnut.check.model.base.BaseDetail;
import com.xforceplus.taxware.chestnut.check.model.util.CommonUtil;
import com.xforceplus.taxware.chestnut.check.model.util.NumberUtil;
import com.xforceplus.taxware.chestnut.check.model.validator.redletter.RedLetterValidator;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.math.BigDecimal;
import java.util.Comparator;
import java.util.Objects;

/**
 * @Auther lv
 * @Date 2023/2/16
 * @Descrption 红票开具时：发票信息与红字确认单比对
 * 红字发票明细和红字确认单必须保持一致
 */
public class RedInvoiceVsRedLetterValidator {

    public static void validate(final InvoiceBaseValidator invoiceBase, final RedLetterValidator redLetter) {

        if (CollectionUtils.isEmpty(invoiceBase.getInvoiceDetailList()) || CollectionUtils.isEmpty(redLetter.getDetails())) {
            throw new TXWR000002Exception("明细行数据不能为空");
        }

        if (!invoiceBase.getInvoiceBaseInfo().getInvoiceType().equals(redLetter.getOriginalInvoiceInfo().getOriginalInvoiceType())) {
            throw new TXWR000002Exception("红票和原蓝票发票类型不匹配");
        }

        // 排序
        invoiceBase.getInvoiceDetailList().sort(Comparator.comparing(InvoiceBaseValidator.InvoiceDetail::getRowNum));
        redLetter.getDetails().sort(Comparator.comparing(RedLetterValidator.Detail::getRowNum));

        // 将所有空字符串替换为null
        setNull(invoiceBase, redLetter);

        for (int i = 0; i < invoiceBase.getInvoiceDetailList().size(); i++) {
            BaseDetail invoiceDetail = invoiceBase.getInvoiceDetailList().get(i);
            BaseDetail invoiceDetailBase = new BaseDetail();
            BeanUtil.copyProperties(invoiceDetail, invoiceDetailBase);

            BaseDetail redletterDetail = redLetter.getDetails().get(i);
            BaseDetail redletterDetailBase = new BaseDetail();
            BeanUtil.copyProperties(redletterDetail, redletterDetailBase);

            if (!Objects.equals(invoiceDetailBase.getOriginalRowNum(), redletterDetailBase.getOriginalRowNum())) {
                throw new TXWR000002Exception(String.format("红票和红字确认单第%s行明细中，原发票明细行序号不相等，红票[originalRowNum=%d],红字确认单[originalRowNum=%d]",
                        i + 1, invoiceDetailBase.getOriginalRowNum(), redletterDetailBase.getOriginalRowNum()));
            }
            if (!Objects.equals(invoiceDetailBase.getRowNum(), redletterDetailBase.getRowNum())) {
                throw new TXWR000002Exception(String.format("红票和红字确认单第%s行明细中，明细行序号不相等，红票[rowNum=%d],红字确认单[rowNum=%d]",
                        i + 1, invoiceDetailBase.getRowNum(), redletterDetailBase.getRowNum()));
            }
            if (!Objects.equals(invoiceDetailBase.getGoodsTaxNo(), redletterDetailBase.getGoodsTaxNo())) {
                throw new TXWR000002Exception(String.format("红票和红字确认单第%s行明细中，税编不相等，红票[goodsTaxNo=%s],红字确认单[goodsTaxNo=%s]",
                        i + 1, invoiceDetailBase.getGoodsTaxNo(), redletterDetailBase.getGoodsTaxNo()));
            }
            if (!Objects.equals(invoiceDetailBase.getItemName(), redletterDetailBase.getItemName())) {
                throw new TXWR000002Exception(String.format("红票和红字确认单第%s行明细中，商品名称不相等，红票[itemName=%s],红字确认单[itemName=%s]",
                        i + 1, invoiceDetailBase.getItemName(), redletterDetailBase.getItemName()));
            }
            if (!Objects.equals(invoiceDetailBase.getUnit(), redletterDetailBase.getUnit())) {
                throw new TXWR000002Exception(String.format("红票和红字确认单第%s行明细中，单位不相等，红票[unit=%s],红字确认单[unit=%s]",
                        i + 1, CommonUtil.cleanBlankString(invoiceDetailBase.getUnit()), CommonUtil.cleanBlankString(redletterDetailBase.getUnit())));
            }

            // 强化校验 先判断单价数量是否为数值 如果是 按数值判断 非数值走强制判断
            if (NumberUtil.isNumber(invoiceDetailBase.getUnitPrice()) && NumberUtil.isNumber(redletterDetailBase.getUnitPrice())) {
                if (new BigDecimal(invoiceDetailBase.getUnitPrice()).compareTo(new BigDecimal(redletterDetailBase.getUnitPrice())) != 0) {
                    throw new TXWR000002Exception(String.format("红票和红字确认单第%s行明细中，单价不相等，红票[unitPrice=%s],红字确认单[unitPrice=%s]",
                            i + 1, invoiceDetailBase.getUnitPrice(), redletterDetailBase.getUnitPrice()));
                }
            } else if (!Objects.equals(invoiceDetailBase.getUnitPrice(), redletterDetailBase.getUnitPrice())) {
                throw new TXWR000002Exception(String.format("红票和红字确认单第%s行明细中，单价不相等，红票[unitPrice=%s],红字确认单[unitPrice=%s]",
                        i + 1, invoiceDetailBase.getUnitPrice(), redletterDetailBase.getUnitPrice()));
            }
            if (NumberUtil.isNumber(invoiceDetailBase.getQuantity()) && NumberUtil.isNumber(redletterDetailBase.getQuantity())) {
                if (new BigDecimal(invoiceDetailBase.getQuantity()).compareTo(new BigDecimal(redletterDetailBase.getQuantity())) != 0) {
                    throw new TXWR000002Exception(String.format("红票和红字确认单第%s行明细中，数量不相等，红票[quantity=%s],红字确认单[quantity=%s]",
                            i + 1, invoiceDetailBase.getQuantity(), redletterDetailBase.getQuantity()));
                }
            } else if (!Objects.equals(invoiceDetailBase.getQuantity(), redletterDetailBase.getQuantity())) {
                throw new TXWR000002Exception(String.format("红票和红字确认单第%s行明细中，数量不相等，红票[quantity=%s],红字确认单[quantity=%s]",
                        i + 1, invoiceDetailBase.getQuantity(), redletterDetailBase.getQuantity()));
            }

            if (!Objects.equals(invoiceDetailBase.getSpecifications(), redletterDetailBase.getSpecifications())) {
                throw new TXWR000002Exception(String.format("红票和红字确认单第%s行明细中，数量不相等，红票[specifications=%s],红字确认单[specifications=%s]",
                        i + 1, invoiceDetailBase.getSpecifications(), redletterDetailBase.getSpecifications()));
            }
            if (invoiceDetailBase.getAmountWithoutTax().compareTo(redletterDetailBase.getAmountWithoutTax()) != 0) {
                throw new TXWR000002Exception(String.format("红票和红字确认单第%s行明细中，不含税金额不相等，红票[amountWithoutTax=%s],红字确认单[amountWithoutTax=%s]",
                        i + 1, invoiceDetailBase.getAmountWithoutTax(), redletterDetailBase.getAmountWithoutTax()));
            }
            if (invoiceDetailBase.getTaxRate() != null && redletterDetailBase.getTaxRate() != null && invoiceDetailBase.getTaxRate().compareTo(redletterDetailBase.getTaxRate()) != 0) {
                throw new TXWR000002Exception(String.format("红票和红字确认单第%s行明细中，税率不相等，红票[taxRate=%s],红字确认单[taxRate=%s]",
                        i + 1, invoiceDetailBase.getTaxRate(), redletterDetailBase.getTaxRate()));
            }

            if (invoiceDetailBase.getTaxAmount().compareTo(redletterDetailBase.getTaxAmount()) != 0) {
                throw new TXWR000002Exception(String.format("红票和红字确认单第%s行明细中，税额不相等，红票[taxAmount=%s],红字确认单[taxAmount=%s]",
                        i + 1, invoiceDetailBase.getTaxAmount(), redletterDetailBase.getTaxAmount()));
            }
        }
    }

    private static void setNull(InvoiceBaseValidator invoiceBase, RedLetterValidator redLetter) {
        invoiceBase.getInvoiceDetailList().forEach(detail -> {
            if (StringUtils.isBlank(detail.getSpecifications())) {
                detail.setSpecifications(null);
            }
            if (StringUtils.isBlank(detail.getQuantity())) {
                detail.setQuantity(null);
            }
            if (StringUtils.isBlank(detail.getUnit())) {
                detail.setUnit(null);
            }
            if (StringUtils.isBlank(detail.getUnitPrice())) {
                detail.setUnitPrice(null);
            }
            if (StringUtils.isBlank(detail.getGoodsTaxNo())) {
                detail.setGoodsTaxNo(null);
            }
            if (StringUtils.isBlank(detail.getItemName())) {
                detail.setItemName(null);
            }
        });
        redLetter.getDetails().forEach(detail -> {
            if (StringUtils.isBlank(detail.getSpecifications())) {
                detail.setSpecifications(null);
            }
            if (StringUtils.isBlank(detail.getQuantity())) {
                detail.setQuantity(null);
            }
            if (StringUtils.isBlank(detail.getUnit())) {
                detail.setUnit(null);
            }
            if (StringUtils.isBlank(detail.getUnitPrice())) {
                detail.setUnitPrice(null);
            }
            if (StringUtils.isBlank(detail.getGoodsTaxNo())) {
                detail.setGoodsTaxNo(null);
            }
            if (StringUtils.isBlank(detail.getItemName())) {
                detail.setItemName(null);
            }
        });
    }
}
