package com.xforceplus.taxware.chestnut.check.model.validator.invoice;

import com.xforceplus.taxware.architecture.g1.domain.exception.TXWR000002Exception;
import com.xforceplus.taxware.chestnut.check.model.common.Alias;
import com.xforceplus.taxware.chestnut.check.model.common.InvoiceStyleTypeGoodsTaxNoProvider;
import com.xforceplus.taxware.chestnut.check.model.common.ValidateResult;
import com.xforceplus.taxware.chestnut.check.model.constant.TransportationTypeEnum;
import com.xforceplus.taxware.chestnut.check.model.util.BasicValidator;
import com.xforceplus.taxware.chestnut.contract.model.constant.enums.CapabilityCodeEnum;
import lombok.Data;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.validator.constraints.Length;

import javax.validation.constraints.Digits;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.List;

/**
 * @author Thj
 * @createdAt: 2024/3/1 14:35
 * @Description:
 */
@Data
public class TransportServiceValidator {

    @Alias("货物运输明细列表")
    private List<TransportServiceDetail> transportServiceDetailList;

    /**
     * 能力编码列表
     */
    private List<String> capabilityCodeList;

    private List<TransportServiceInfo> transportServiceInfoList;

    @Data
    public static class TransportServiceDetail {

        /**
         * 序号
         * 8位，从1开始。
         */
        @Alias("序号")
        @NotNull
        @Digits(integer = 8, fraction = 0)
        private Integer rowNum;

        @Alias("货物运输税编")
        @NotEmpty
        @Length(max = 19)
        private String goodsTaxNo;
    }

    @Data
    public static class TransportServiceInfo {

        /**
         * 起运地
         * varchar
         * 80
         * 是
         */
        private String placeOfShipment;

        /**
         * 到达地
         * varchar
         * 80
         * 是
         */
        private String destination;

        /**
         * 运输工具种类
         * varchar
         * 40
         * 是
         * 铁路运输
         * 公路运输
         * 水路运输
         * 航空运输
         * 管道运输
         */
        private TransportationTypeEnum toolType;

        /**
         * 运输工具牌号
         * varchar
         * 40
         * 是
         */
        private String toolBrand;

        /**
         * yshwmc
         * varchar
         * 80
         * 是
         */
        private String nameOfGoods;
    }

    public ValidateResult validate() {
        // 先做主信息合规校验，不通过直接返回，不进行数据校验
        var result = ValidateResult.success();
        if (transportServiceDetailList == null || transportServiceDetailList.isEmpty()) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("货物运输明细列表明细不能为空"));
            return result;
        }

        if (transportServiceInfoList == null || transportServiceInfoList.isEmpty()) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("货物运输信息列表不能为空"));
            return result;
        }

        for (int i = 0; i < transportServiceInfoList.size(); i++) {
            final var transportServiceInfo = transportServiceInfoList.get(i);
            if (transportServiceInfo == null) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d条货物运输明细，货物运输信息不能为空", i + 1)));
                return result;
            }

            if (StringUtils.isBlank(transportServiceInfo.getPlaceOfShipment())) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d条货物运输明细，货物运输起运地不能为空", i + 1)));
                return result;
            }

            if (StringUtils.isBlank(transportServiceInfo.getDestination())) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d条货物运输明细，货物运输到达地不能为空", i + 1)));
                return result;
            }

            if (StringUtils.isBlank(transportServiceInfo.getToolBrand())) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d条货物运输明细，货物运输运输工具牌号不能为空", i + 1)));
                return result;
            }

            if (StringUtils.isBlank(transportServiceInfo.getNameOfGoods())) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("第%d条货物运输明细，货物运输运输货物名称不能为空", i + 1)));
                return result;
            }

            if (transportServiceInfo.getToolType() == null) {
                result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("货物运输工具种类不能为空、且应为[铁路运输、公路运输、水路运输、航空运输、管道运输]中之一"));
                return result;
            }
        }

        // 能力编码校验
        if (CollectionUtils.isEmpty(this.getCapabilityCodeList()) || !this.getCapabilityCodeList().contains("04")) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail(String.format("请前往电子税局“乐企数字开放平台”检查所使用的[%s]是否已被授权邀请", CapabilityCodeEnum.fromCode("04").toDesc())));
            return result;
        }

        return result;
    }

    public void validateTransportServiceData(InvoiceStyleTypeGoodsTaxNoProvider provider) {

        final var transportServiceDetailList = this.getTransportServiceDetailList();

        final var transportServiceGoodTaxNoList = provider.provideGoodsTaxNo();

        transportServiceDetailList.forEach(detail -> {
            if (!transportServiceGoodTaxNoList.contains(detail.getGoodsTaxNo())) {
                throw new TXWR000002Exception(String.format("第%d行明细，开具发票时，使用的商品编码应为税收分类编码中货物运输类商品编码，请检查。传入值为【%s】", detail.getRowNum(), detail.getGoodsTaxNo()));
            }
        });
    }
}
