package com.xforceplus.taxware.chestnut.check.model.validator.redletter;

import com.xforceplus.taxware.chestnut.check.model.base.BaseDetail;
import com.xforceplus.taxware.chestnut.check.model.base.BuyerInfo;
import com.xforceplus.taxware.chestnut.check.model.base.OriginalInvoiceInfo;
import com.xforceplus.taxware.chestnut.check.model.base.SellerInfo;
import com.xforceplus.taxware.chestnut.check.model.common.Alias;
import com.xforceplus.taxware.chestnut.check.model.common.ValidateResult;
import com.xforceplus.taxware.chestnut.check.model.util.BasicValidator;
import com.xforceplus.taxware.chestnut.check.model.util.CommonUtil;
import lombok.Data;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.validator.constraints.Length;

import javax.validation.constraints.Digits;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import javax.validation.constraints.Pattern;
import java.math.BigDecimal;
import java.util.*;

/**
 * 全电红字确认单 - 校验模型
 */
@Data
public class RedLetterValidator {

    /**
     * 录入方身份
     * 0：销方
     * 1：购方
     */
    @Alias("录入方身份")
    @NotEmpty(message = "录入方身份不能为空")
    @Length(max = 1, message = "录入方身份长度不能超过1")
    @Pattern(regexp = "0|1", message = "录入方身份不是有效值")
    private String applyIdentity;

    /**
     * 销方信息
     */
    private SellerInfo sellerInfo;

    /**
     * 购方信息
     */
    private BuyerInfo buyerInfo;

    /**
     * 原蓝票发票信息
     */
    private OriginalInvoiceInfo originalInvoiceInfo;

    /**
     * 红字冲销金额
     */
    @Alias("红字冲销金额")
    @NotNull
    @Digits(integer = 18, fraction = 2)
    private BigDecimal reverseAmountWithoutTax;

    /**
     * 红字冲销税额
     */
    @Alias("红字冲销税额")
    @NotNull
    @Digits(integer = 18, fraction = 2)
    private BigDecimal reverseTaxAmount;

    /**
     * 红字发票冲红原因代码
     * 01：开票有误
     * 02：销货退回
     * 03：服务中止
     * 04：销售折让
     */
    @Alias("红字发票冲红原因代码")
    @NotEmpty
    @Pattern(regexp = "0[1-4]")
    private String applyReason;

    private List<Detail> details;

    public ValidateResult validate() {
        var result = BasicValidator.validate(this);
        result = BasicValidator.mergeValidateResult(result, BasicValidator.validate(this.getSellerInfo()));
        result = BasicValidator.mergeValidateResult(result, BasicValidator.validate(this.getBuyerInfo()));
        result = BasicValidator.mergeValidateResult(result, BasicValidator.validate(this.getOriginalInvoiceInfo()));

        final var detailList = this.getDetails();
        if (detailList == null || detailList.size() == 0) {
            result = BasicValidator.mergeValidateResult(result, ValidateResult.fail("明细行数据不能为空"));
            return result;
        }
        // 特定要素
        final var invoiceStyleType = this.getOriginalInvoiceInfo().getInvoiceStyleType();

        //先做明细信息合规校验，不通过直接返回，不再进行数据校验
        for (var detail : detailList) {
            final var validateResult = BasicValidator.validate(detail, detail.getRowNum() + "");
            result = BasicValidator.mergeValidateResult(result, validateResult);

            // 特定要素
            if (StringUtils.isNotBlank(invoiceStyleType)) {
                result = BasicValidator.mergeValidateResult(result, validateStyleType(invoiceStyleType, detail));
            }
        }
        if (result.isSuccess()) {
            result = BasicValidator.mergeValidateResult(result, validateData());
        }
        return result;
    }

    private ValidateResult validateStyleType(final String invoiceStyleType, final Detail detail) {

        // 成品油
        if (Objects.equals(invoiceStyleType, "01")) {
            final var unit = detail.getUnit();
            if (StringUtils.isBlank(unit) || !Arrays.asList("升", "吨").contains(unit)) {
                return ValidateResult.fail(
                        Collections.singletonList(
                                new ValidateResult.ErrorInfo(String.format("成品油第%d行明细单位错误,只能是'升'、'吨'两个选项，传入值为【%s】", detail.getRowNum(), CommonUtil.cleanBlankString(unit)))
                        )
                );
            }
        }

        return ValidateResult.success();
    }

    /**
     * 数据校验
     */
    private ValidateResult validateData() {
        // 含税金额=税额+金额，不能有误差。  没有含税金额,不做校验.

        final List<ValidateResult.ErrorInfo> errorInfoList = new ArrayList<>();

        if (this.getReverseAmountWithoutTax().compareTo(this.getOriginalInvoiceInfo().getInvoiceAmountInfo().getAmountWithoutTax()) > 0) {
            errorInfoList.add(new ValidateResult.ErrorInfo(String.format("红票开具金额大于对应蓝票开具金额，红票开具金额绝对值为【%s】，对应蓝票开具金额为【%s】",
                    this.getReverseAmountWithoutTax().abs(), this.getOriginalInvoiceInfo().getInvoiceAmountInfo().getAmountWithoutTax())));
        }
        if (errorInfoList.size() > 0) {
            return ValidateResult.fail(errorInfoList);
        }
        return ValidateResult.success();
    }

    @Data
    public static class Detail extends BaseDetail {
        /**
         * 货物或应税劳务、服务名称
         */
        @Alias("货物或应税劳务、服务名称")
        @NotEmpty
        @Length(max = 300)
        private String itemSimpleName;

        /**
         * 商品服务简称
         */
        @Alias("商品服务简称")
        @NotEmpty
        @Length(max = 120)
        private String itemShortName;
    }
}
