package com.xforceplus.taxware.chestnut.check.model.validator.tax;

import cn.hutool.core.util.StrUtil;
import com.xforceplus.taxware.architecture.g1.domain.exception.TXWR000002Exception;
import com.xforceplus.taxware.architecture.g1.domain.exception.TXWR000004Exception;
import com.xforceplus.taxware.chestnut.check.model.base.TaxCode;
import com.xforceplus.taxware.chestnut.contract.model.constant.enums.TaxCodeStatusEnum;
import org.apache.commons.lang3.StringUtils;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.ZoneId;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @Auther lv
 * @Date 2023/2/24
 * 失败情况：
 * 1。税编记录不存在
 * 2。税编不是叶子节点：sfhzx：Y
 * 3。不在有效期内：小于qyrq启用日期 || 大于tyrq停用日期
 * 4。简称是否跟查询到的税编一致
 * todo 甄孟莹 5。税率范围应在多税率范围内：TXWRECMI0014-发票明细有 1 行不合法。第 2 行【满口香】，税编是 1030110040000000000：税率不合法!请使用如下税率:0.09；
 */
public class TaxCodeValidator {
    public static void validate(String itemName, TaxCode taxCode) {
        if (null == taxCode) {
            throw new TXWR000004Exception(String.format("税编[%s]不存在，禁止开票", itemName));
        }

        if (!StringUtils.equals(taxCode.getItemShortName(), itemName)) {
            throw new TXWR000004Exception(String.format("税编[%s]简称有误，期望简称[%s], 实际简称[%s], 禁止开票", taxCode.getGoodsTaxNo(), taxCode.getItemShortName(), itemName));
        }

        if (TaxCodeStatusEnum.TAX_RATE_STATUS_Y.getStatus().equals(taxCode.getSummarizeItemFlag())) {
            throw new TXWR000004Exception(String.format("税编[%s]汇总项标识错误，该税编是汇总项，禁止开票", taxCode.getGoodsTaxNo()));
        }

        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
        Date now = Date.from(LocalDate.now().atStartOfDay(ZoneId.systemDefault()).toInstant());

        try {
            if (StrUtil.isBlank(taxCode.getStartDate())) {
                throw new TXWR000004Exception(String.format(String.format("税编[%s]启用日期为空，禁止开票", taxCode.getGoodsTaxNo())));
            }

            Date startDate = format.parse(taxCode.getStartDate());
            // 启用日期不可为空
            if (null == startDate || startDate.compareTo(now) > 0) {
                throw new TXWR000004Exception(String.format("税编[%s]启用日期为[%s], 不在有效期范围内，禁止开票", taxCode.getGoodsTaxNo(), taxCode.getStartDate()));
            }

            if (StrUtil.isNotBlank(taxCode.getEndDate())) {
                Date endDate = format.parse(taxCode.getEndDate());
                // 停用日期可为空
                if (null != endDate && endDate.compareTo(now) < 0) {
                    throw new TXWR000004Exception(String.format("税编[%s]停用日期为[%s], 不在有效期范围内，禁止开票", taxCode.getGoodsTaxNo(), taxCode.getEndDate()));
                }
            }
        } catch (ParseException e) {
            throw new TXWR000004Exception(String.format("税编[%s]时间有误，禁止开票", taxCode.getGoodsTaxNo()));
        }
    }

    public static void validateFarmProduce(List<String> taxCodeList, List<String> farmProduceTaxCodeList) {
        final var taxCodeFilterList = taxCodeList.stream()
                .map(farmProduceTaxCodeList::contains)
                .distinct()
                .collect(Collectors.toList());

        if (taxCodeFilterList.size() > 1) {
            throw new TXWR000002Exception("农产品编码和非农产品编码不能混开");
        }
    }
}
