package com.xforceplus.taxware.chestnut.contract.model;

import lombok.Data;
import org.hibernate.validator.constraints.Length;
import org.hibernate.validator.constraints.ScriptAssert;

import javax.validation.constraints.Digits;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import javax.validation.constraints.Pattern;
import java.math.BigDecimal;
import java.util.List;
import java.util.Map;

/**
 * @author zhenmengying
 * @date 2023/1/18
 * @describe: 全电发票上传
 */
public class MakeInvoiceMessage {

    @Data
    public static class Request {
        /**
         * taskId
         */
        @NotEmpty
        private String taskId;

        private List<InvoiceInfo> invoiceInfos;
    }

    @Data
    public static class InvoiceInfo {

        /**
         * pid
         */
        @NotEmpty
        private String pid;

        /**
         * 租户Id
         */
        private String tenantId;

        /**
         * 蓝字发票标志
         * 0：蓝字发票
         * 1：红字发票
         */
        @NotEmpty
        @Length(max = 1)
        @Pattern(regexp = "0|1")
        private String invoiceFlag;

        /**
         * 发票票种
         * 01：全电专
         * 02：全电普
         */
        @NotEmpty
        @Length(max = 2)
        @Pattern(regexp = "0[1-2]")
        private String invoiceType;

        /**
         * 特定要素（非必须）
         * 空：非特定要素 空:""
         * 01：成品油发票
         * 02：稀土发票
         * 03：建筑服务发票
         * 04：货物运输服务发票
         * 05：不动产销售服务发票
         * 06：不动产租赁服务发票
         * 07：代收车船税
         * 08：通行费
         * 09：旅客运输服务发票
         * 10：医疗服务（住院）发票
         * 11：医疗服务（门诊）发票
         * 12：自产农产品销售发票
         * 13：拖拉机和联合收割机发票
         * 14：机动车
         * 15：二手车
         * 16：农产品收购发票
         * 17：光伏收购发票
         * 18：卷烟发票
         */
        @Length(max = 2)
        @Pattern(regexp = "0[1-9]|1[0-8]")
        private String invoiceStyleType;

        /**
         * 增值税即征即退代码（非必须）
         * 空：非增值税即征即退
         * 01：软件产品发票
         * 02：资源综合利用产品发票
         * 03：管道运输服务发票
         * 04：有形动产融资租赁服务
         * 05：有形动产融资性售后回租服务发票
         * 06：新型墙体材料发票
         * 07：风力发电产品发票
         * 08：光伏发电产品发票
         * 09：动漫软件产品发票
         * 10：飞机维修劳务发票
         * 11：黄金发票
         * 12：铂金发票
         */
        @Length(max = 2)
        @Pattern(regexp = "0[1-9]|1[0-2]")
        private String vatRefundType;

        /**
         * 销售方统一社会信用代码/纳税人识别号/身份证件号码
         */
        @NotEmpty
        @Length(max = 20)
        @Pattern(regexp = "[A-Z0-9]+")
        private String sellerTaxNo;

        /**
         * 销售方名称
         */
        @NotEmpty
        @Length(max = 300)
        private String sellerName;

        /**
         * 销售方地址（非必须）
         */
        @Length(max = 300)
        private String sellerAddress;

        /**
         * 销售方电话（非必须）
         */
        @Length(max = 60)
        private String sellerTel;

        /**
         * 销售方开户行（非必须）
         */
        @Length(max = 120)
        private String sellerBankName;

        /**
         * 销售方账号（非必须）
         */
        @Length(max = 100)
        private String sellerBankAccount;

        /**
         * 购买方统一社会信用代码/纳税人识别号/身份证件号码（非必须）
         */
        @Length(max = 20)
        @Pattern(regexp = "[A-Z0-9]+")
        private String buyerTaxNo;

        /**
         * 购买方名称
         */
        @NotEmpty
        @Length(max = 300)
        private String buyerName;

        /**
         * 购买方地址（非必须）
         */
        @Length(max = 300)
        private String buyerAddress;

        /**
         * 购买方电话（非必须）
         */
        @Length(max = 60)
        private String buyerTel;

        /**
         * 购买方开户行（非必须）
         */
        @Length(max = 120)
        private String buyerBankName;

        /**
         * 购买方账号（非必须）
         */
        @Length(max = 100)
        private String buyerBankAccount;

        /**
         * 购买方经办人姓名（非必须）
         */
        @Length(max = 150)
        private String buyerOperatorName;

        /**
         * 购买方经办人证件号码（非必须）
         */
        @Length(max = 30)
        private String buyerOperatorIdentityNo;

        /**
         * 购买方经办人联系电话（非必须）
         * jbrlxdh修改为gmfjbrlxdh
         */
        @Length(max = 60)
        private String buyerOperatorTel;

        /**
         * 是否自然人标识
         */
        private Boolean isNature = false;

        /**
         * 合计金额
         */
        @NotNull
        @Digits(integer = 18, fraction = 2)
        private BigDecimal amountWithoutTax;

        /**
         * 合计税额
         */
        @NotNull
        @Digits(integer = 18, fraction = 2)
        private BigDecimal taxAmount;

        /**
         * 价税合计
         */
        @NotNull
        @Digits(integer = 18, fraction = 2)
        private BigDecimal amountWithTax;

        /**
         * 收款银行名称（非必须）
         */
        @Length(max = 120)
        private String payeeBankName;

        /**
         * 收款银行账号（非必须）
         */
        @Length(max = 100)
        private String payeeBankAccount;

        /**
         * 结算方式（非必须）
         * 01：现金
         * 02：银行转账
         * 03：票据
         * 04：第三方支付
         * 05：预付卡
         * 99：其他
         */
        @Length(max = 2)
        @Pattern(regexp = "0[1-5]|99")
        private String settlementMode;

        /**
         * 开票人
         */
        @NotEmpty
        @Length(max = 300)
        private String issuer;

        /**
         * 收款人
         */
        @Length(max = 150)
        private String payee;

        /**
         * 复核人
         */
        @Length(max = 75)
        private String reviewer;

        /**
         * 开票人证件号码（非必须）
         */
        @Length(max = 30)
        private String issuerIdentityNo;

        /**
         * 开票人证件类型（非必须）
         */
        @Length(max = 4)
        private String issuerIdentityType;

        /**
         * 对应蓝字发票号码（非必须）
         */
        @Length(max = 20)
        private String originalInvoiceNo;

        /**
         * 红字确认信息单编号（非必须）
         * 是否蓝字发票标志为1时，此节点有内容
         * 红票开具时必传
         */
        @Length(max = 20)
        private String redLetterNumber;

        /**
         * 红字确认单uuid（非必须）
         * 是否蓝字发票标志为1时，此节点有内容红票开具时必传
         */
        @Length(max = 32)
        private String redLetterUuid;

        /**
         * 备注（非必须）
         */
        @Length(max = 450)
        private String remark;

        /**
         * 是否展示购方银行控制项(当值为true 会在备注展示购方银行信息)
         */
        private Boolean showBuyerBankControl = false;

        /**
         * 是否展示销方银行控制项(当值为true 会在备注展示销方银行信息)
         */
        private Boolean showSellerBankControl = false;

        private List<InvoiceDetail> invoiceDetailList;

        /**
         * 特定业务扩展要素
         */
        private List<Map<String, Object>> extraList;

        @ScriptAssert.List(
                {
                        @ScriptAssert(
                                lang = "javascript",
                                script = "com.xforceplus.taxware.chestnut.contract.model.PostUploadAllElectricInvoiceResultMessage.InvoiceDetail.isValidUnionInfo(_this)",
                                message = "数量、单价必须同时为空或同时不为空，请检查"
                        ),
                }
        )
        @Data
        public static class InvoiceDetail {
            /**
             * 明细序号
             */
            @NotNull
            @Digits(integer = 8, fraction = 0)
            private Integer rowNum;

            /**
             * 对应蓝字发票明细序号（非必须）
             * 红票必传
             */
            @Digits(integer = 8, fraction = 0)
            private Integer originalRowNum;

            /**
             * 项目名称
             * 格式必须是“*简称*商品名称
             */
            @NotEmpty
            @Length(max = 600)
            private String itemName;

            /**
             * 规格型号（非必须）
             */
            @Length(max = 150)
            private String specifications;

            /**
             * 单位（非必须）
             * 如“数量”“单价”栏次非空，则本栏次必须非空
             */
            @Length(max = 300)
            private String unit;

            /**
             * 数量（非必须）
             * 如“单价”“单位”栏次非空，则本栏次必须非空
             */
            @Length(max = 25)
            private String quantity;

            /**
             * 单价（非必须）
             * 如“数量”“单位”栏次非空，则本栏次必须非空
             */
            @Length(max = 25)
            private String unitPrice;

            /**
             * 金额
             * 最大长度18位，最多2位小数。//目前是整数最多18位，小数2位
             */
            @NotNull
            @Digits(integer = 18, fraction = 2)
            private BigDecimal amountWithoutTax;

            /**
             * 增值税税率/征收率
             */
            @NotNull
            @Digits(integer = 16, fraction = 6)
            private BigDecimal taxRate;

            /**
             * 税额
             */
            @NotNull
            @Digits(integer = 18, fraction = 2)
            private BigDecimal taxAmount;

            /**
             * 含税金额
             */
            @NotNull
            @Digits(integer = 18, fraction = 2)
            private BigDecimal amountWithTax;

            /**
             * 扣除额（非必须）
             */
            @Digits(integer = 18, fraction = 2)
            private BigDecimal deduction;

            /**
             * 商品和服务税收分类合并编码
             * 必须在商品编码信息中存在。
             */
            @NotEmpty
            @Length(max = 19)
            private String goodsTaxNo;

            /**
             * 发票行性质
             * 00：正常行
             * 01：折扣行
             * 02：被折扣行
             */
            @NotEmpty
            @Length(max = 2)
            @Pattern(regexp = "0[0-2]")
            private String discountType;

            /**
             * 优惠政策标识（非必须）
             * 01：简易征收
             * 02：稀土产品
             * 03：免税
             * 04：不征税
             * 05：先征后退
             * 06：100%先征后退
             * 07：50%先征后退
             * 08：按3%简易征收
             * 09：按5%简易征收
             * 10：按5%简易征收减按1.5%计征
             * 11：即征即退30%
             * 12：即征即退50%
             * 13：即征即退70%
             * 14：即征即退100%
             * 15：超税负3%即征即退
             * 16：超税负8%即征即退
             * 17：超税负12%即征即退
             * 18：超税负6%即征即退
             */
            @Pattern(regexp = "0[1-9]|1[0-8]")
            private String taxIncentivesType;
        }

        @Data
        public static class Additional {

            /**
             * 附加要素名称（非必须）
             */
            @Length(max = 200)
            private String additionalElementName;

            /**
             * 附加要素类型（非必须）
             */
            @Length(max = 200)
            private String additionalElementType;

            /**
             * 附加要素值（非必须）
             */
            @Length(max = 200)
            private String additionalElementValue;
        }
    }
}
