package com.xforceplus.taxware.chestnut.contract.model.constant.enums;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 能力编码枚举
 *
 * @author Bobo
 * @create 2023/3/21 10:05
 * @since 1.0.0
 */
public enum CapabilityCodeEnum {

    /**
     * 能力编码
     */
    COMMON("00", 1, "202007", "911001","乐企开票能力（基础版）"),
    OIL("01", 2, "202055", "999888","乐企数字化电子发票（成品油）开票能力"),
    CONSTRUCTION_SERVICE("03", 4, "202044", "202206","乐企数字化电子发票（建筑服务）开票能力"),
    FREIGHT_SERVICE("04", 8, "202026", "230207004", "乐企数字化电子发票（货物运输服务）开票能力"),
    REAL_ESTATE_SALES("05", 16, "202046", "202204", "乐企数字化电子发票（不动产销售）开票能力"),
    REAL_ESTATE_LEASE("06", 32, "202038", "202036", "乐企数字化电子发票（不动产经营租赁）开票能力"),
    PASSENGER_TRANSPORTATION("09", 64, "202027", "230207003", "乐企数字化电子发票（旅客运输服务）开票能力"),
    PRODUCE_SALES("12", 128, "202031", "1234567", "乐企数字化电子发票（农产品销售）开票能力"),
    PRODUCE_PURCHASE("16", 256, "202019", "202019", "乐企全电发票（农产品收购）开票能力"),
    PHOTOVOLTAIC_ACQUISITION("17", 512, "202017", "202017", "乐企全电发票（光伏收购）开票能力"),
    COLLECTION("99", 1024, "203067", "203067", "乐企归集能力"),
    DEDUCTION("100", 2048, "203065", "203065", "乐企增值税抵扣勾选能力"),
    TAX_REFUND_DEDUCTION("101", 4096, "203064", "203064", "乐企增值税退税勾选能力"),
    DEPUTY_TAX_REFUND_DEDUCTION("102", 8192, "203066", "203066", "乐企代办退税勾选能力"),
    ENTRY("103", 16384, "203057", "203057", "发票入账能力"),
    VERIFICATION("104", 32768, "203059", "203059", "乐企发票查验能力"),
    UNIT_MANAGEMENT("105", 262144, "100010","", "使用单位管理能力"),
    COMBINED_COMMON("200", 65536, "202076", "102302", "乐企联用数电发票（基础版）开票能力"),
    OTHER_COMMON("300", 131072, "202077", "102301", "乐企他用数电发票（基础版）开票能力");

    private final String code;
    private final int numCode;
    private final String capabilityCode;
    private final String caseCode;
    private final String desc;

    CapabilityCodeEnum(String code, int numCode, String capabilityCode,String caseCode, String desc) {
        this.code = code;
        this.numCode = numCode;
        this.capabilityCode = capabilityCode;
        this.caseCode = caseCode;
        this.desc = desc;
    }

    public String toCode() {
        return this.code;
    }

    public int toNumCode() {
        return this.numCode;
    }

    public String toCapabilityCode() {
        return this.capabilityCode;
    }

    public String toDesc() {
        return this.desc;
    }

    public String toCaseCode() {
        return this.caseCode;
    }

    public static CapabilityCodeEnum fromCode(final String code) {
        return Arrays.stream(CapabilityCodeEnum.values())
                .filter(x -> Objects.equals(x.code, code))
                .findAny()
                .orElse(COMMON);
    }

    public static CapabilityCodeEnum fromCapabilityCode(final String capabilityCode) {
        return Arrays.stream(CapabilityCodeEnum.values())
                .filter(x -> Objects.equals(x.capabilityCode, capabilityCode))
                .findAny()
                .orElse(COMMON);
    }

    public static CapabilityCodeEnum fromCapabilityCodeNoDefault(final String capabilityCode) {
        return Arrays.stream(CapabilityCodeEnum.values())
                .filter(x -> Objects.equals(x.capabilityCode, capabilityCode))
                .findAny()
                .orElse(null);
    }

    public static List<CapabilityCodeEnum> fromCodeList(final List<String> codeList) {
        if (codeList == null) {
            return List.of(CapabilityCodeEnum.COMMON);
        }

        return codeList.stream().map(CapabilityCodeEnum::fromCode).collect(Collectors.toList());
    }

    public static List<CapabilityCodeEnum> fromCodeList(final String[] codeList) {
        if (codeList == null) {
            return List.of(CapabilityCodeEnum.COMMON);
        }

        return Arrays.stream(codeList).map(CapabilityCodeEnum::fromCapabilityCode).collect(Collectors.toList());
    }

    public static List<CapabilityCodeEnum> fromCodeList(final String[] codeList, boolean noDefault) {
        return Arrays.stream(codeList).map(CapabilityCodeEnum::fromCapabilityCodeNoDefault).filter(Objects::nonNull).collect(Collectors.toList());
    }

    public static List<CapabilityCodeEnum> fromNumCode(Integer numCode) {
        if (numCode == null || numCode == 0) {
            return List.of(CapabilityCodeEnum.COMMON);
        }

        return Arrays.stream(CapabilityCodeEnum.values())
                .filter(x -> (x.numCode & numCode) > 0)
                .collect(Collectors.toList());
    }

    public static int to(List<CapabilityCodeEnum> capabilityCodeEnums) {
        int numCode = 0;
        if (capabilityCodeEnums == null) {
            return CapabilityCodeEnum.COMMON.numCode;
        }

        for (CapabilityCodeEnum capabilityCodeEnum : capabilityCodeEnums) {
            numCode += capabilityCodeEnum.numCode;
        }
        return numCode;
    }

    public static List<String> toCodeList(List<CapabilityCodeEnum> capabilityCodeEnums) {
        if (capabilityCodeEnums == null) {
            return Collections.singletonList(CapabilityCodeEnum.COMMON.code);
        }

        return capabilityCodeEnums.stream().map(CapabilityCodeEnum::toCode).collect(Collectors.toList());
    }
}
