package z1.pdf.verify;

import com.itextpdf.text.pdf.AcroFields;
import com.itextpdf.text.pdf.PdfReader;
import com.itextpdf.text.pdf.parser.PdfTextExtractor;
import com.itextpdf.text.pdf.security.PdfPKCS7;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import z1.pdf.verify.exception.PdfVerifyTerminateException;

import java.security.Security;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.util.List;

/**
 * √修订版1:由广州富力国际空港综合物流园有限公司签名
 * 签名有效
 * 本修订版文档未被更改
 * 在此修订版本之后,文档已被更改
 * 签名者的身份有效
 * 签名时间来自签名者计算机上的时钟。
 * 签名已启用LTV
 * √签名的详细信息
 * 原因:
 * 位置:
 * 证书详细信息
 * 上一次检查时间:2021.01.2716:46:43+0800
 * 域: Signature1位于第1页
 * 单击以查看本版本
 * √已创建批注
 * nk批注,位于第1页
 */
public class PdfVerifyUtils {

    // 提供算法库
    static {
        Security.addProvider(new BouncyCastleProvider());
    }

    /**
     * pdf校验
     *
     * @param bytes
     * @return 校验结果
     * @throws Exception IOException|GeneralSecurityException
     * @author xuLiang
     * @since 1.0.0
     */
    public static void verify(byte[] bytes) throws Exception {
        if (bytes == null) {
            throw new PdfVerifyTerminateException("参数不能为空");
        }
        // pdf文件格式,2进制前三个字节是16进制255044
        if (bytes[0] != 0x25 || bytes[1] != 0x50 || bytes[2] != 0x44) {
            throw new PdfVerifyTerminateException("文件格式不是PDF");
        }
        PdfReader reader = new PdfReader(bytes);
        AcroFields acroFields = reader.getAcroFields();
        List<String> signNames = acroFields.getSignatureNames();
        if (signNames.isEmpty()) {
            throw new PdfVerifyTerminateException("未找到PDF中的签章信息");
        }
        if (signNames.size() > 1) {
            throw new PdfVerifyTerminateException("文件不合规,检测到多个签章");
        }
        if ("".equals(PdfTextExtractor.getTextFromPage(reader, 1).toString())) {
            throw new PdfVerifyTerminateException("文件不合规,票面文本信息缺失");
        }
        for (String name : signNames) {
            PdfPKCS7 sign = acroFields.verifySignature(name);

            //验证是否篡改
            if (!acroFields.signatureCoversWholeDocument(name)) {
                throw new PdfVerifyTerminateException("文件被篡改");
            }
            //System.out.println(sign.getSignDate().getTime().getTime());
            //System.out.println(sign.getSigningCertificate().getSubjectDN().toString());
            //System.out.println(sign.getSigningCertificate().getIssuerDN().toString());
            //System.out.println(sign.getSigningCertificate().getNotBefore().getTime());
            //System.out.println(sign.getSigningCertificate().getNotAfter().getTime());

            //验证签名时间有效性
            if (sign.getSignDate().getTime().getTime() < sign.getSigningCertificate().getNotBefore().getTime()) {
                throw new PdfVerifyTerminateException("签章时间早于签章证书有效起始时间");
            }
            if (sign.getSignDate().getTime().getTime() > sign.getSigningCertificate().getNotAfter().getTime()) {
                throw new PdfVerifyTerminateException("签章时间超过签章证书有效截止时间");
            }

            //签名有效性
            if (!sign.verify()) {
                throw new PdfVerifyTerminateException("签名验证无效");
            }

            // 获取根证书对象
            //java.security.cert.CertificateFactory cf = java.security.cert.CertificateFactory.getInstance("X.509", "BC");
            //X509Certificate rootCa = (X509Certificate) cf.generateCertificate(new FileInputStream("/Users/zlinfo/work/code/taxware/taxware-z1/z1-pdf-verify/src/test/resources/cert/szca.cer"));
            //for (Certificate certificate : sign.getCertificates()) {
            //    CertificateVerifyUtil.doVerify((X509Certificate) certificate, rootCa);
            //}

            try {
                //签章证书有效期验证
                sign.getSigningCertificate().checkValidity();
            } catch (CertificateExpiredException e) {
                System.out.println("证书过期");
            } catch (CertificateNotYetValidException e) {
                System.out.println("证书未到起始有效期");
            }
            boolean hasTimeStampToken = sign.getTimeStampToken() == null ? false : true;
            if (hasTimeStampToken) {
                if (!sign.verifyTimestampImprint()) {
                    throw new PdfVerifyTerminateException("签章时间戳证书无效");
                }
            }
        }
    }
}
