package z1.pdf.verify.util;

import okhttp3.*;
import org.bouncycastle.asn1.*;
import z1.pdf.verify.exception.CertificateVerifyTerminateException;

import javax.naming.Context;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import javax.naming.ldap.InitialLdapContext;
import javax.naming.ldap.LdapContext;
import java.io.ByteArrayInputStream;
import java.security.cert.*;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * parent
 *
 * @Author 👽 zhaolei
 * @Date 📅 2020/4/26 15:51
 * motto: Tools change the world. 🛠
 * copyright: xforceplus 🇨🇳
 * description:
 */
public class CertificateVerifyUtil {

    /**
     * 验证证书入口方法
     */
    public static void doVerify(X509Certificate certificate, X509Certificate rootCa) throws Exception {
        try {
            certificate.checkValidity();
        } catch (CertificateExpiredException e) {
            System.out.println("[" + certificate.getSubjectDN().toString() + "]证书过期");
        } catch (CertificateNotYetValidException e) {
            System.out.println("[" + certificate.getSubjectDN().toString() + "]证书未到起始使用时间");
        }

        // 验证根证书信任链
        if (rootCa != null) {
            certificate.verify(rootCa.getPublicKey());
        }

        // 验证密钥用法是否正确
        verifyKeyUsage(certificate);

        // 验证证书是否被吊销
        checkRevocationStatus(certificate);
    }

    /**
     * 验证证书吊销状态
     * 示例
     * //crlUrl = "ldap://ldap2.sheca.com:389/cn=CRL22801.crl,ou=RA12050100,ou=CA10011,ou=crl,o=UniTrust?certificateRevocationList?base?objectClass=cRLDistributionPoint";
     * //crlUrl = "http://crl.securetrust.com/STCA.crl";
     *
     * @param certificate
     * @throws Exception
     */
    static void checkRevocationStatus(X509Certificate certificate) throws Exception {
        X509CRL crl = getCrlObj(certificate);
        if (crl == null) {
            System.out.println("未能解析到CRL对象信息");
        }
        if (crl.isRevoked(certificate)) {
            X509CRLEntry x509CRLEntry = crl.getRevokedCertificate(certificate.getSerialNumber());
            if (x509CRLEntry == null) {
                throw new CertificateVerifyTerminateException("证书已吊销");
            }
            SimpleDateFormat sdf1 = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");

            throw new CertificateVerifyTerminateException("证书已吊销,时间: " + sdf1.format(x509CRLEntry.getRevocationDate()) + ", 原因: " +
                    formatRevocationReason(x509CRLEntry.getRevocationReason()));
        }
    }

    static String formatRevocationReason(CRLReason reasonCode) {
        if (reasonCode.ordinal() == CRLReason.UNSPECIFIED.ordinal()) {
            return "未说明撤销原因";
        } else if (reasonCode.ordinal() == CRLReason.KEY_COMPROMISE.ordinal()) {
            return "已知或怀疑证书主题的私钥已被泄露";
        } else if (reasonCode.ordinal() == CRLReason.CA_COMPROMISE.ordinal()) {
            return "已知或怀疑证书主题的私钥已被泄露";
        } else if (reasonCode.ordinal() == CRLReason.AFFILIATION_CHANGED.ordinal()) {
            return "主题的名称或其他信息已更改";
        } else if (reasonCode.ordinal() == CRLReason.SUPERSEDED.ordinal()) {
            return "证书已被取代";
        } else if (reasonCode.ordinal() == CRLReason.CESSATION_OF_OPERATION.ordinal()) {
            return "不再需要证书";
        } else if (reasonCode.ordinal() == CRLReason.CERTIFICATE_HOLD.ordinal()) {
            return "证书已被搁置";
        } else if (reasonCode.ordinal() == CRLReason.UNUSED.ordinal()) {
            return "未使用";
        } else if (reasonCode.ordinal() == CRLReason.REMOVE_FROM_CRL.ordinal()) {
            return "证书以前处于暂停状态，应从CRL中删除";
        } else if (reasonCode.ordinal() == CRLReason.PRIVILEGE_WITHDRAWN.ordinal()) {
            return "授予证书主题的权限已被撤销";
        } else if (reasonCode.ordinal() == CRLReason.AA_COMPROMISE.ordinal()) {
            return "已知或怀疑证书主题的私钥已被泄露";
        } else {
            return reasonCode.name();
        }
    }

    /**
     * 验证密钥用法是否正确
     * 基于现有证书用法做验证,待测试
     *
     * @param certificate
     * @throws CertificateVerifyTerminateException
     */
    private static void verifyKeyUsage(X509Certificate certificate) throws CertificateVerifyTerminateException {

        boolean[] keyUsages = certificate.getKeyUsage();

        // 0 digital_signature
        if (!keyUsages[0]) {
            throw new CertificateVerifyTerminateException("密钥用法验证无效:digital_signature");
        }

        // 1 non_repudiation
        if (!keyUsages[1]) {
            throw new CertificateVerifyTerminateException("密钥用法验证无效:non_repudiation");
        }
    }


    private static X509CRL getCrlObj(X509Certificate certificate) throws Exception {

        // 获取 CrlDistributionPoints
        byte[] extensionValue = certificate.getExtensionValue("2.5.29.31");
        if (extensionValue == null) {
            System.out.println("证书文件不存在CRL分发点信息");
            return null;
        }

        // 解析crl分发点扩展信息
        ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(extensionValue);
        ASN1InputStream asn1InputStream = new ASN1InputStream(byteArrayInputStream);
        DEROctetString derObject = (DEROctetString) asn1InputStream.readObject();
        DLSequence seq = (DLSequence) new ASN1InputStream(derObject.getOctets()).readObject();

        //获取所有crl链接
        List<String> crlUrls = new ArrayList<String>();
        for (int i = 0; i < seq.size(); i++) {
            ASN1Encodable tobj = seq.getObjectAt(i);
            while (!(tobj instanceof DEROctetString)) {
                if (tobj instanceof ASN1Sequence && ((ASN1Sequence) tobj).size() > 0) {
                    tobj = ((ASN1Sequence) tobj).getObjectAt(0);
                }
                if (tobj instanceof ASN1TaggedObject) {
                    tobj = ((ASN1TaggedObject) tobj).getObject();
                }
            }

            String crlUrl = new String(((DEROctetString) tobj).getOctets());
            System.out.println(crlUrl);
            crlUrls.add(crlUrl);
        }

        X509CRL crl = null;
        for (String crlUrl : crlUrls) {
            if (crlUrl.startsWith("http")) {
                crl = getCrlByHttp(crlUrl);
            } else if (crlUrl.startsWith("ldap://")) {
                crl = getCrlByLdap(crlUrl);
            }
            if (crl != null) {
                break;
            }
        }
        return crl;
    }

    private static X509CRL getCrlByHttp(String crlUrl) throws Exception {
        ByteArrayInputStream in = null;
        X509CRL crl = null;
        try {
            byte[] result = httpGet(crlUrl);
            in = new ByteArrayInputStream(result);
            CertificateFactory cf = CertificateFactory.getInstance("X.509");
            crl = (X509CRL) cf.generateCRL(in);
        } catch (Exception e) {
            e.printStackTrace();
            return crl;
        } finally {
            if (in != null) {
                in.close();
            }
        }
        return crl;
    }

    /**
     * 基于CRL链接,获取CRL对象
     *
     * @param crlUrl 目前仅有一个可用示例,如下
     *               ldap://ldap2.sheca.com:389/cn=CRL22801.crl,ou=RA12050100,ou=CA10015,ou=crl,o=UniTrust?certificateRevocationList?base?objectClass=cRLDistributionPoint
     * @return
     * @throws Exception
     */
    private static X509CRL getCrlByLdap(String crlUrl) throws Exception {
        if (crlUrl == null && "".equals(crlUrl)) {
            return null;
        }
        ByteArrayInputStream in = null;
        X509CRL crl = null;
        try {
            crlUrl = crlUrl.replace("ldap://", "LDAP://");
            LdapContext ctx = ldapConnect(crlUrl);
            byte[] fileContent = null;
            if (ctx == null) {
                return null;
            }

            // todo 完善ldap搜索,目前可用ldap链接只有一个
            // 过滤条件
            String filter = "objectClass=cRLDistributionPoint";
            String[] attrPersonArray = {"certificateRevocationList;binary"};
            SearchControls searchControls = new SearchControls();//搜索控件
            searchControls.setSearchScope(SearchControls.SUBTREE_SCOPE);//搜索范围
            searchControls.setReturningAttributes(attrPersonArray);

            // 1.要搜索的上下文或对象的名称；2.过滤条件，可为null，默认搜索所有信息；3.搜索控件，可为null，使用默认的搜索控件
            NamingEnumeration<SearchResult> answer = ctx.search("", filter.toString(), searchControls);
            while (answer.hasMore() && fileContent == null) {
                SearchResult result = (SearchResult) answer.next();
                NamingEnumeration<? extends Attribute> attrs = result.getAttributes().getAll();
                while (attrs.hasMore()) {
                    Attribute attr = (Attribute) attrs.next();
                    fileContent = (byte[]) attr.get();
                    break;
                }
            }

            in = new ByteArrayInputStream(fileContent);
            CertificateFactory cf = CertificateFactory.getInstance("X.509");
            crl = (X509CRL) cf.generateCRL(in);
        } catch (Exception e) {
            e.printStackTrace();
            return crl;
        } finally {
            if (in != null) {
                in.close();
            }
        }
        return crl;
    }

    /**
     * 读取证书序列号
     *
     * @param cert
     * @return
     */
    private static String getSerialNumber(X509Certificate cert) {
        if (null == cert) {
            return null;
        }
        byte[] serial = cert.getSerialNumber().toByteArray();
        if (serial.length > 0) {
            String serialNumberString = new String();
            for (int i = 0; i < serial.length; i++) {
                String s = Integer.toHexString(Byte.valueOf(serial[i]).intValue());
                if (s.length() == 8) {
                    s = s.substring(6);
                } else if (1 == s.length()) {
                    s = "0" + s;
                }
                serialNumberString += s + " ";
            }
            return serialNumberString;
        }
        return null;
    }

    //建立ldap链接
    private static LdapContext ldapConnect(String url) {
        Hashtable<String, String> env = new Hashtable<>();

        String factory = "com.sun.jndi.ldap.LdapCtxFactory";
        env.put(Context.INITIAL_CONTEXT_FACTORY, factory);
        env.put(Context.SECURITY_AUTHENTICATION, "none");
        env.put(Context.PROVIDER_URL, url);
        env.put(Context.SECURITY_PRINCIPAL, "");
        env.put(Context.SECURITY_CREDENTIALS, "");
        env.put(Context.BATCHSIZE, "1000");
        env.put("com.sun.jndi.ldap.connect.timeout", "5000");//连接超时设置为3秒

        LdapContext ldapContext = null;
        try {
            ldapContext = new InitialLdapContext(env, null);
            System.out.println("ldap connect success");
        } catch (javax.naming.AuthenticationException e) {
            System.out.println("ldap认证失败");
        } catch (NamingException e) {
            System.out.println("ldap参数有误导致连接失败");
        }
        return ldapContext;
    }


    private static byte[] httpGet(String url) throws Exception {
        MediaType mediaType = MediaType.parse("application/json; charset=utf-8");
        OkHttpClient client = new OkHttpClient()
                .newBuilder()
                .connectTimeout(10, TimeUnit.SECONDS)
                .readTimeout(10, TimeUnit.SECONDS)
                .writeTimeout(10, TimeUnit.SECONDS)
                .build();

        HttpUrl.Builder urlBuilder = HttpUrl.parse(url).newBuilder();
        HttpUrl finalUrl = urlBuilder.build();
        Request request = (new Request.Builder())
                .get()
                .url(finalUrl)
                .build();
        Response response = null;
        response = client.newCall(request).execute();
        byte[] result = response.body().bytes();
        if (response.isSuccessful()) {

            return result;
        } else {
            throw new Exception("http请求失败,Url: " + url + "状态码: " + response.code() + ", 请求结果: " + result);
        }
    }
}
