package com.xforceplus.tech.base.core.context.route.impl;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.xforceplus.tech.base.core.context.route.RouteConfigContext;
import com.xforceplus.tech.base.core.context.route.RouteInfo;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;

public class DefaultRouteConfigContextImpl implements RouteConfigContext {
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final String cacheFileName = "xep-route-config.json";

    private List<String> epList;
    private Map<String, List<RouteInfo>> routeConfig;
    private boolean isInit;

    public DefaultRouteConfigContextImpl() {
        this.routeConfig = new ConcurrentHashMap<>();
        this.epList = new ArrayList<>();
    }


    @Override
    public void setEpList(List<String> epList) {
        this.epList = epList;
    }

    @Override
    public List<String> getEpList() {
        return epList;
    }

    @Override
    public void initRouteConfig(Map<String, List<RouteInfo>> value) throws IOException {
        if (value == null || value.isEmpty() || isInit) {
            return;
        }
        log.debug("初始化路由配置");
        value.keySet().stream().forEach(key -> {
            routeConfig.put(key, value.get(key));
        });
        isInit = true;
    }


    @Override
    public boolean initRouteConfigByCacheFile() throws IOException {
        File file = new File(getCurrentPath() + cacheFileName);
        if (file.exists()) {
            String configContent = FileUtils.readFileToString(file, StandardCharsets.UTF_8);
            if (configContent != null && configContent.length() > 0) {
                log.debug("加载配置缓存文件内容： {}", configContent);
                Map<String, List<RouteInfo>> config = JSONObject.parseObject(configContent,
                        new TypeReference<Map<String, List<RouteInfo>>>() {
                        });
                initRouteConfig(config);
                return true;
            }
        }
        return false;
    }

    @Override
    public void saveCacheConfigByFile() throws IOException {
        String configContent = JSONObject.toJSONString(routeConfig);
        File file = new File(getCurrentPath() + cacheFileName);
        log.debug("保存配置缓存文件内容： {}", configContent);
        FileUtils.writeStringToFile(file, configContent, StandardCharsets.UTF_8);
    }

    private String getCurrentPath() {
        String path = this.getClass().getResource("/").getPath();
        // jar包运行模式,这里获取到的是个包内路径
        if (path.indexOf('!') > -1) {
            // jar包运行模式默认为当前程序路径
            path = "";
        }
        return path;
    }

    @Override
    public void updateRouteConfig(Map<String, List<RouteInfo>> value) throws IOException {
        if (value == null || value.isEmpty()) {
            return;
        }
        log.debug("更新路由配置");
        synchronized (routeConfig) {
            routeConfig.clear();
            value.keySet().stream().forEach(key -> {
                routeConfig.put(key, value.get(key));
            });
        }
    }

    @Override
    public void setConfig(String key, List<RouteInfo> value) {
        routeConfig.put(key, value);
    }

    @Override
    public List<RouteInfo> getConfig(String key) {
        return routeConfig.get(key);
    }

    @Override
    public Map<String, List<RouteInfo>> getAllConfig() {
        return routeConfig;
    }

    @Override
    public void clearConfig() {
        routeConfig.clear();
    }

    @Override
    public RouteInfo getMatchConfig(String epCode, String tenantCode, Object[] args) {
        List<RouteInfo> routerRules = this.getConfig(epCode);
        if (routerRules == null || routerRules.size() == 0) {
            return null;
        }

        //todo 以后添加业务身份的匹配逻辑
        Optional<RouteInfo> firstRouter = routerRules.stream().filter(routerRule -> {
            if (routerRule.getTenantCode().equals(tenantCode)) {
                return true;
            }
            return false;
        }).findFirst();
        if (firstRouter.isPresent()) {
            return firstRouter.get();
        }
        return null;
    }

}
