package com.xforceplus.api.tenant.company;

import com.xforceplus.api.common.Uri;
import com.xforceplus.api.common.response.ResponseEntity;
import com.xforceplus.api.model.CompanyModel.Request.BindPackages;
import com.xforceplus.api.model.CompanyModel.Request.CompanyName;
import com.xforceplus.api.model.CompanyModel.Request.Query;
import com.xforceplus.api.model.CompanyModel.Request.Save;
import com.xforceplus.domain.company.CompanyDto;
import com.xforceplus.domain.org.OrgDto;
import com.xforceplus.domain.validation.ValidationGroup;
import io.geewit.data.jpa.envers.domain.ComparedRevision;
import org.hibernate.validator.constraints.Range;
import org.springframework.cloud.openfeign.SpringQueryMap;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.Date;
import java.util.Map;


/**
 * 公司相关接口 (tenant)
 *
 * @author geewit
 */
public interface CompanyApi {
    /**
     * 公司分页列表
     *
     * @param tenantKey
     * @param query
     * @param pageable
     * @param <C>
     * @return
     */
    @ResponseBody
    @RequestMapping(name = "公司列表", value = Path.PAGE, method = RequestMethod.GET)
    <C extends CompanyDto<O>, O extends OrgDto<O>> ResponseEntity<Page<C>> page(@PathVariable("tenantKey") String tenantKey,
                                                                                @SpringQueryMap Query query,
                                                                                Pageable pageable);

    /**
     * 公司分页列表
     *
     * @param tenantId
     * @param query
     * @param pageable
     * @param <C>
     * @return
     */
    default <C extends CompanyDto<O>, O extends OrgDto<O>> ResponseEntity<Page<C>> page(long tenantId,
                                                                                        Query query,
                                                                                        Pageable pageable) {
        query.setByTenantCode(false);
        return this.page(String.valueOf(tenantId), query, pageable);
    }


    /**
     * 获取公司信息
     *
     * @param tenantKey
     * @param companyKey
     * @param revisionDate
     * @param withExtendParams
     * @param byTenantCode     是否按租户Code查
     * @param byTaxNum         是否按税号查
     * @param <C>
     * @return
     */
    @ResponseBody
    @RequestMapping(name = "获取公司信息", value = Path.INFO, method = RequestMethod.GET)
    <C extends CompanyDto<O>, O extends OrgDto<O>> ResponseEntity<C> info(@PathVariable("tenantKey") String tenantKey,
                                                                          @PathVariable("companyKey") String companyKey,
                                                                          @RequestParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                                          @RequestParam(value = "byTaxNum", required = false, defaultValue = "false") boolean byTaxNum,
                                                                          @RequestParam(value = "revisionDate", required = false) Date revisionDate,
                                                                          @RequestParam(value = "withExtendParams", required = false) String withExtendParams);


    /**
     * 获取公司信息
     *
     * @param tenantId
     * @param companyId
     * @param revisionDate
     * @param withExtendParams
     * @param <C>
     * @return
     */
    default <C extends CompanyDto<O>, O extends OrgDto<O>> ResponseEntity<C> info(long tenantId,
                                                                                  long companyId,
                                                                                  Date revisionDate,
                                                                                  String withExtendParams) {
        return this.info(String.valueOf(tenantId), String.valueOf(companyId), false, false, revisionDate, withExtendParams);
    }

    /**
     * 查指定公司变更历史分页记录
     *
     * @param byTenantCode 是否按租户Code查
     * @param byTaxNum     是否按税号查
     * @param pageable
     * @return
     */
    @ResponseBody
    @RequestMapping(name = "查指定公司变更历史分页记录", value = Path.HISTORIES, method = RequestMethod.GET)
    <C extends CompanyDto<O>, O extends OrgDto<O>> ResponseEntity<Page<ComparedRevision<C, String>>> histories(@PathVariable("tenantKey") String tenantKey,
                                                                                                               @PathVariable("companyKey") String companyKey,
                                                                                                               @RequestParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                                                                               @RequestParam(value = "byTaxNum", required = false, defaultValue = "false") boolean byTaxNum,
                                                                                                               Pageable pageable);

    /**
     * 查指定公司变更历史分页记录
     *
     * @param pageable
     * @return
     */
    default <C extends CompanyDto<O>, O extends OrgDto<O>> ResponseEntity<Page<ComparedRevision<C, String>>> histories(long tenantId,
                                                                                                                       long companyId,
                                                                                                                       Pageable pageable) {
        return this.histories(String.valueOf(tenantId), String.valueOf(companyId), false, false, pageable);
    }


    /**
     * 新增公司
     *
     * @param tenantKey
     * @param byTenantCode 是否按租户Code查
     * @param company
     * @param <C>
     * @param <O>
     * @return
     */
    @ResponseBody
    @RequestMapping(name = "新增公司", value = Path.CREATE, method = RequestMethod.POST)
    <C extends CompanyDto<O>, O extends OrgDto<O>> ResponseEntity<C> create(@PathVariable("tenantKey") String tenantKey,
                                                                            @RequestParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                                            @RequestBody @Validated(ValidationGroup.OnCreate.class) Save company);

    default <C extends CompanyDto<O>, O extends OrgDto<O>> ResponseEntity<C> create(long tenantId,
                                                                                    @RequestBody Save company) {
        return this.create(String.valueOf(tenantId), false, company);
    }

    /**
     * 更新公司
     *
     * @param tenantKey
     * @param companyKey
     * @param company
     * @param <C>
     * @return
     */
    @ResponseBody
    @RequestMapping(name = "更新公司", value = Path.UPDATE, method = RequestMethod.PUT)
    <C extends CompanyDto<O>, O extends OrgDto<O>> ResponseEntity<C> update(@PathVariable("tenantKey") String tenantKey,
                                                                            @PathVariable("companyKey") String companyKey,
                                                                            @RequestParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                                            @RequestParam(value = "byTaxNum", required = false, defaultValue = "false") boolean byTaxNum,
                                                                            @RequestBody Save company);

    /**
     * 更新公司
     *
     * @param tenantId
     * @param companyId
     * @param company
     * @param <C>
     * @return
     */
    default <C extends CompanyDto<O>, O extends OrgDto<O>> ResponseEntity<C> update(long tenantId,
                                                                                    long companyId,
                                                                                    Save company) {
        return this.update(String.valueOf(tenantId), String.valueOf(companyId), false, false, company);
    }

    @ResponseBody
    @RequestMapping(name = "公司绑定服务包", value = Path.BIND_PACKAGES, method = RequestMethod.POST)
    ResponseEntity<String> bindPackages(@PathVariable("tenantKey") String tenantKey,
                                        @PathVariable("companyKey") String companyKey,
                                        @RequestParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                        @RequestParam(value = "byTaxNum", required = false, defaultValue = "false") boolean byTaxNum,
                                        @RequestBody BindPackages bindPackage);

    default ResponseEntity<String> bindPackages(long tenantId,
                                                long companyId,
                                                BindPackages bindPackage) {
        return this.bindPackages(String.valueOf(tenantId), String.valueOf(companyId), false, false, bindPackage);
    }

    @ResponseBody
    @RequestMapping(name = "修改公司名称", value = Path.UPDATE_COMPANY_NAME, method = RequestMethod.POST)
    ResponseEntity<String> updateCompanyName(@PathVariable("tenantKey") String tenantKey,
                                             @RequestParam(value = "byTenantCode", defaultValue = "false") Boolean byTenantCode,
                                             @RequestBody @Valid CompanyName companyName);

    default ResponseEntity<String> updateCompanyName(long tenantId,
                                                     CompanyName companyName) {
        return this.updateCompanyName(String.valueOf(tenantId), false, companyName);
    }

    /**
     * 设置或者更新公司终端鉴权
     *
     * @param tenantKey      租户ID或者key
     * @param companyKey     公司ID或者税号
     * @param byTenantCode   是否按租户Code查
     * @param byTaxNum       true:用税号查，false:用companyId查
     * @param terminalSwitch 鉴权开关，0关，1开
     * @return ture:成功，false:失败
     */
    @ResponseBody
    @RequestMapping(name = "设置公司终端鉴权", value = Path.AUTH_SWITCH, method = RequestMethod.POST)
    ResponseEntity<Boolean> setCompanyTerminalAuthSwitch(@PathVariable("tenantKey") String tenantKey,
                                                         @PathVariable("companyKey") String companyKey,
                                                         @RequestParam(value = "byTenantCode", defaultValue = "false") Boolean byTenantCode,
                                                         @RequestParam(value = "byTaxNum", defaultValue = "false") Boolean byTaxNum,
                                                         @RequestParam(value = "terminalSwitch") @Valid @Range(min = 0, max = 1) int terminalSwitch);

    default ResponseEntity<Boolean> setCompanyTerminalAuthSwitch(long tenantId,
                                                                 long companyId,
                                                                 int terminalSwitch) {
        return this.setCompanyTerminalAuthSwitch(String.valueOf(tenantId),
                String.valueOf(companyId), false, false, terminalSwitch);
    }

    /**
     * 查询公司终端鉴权开关状态
     *
     * @param tenantKey    租户ID
     * @param companyKey   公司ID或者税号
     * @param byTenantCode 是否按租户Code查
     * @param byTaxNum     true:用税号查，false:用companyId查
     * @return 1：开，0：关
     */
    @ResponseBody
    @RequestMapping(name = "查询公司终端鉴权", value = Path.AUTH_SWITCH, method = RequestMethod.GET)
    ResponseEntity<Integer> getCompanyTerminalAuthSwitch(@PathVariable("tenantKey") String tenantKey,
                                                         @PathVariable("companyKey") String companyKey,
                                                         @RequestParam(value = "byTenantCode", defaultValue = "false") Boolean byTenantCode,
                                                         @RequestParam(value = "byTaxNum", defaultValue = "false") Boolean byTaxNum);

    default ResponseEntity<Integer> getCompanyTerminalAuthSwitch(long tenantId,
                                                                 long companyId) {
        return this.getCompanyTerminalAuthSwitch(String.valueOf(tenantId),
                String.valueOf(companyId), false, false);
    }


    /**
     * 验证租户下公指定公司的多个服务包是否被角色绑定
     *
     * @param tenantKey    租户ID
     * @param companyKey   公司ID或者税号
     * @param byTenantCode 是否按租户Code查
     * @param byTaxNum     true:用税号查，false:用companyId查
     * @param packageIds   服务包id集合
     * @return 多个服务包的验证结果
     */
    @ResponseBody
    @RequestMapping(name = "验证租户下公指定公司的多个服务包是否被角色绑定", value = Path.VALID_PACKAGES_WERE_BOUND, method = RequestMethod.GET)
    ResponseEntity<Map<Long, Boolean>> validCompaniesPackagesBound(
            @PathVariable("tenantKey") String tenantKey,
            @PathVariable("companyKey") String companyKey,
            @RequestParam(value = "byTenantCode", defaultValue = "false") Boolean byTenantCode,
            @RequestParam(value = "byTaxNum", defaultValue = "false") Boolean byTaxNum,
            @RequestParam(value = "packageIds") String packageIds);

    default ResponseEntity<Map<Long, Boolean>> validCompaniesPackagesBound(long tenantId,
                                                                           long companyId,
                                                                           String packageIds) {
        return this.validCompaniesPackagesBound(
                String.valueOf(tenantId),
                String.valueOf(companyId),
                false,
                false,
                packageIds);
    }

    @SuppressWarnings("all")
    interface Path extends Uri {
        String API_PREFIX = Uri.PATH_PREFIX + "/{tenantKey}" + Uri.API_VERSION + "/companies";
        String PAGE = API_PREFIX;
        String LIST = API_PREFIX + "/list";
        String INFO = API_PREFIX + "/{companyKey}";
        String HISTORIES = API_PREFIX + "/{companyKey}/histories";
        String CREATE = API_PREFIX;
        String UPDATE = API_PREFIX + "/{companyKey}";
        String BIND_PACKAGES = API_PREFIX + "/{companyKey}/packages";
        String UPDATE_COMPANY_NAME = API_PREFIX + "/companyname";
        String AUTH_SWITCH = Uri.PATH_PREFIX + "/{tenantKey}" + Uri.API_VERSION + "/terminal/{companyKey}";
        String VALID_PACKAGES_WERE_BOUND = API_PREFIX + "/{companyKey}/packages/validation";

    }
}
