package com.xforceplus.api.tenant.org;

import com.xforceplus.api.common.Uri;
import com.xforceplus.api.common.response.ResponseEntity;
import com.xforceplus.api.model.OrgModel.Request.Query;
import com.xforceplus.api.model.OrgModel.Request.Save;
import com.xforceplus.api.model.OrgModel.Request.TreeQuery;
import com.xforceplus.api.model.TreeModel;
import com.xforceplus.api.model.UserModel;
import com.xforceplus.domain.account.AccountDto;
import com.xforceplus.domain.org.OrgDto;
import com.xforceplus.domain.tenant.RoleDto;
import com.xforceplus.domain.user.UserDto;
import com.xforceplus.domain.validation.ValidationGroup;
import org.hibernate.validator.constraints.Range;
import org.springframework.cloud.openfeign.SpringQueryMap;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;

/**
 * 组织相关接口 (tenant)
 *
 * @author geewit
 */
public interface OrgApi {
    @ResponseBody
    @RequestMapping(name = "租户组织分页列表", value = Path.PAGE, method = RequestMethod.GET)
    <O extends OrgDto<O>> ResponseEntity<Page<O>> page(@PathVariable("tenantKey") String tenantKey,
                                                       @SpringQueryMap Query query,
                                                       Pageable pageable);

    default <O extends OrgDto<O>> ResponseEntity<Page<O>> page(long tenantId,
                                                               Query query,
                                                               Pageable pageable) {
        query.setByTenantCode(false);
        return this.page(String.valueOf(tenantId), query, pageable);
    }

    @ResponseBody
    @RequestMapping(name = "租户组织列表", value = Path.LIST, method = RequestMethod.GET)
    <O extends OrgDto<O>> ResponseEntity<List<O>> list(@PathVariable("tenantKey") String tenantKey,
                                                       @SpringQueryMap Query query,
                                                       Sort sort);

    default <O extends OrgDto<O>> ResponseEntity<List<O>> list(long tenantId,
                                                               Query query,
                                                               Sort sort) {
        query.setByTenantCode(false);
        return this.list(String.valueOf(tenantId), query, sort);
    }

    @ResponseBody
    @RequestMapping(name = "租户新增组织", value = Path.CREATE, method = RequestMethod.POST)
    <O extends OrgDto<O>> ResponseEntity<O> create(@PathVariable("tenantKey") String tenantKey,
                                                   @RequestParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                   @RequestBody @Validated(ValidationGroup.OnCreate.class) Save model);

    default <O extends OrgDto<O>> ResponseEntity<O> create(long tenantId,
                                                           Save model) {
        return this.create(String.valueOf(tenantId), false, model);
    }

    @ResponseBody
    @RequestMapping(name = "租户向指定父组织批量导入组织", value = Path.BATCH_IMPORT_CHILDREN_INTO_PARENT, method = RequestMethod.POST)
    ResponseEntity<Boolean> batchImportChildrenIntoParent(@PathVariable("tenantKey") String tenantKey,
                                                          @PathVariable("parentKey") String parentKey,
                                                          @RequestParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                          @RequestParam(value = "byOrgCode", required = false) Boolean byOrgCode,
                                                          @RequestBody @Validated(ValidationGroup.OnCreate.class) List<@Valid Save> models);

    default ResponseEntity<Boolean> batchImportChildrenIntoParent(long tenantId,
                                                                  long parentId,
                                                                  List<Save> models) {
        return this.batchImportChildrenIntoParent(String.valueOf(tenantId), String.valueOf(parentId), false, false, models);
    }

    @ResponseBody
    @RequestMapping(name = "租户批量导入组织", value = Path.BATCH_IMPORT, method = RequestMethod.POST)
    ResponseEntity<Boolean> batchImport(@PathVariable("tenantKey") String tenantKey,
                                        @RequestParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                        @RequestBody @Validated(ValidationGroup.OnCreate.class) List<@Valid Save> models);

    default ResponseEntity<Boolean> batchImport(long tenantId,
                                                List<Save> models) {
        return this.batchImport(String.valueOf(tenantId), false, models);
    }

    @ResponseBody
    @RequestMapping(name = "租户更新组织", value = Path.UPDATE, method = RequestMethod.PUT)
    <O extends OrgDto<O>> ResponseEntity<O> update(@PathVariable("tenantKey") String tenantKey,
                                                   @PathVariable("orgKey") String orgKey,
                                                   @RequestParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                   @RequestParam(value = "byOrgCode", required = false) Boolean byOrgCode,
                                                   @RequestBody @Validated(ValidationGroup.OnUpdate.class) Save org);

    default <O extends OrgDto<O>> ResponseEntity<O> update(long tenantId,
                                                           long orgId,
                                                           Save org) {
        return this.update(String.valueOf(tenantId), String.valueOf(orgId), false, false, org);
    }

    @ResponseBody
    @RequestMapping(name = "租户更新组织状态", value = Path.UPDATE_STATUS, method = RequestMethod.PATCH)
    ResponseEntity<String> updateStatus(@PathVariable("tenantKey") String tenantKey,
                                        @PathVariable("orgKey") String orgKey,
                                        @PathVariable("status") @Valid @Range(max = 1) int status,
                                        @RequestParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                        @RequestParam(value = "byOrgCode", required = false) Boolean byOrgCode);

    default ResponseEntity<String> updateStatus(long tenantId, long orgId, int status) {
        return this.updateStatus(String.valueOf(tenantId), String.valueOf(orgId), status, false, false);
    }

    @ResponseBody
    @RequestMapping(name = "租户组织信息", value = Path.INFO, method = RequestMethod.GET)
    <O extends OrgDto<O>> ResponseEntity<O> info(@PathVariable("tenantKey") String tenantKey,
                                                 @PathVariable("orgKey") String orgKey,
                                                 @RequestParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                 @RequestParam(value = "byOrgCode", required = false) Boolean byOrgCode);

    default <O extends OrgDto<O>> ResponseEntity<O> info(long tenantId, long orgId) {
        return this.info(String.valueOf(tenantId), String.valueOf(orgId), false, false);
    }

    @ResponseBody
    @RequestMapping(name = "删除租户组织", value = Path.DELETE, method = RequestMethod.DELETE)
    ResponseEntity<String> delete(@PathVariable("tenantKey") String tenantKey,
                                  @PathVariable("orgKey") String orgKey,
                                  @RequestParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                  @RequestParam(value = "byOrgCode", required = false) Boolean byOrgCode);

    default ResponseEntity<String> delete(long tenantId, long orgId) {
        return this.delete(String.valueOf(tenantId), String.valueOf(orgId), false, false);
    }

    @ResponseBody
    @RequestMapping(name = "租户组织树", value = Path.TREE, method = RequestMethod.GET)
    <O extends OrgDto<O>> ResponseEntity<O> tree(@PathVariable("tenantKey") String tenantKey,
                                                 @SpringQueryMap TreeQuery treeQuery);

    default <O extends OrgDto<O>> ResponseEntity<O> tree(long tenantId, TreeQuery treeQuery) {
        return this.tree(String.valueOf(tenantId), treeQuery);
    }

    @ResponseBody
    @RequestMapping(name = "租户根组织列表", value = Path.ROOTS, method = RequestMethod.GET)
    <O extends OrgDto<O>> ResponseEntity<List<O>> roots(@PathVariable("tenantKey") String tenantKey,
                                                        @RequestParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                        @RequestParam(value = "userId", required = false) Long userId,
                                                        @RequestParam(value = "withUserBoundFlag", required = false) Boolean withUserBoundFlag);

    default <O extends OrgDto<O>> ResponseEntity<List<O>> roots(long tenantId, Long userId, Boolean withUserBoundFlag) {
        return this.roots(String.valueOf(tenantId), false, userId, withUserBoundFlag);
    }

    @ResponseBody
    @RequestMapping(name = "租户组织子节点", value = Path.CHILDREN, method = RequestMethod.GET)
    <O extends OrgDto<O>> ResponseEntity<List<O>> children(@PathVariable("tenantKey") String tenantKey,
                                                           @PathVariable("orgKey") String orgKey,
                                                           @RequestParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                           @RequestParam(value = "byOrgCode", required = false) Boolean byOrgCode,
                                                           @RequestParam(value = "userId", required = false) Long userId,
                                                           @RequestParam(value = "withUserBoundFlag", required = false) Boolean withUserBoundFlag);


    default <O extends OrgDto<O>> ResponseEntity<List<O>> children(long tenantId, long orgId, Long userId, Boolean withUserBoundFlag) {
        return this.children(String.valueOf(tenantId), String.valueOf(orgId), false, false, userId, withUserBoundFlag);
    }

    @ResponseBody
    @RequestMapping(name = "租户组织子孙", value = Path.DESCENDANTS, method = RequestMethod.GET)
    <O extends OrgDto<O>> ResponseEntity<List<O>> descendants(@PathVariable("tenantKey") String tenantKey,
                                                              @PathVariable("orgKey") String orgKey,
                                                              @RequestParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                              @RequestParam(value = "byOrgCode", required = false) Boolean byOrgCode);

    default <O extends OrgDto<O>> ResponseEntity<List<O>> descendants(long tenantId, long orgId) {
        return this.descendants(String.valueOf(tenantId), String.valueOf(orgId), false, false);
    }

    @ResponseBody
    @RequestMapping(name = "租户组织下的用户分页列表", value = Path.USERS, method = RequestMethod.GET)
    <U extends UserDto<O, R, A>, O extends OrgDto<O>, R extends RoleDto, A extends AccountDto> ResponseEntity<Page<U>> users(@PathVariable("tenantKey") String tenantKey,
                                                                                                                             @PathVariable("orgKey") String orgKey,
                                                                                                                             @SpringQueryMap UserModel.Request.Query query,
                                                                                                                             Pageable pageable);

    default <U extends UserDto<O, R, A>, O extends OrgDto<O>, R extends RoleDto, A extends AccountDto> ResponseEntity<Page<U>> users(long tenantId,
                                                                                                                                     long orgId,
                                                                                                                                     UserModel.Request.Query query,
                                                                                                                                     Pageable pageable) {
        query.setByTenantCode(false);
        query.setByOrgCode(false);
        return this.users(String.valueOf(tenantId), String.valueOf(orgId), query, pageable);
    }

    @ResponseBody
    @RequestMapping(name = "根据节点ids查询组织（返回ids对应分组）", value = Path.LIST_BY_ORGKEYS, method = RequestMethod.GET)
    <O extends OrgDto<O>> ResponseEntity<List<O>> listByOrgKeys(@PathVariable("tenantKey") String tenantKey,
                                                                @RequestParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                                @RequestParam(value = "scope") TreeModel.OrgScope scope,
                                                                @RequestParam(value = "byId", required = false, defaultValue = "true") boolean byId,
                                                                @RequestParam(value = "orgKeys") String orgKeys);


    @SuppressWarnings("all")
    interface Path extends Uri {
        String API_PREFIX = Uri.PATH_PREFIX + "/{tenantKey}" + Uri.API_VERSION + "/orgs";
        String PAGE = API_PREFIX;
        String LIST = API_PREFIX + "/list";
        String CREATE = API_PREFIX;
        String BATCH_IMPORT_CHILDREN_INTO_PARENT = API_PREFIX + "/{parentKey}/batch";
        String BATCH_IMPORT_WITH_RESULT = API_PREFIX + "/batch-with-result";
        String BATCH_IMPORT = API_PREFIX + "/batch";
        String UPDATE = API_PREFIX + "/{orgKey}";
        String UPDATE_STATUS = API_PREFIX + "/{orgKey}/status/{status}";
        String INFO = API_PREFIX + "/{orgKey}";
        String TREE = API_PREFIX + "/tree";
        String DELETE = API_PREFIX + "/{orgKey}";
        String ROOTS = API_PREFIX + "/roots";
        String CHILDREN = API_PREFIX + "/{orgKey}/children";
        String DESCENDANTS = API_PREFIX + "/{orgKey}/descendants";
        String USERS = API_PREFIX + "/{orgKey}/users";
        String BATCH_IMPORT_2 = API_PREFIX + "/excel";
        String BATCH_EXPORT_EXCEL = API_PREFIX + "/export";
        String DOWNLOAD_EXCEL = API_PREFIX + "/download/{fileId}";
        String LIST_BY_ORGKEYS = API_PREFIX + "/list/orgKeys";

    }
}
