package com.xforceplus.api.tenant.user;

import com.xforceplus.api.common.Uri;
import com.xforceplus.api.common.response.ResponseEntity;
import com.xforceplus.api.model.RoleModel.Request.*;
import com.xforceplus.api.model.UserModel;
import com.xforceplus.domain.account.AccountDto;
import com.xforceplus.domain.org.OrgDto;
import com.xforceplus.domain.resource.ResourceDto;
import com.xforceplus.domain.tenant.RoleDto;
import com.xforceplus.domain.user.UserDto;
import com.xforceplus.domain.validation.ValidationGroup;
import org.hibernate.validator.constraints.Range;
import org.springframework.cloud.openfeign.SpringQueryMap;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.validation.Valid;
import java.util.List;

/**
 * 角色相关接口 (tenant)
 *
 * @author geewit
 */
public interface RoleApi {

    @ResponseBody
    @RequestMapping(name = "角色分页列表", value = Path.PAGE, method = RequestMethod.GET)
    <T extends RoleDto> ResponseEntity<Page<T>> page(@PathVariable("tenantKey") String tenantKey,
                                                     @SpringQueryMap Query query,
                                                     Pageable pageable);

    default <T extends RoleDto> ResponseEntity<Page<T>> page(long tenantId,
                                                             Query query,
                                                             Pageable pageable) {
        query.setByTenantCode(false);
        return this.page(String.valueOf(tenantId), query, pageable);
    }

    @ResponseBody
    @RequestMapping(name = "新增角色", value = Path.CREATE, method = RequestMethod.POST)
    <T extends RoleDto> ResponseEntity<T> create(@PathVariable("tenantKey") String tenantKey,
                                                 @RequestParam(value = "byTenantCode", required = false) @Validated(ValidationGroup.OnCurrentTenantCreate.class) Boolean byTenantCode,
                                                 @RequestBody Create model);

    default <T extends RoleDto> ResponseEntity<T> create(long tenantId,
                                                         Create model) {
        return this.create(String.valueOf(tenantId), false, model);
    }

    @ResponseBody
    @RequestMapping(name = "新增角色", value = Path.BATCH_CREATE, method = RequestMethod.POST)
    <R extends RoleDto> ResponseEntity<List<R>> batchCreate(@PathVariable("tenantKey") String tenantKey,
                                                            @RequestParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                            @RequestBody List<Save> models)  ;

    default <R extends RoleDto> ResponseEntity<List<R>> batchCreate(long tenantId,
                                                                    List<Save> models) {
        return this.batchCreate(String.valueOf(tenantId), false, models);
    }

    @ResponseBody
    @RequestMapping(name = "更新角色", value = Path.UPDATE, method = RequestMethod.PUT)
    <T extends RoleDto> ResponseEntity<T> update(@PathVariable("tenantKey") String tenantKey,
                                                 @PathVariable("roleKey") String roleKey,
                                                 @RequestParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                                 @RequestParam(value = "byRoleCode", required = false) Boolean byRoleCode,
                                                 @RequestBody @Valid Update model);

    default <T extends RoleDto> ResponseEntity<T> update(long tenantId,
                                                         long roleId,
                                                         Update model) {
        return this.update(String.valueOf(tenantId), String.valueOf(roleId), false, false, model);
    }

    @ResponseBody
    @RequestMapping(name = "更新角色状态", value = Path.UPDATE_STATUS, method = RequestMethod.PATCH)
    ResponseEntity<String> updateStatus(@PathVariable("tenantKey") String tenantKey,
                                        @PathVariable("roleKey") String roleKey,
                                        @RequestParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                        @RequestParam(value = "byRoleCode", required = false) Boolean byRoleCode,
                                        @PathVariable("status") @Valid @Range(max = 1L) int status);

    default ResponseEntity<String> updateStatus(long tenantId,
                                                long roleId,
                                                int status) {
        return this.updateStatus(String.valueOf(tenantId), String.valueOf(roleId), false, false, status);
    }

    @ResponseBody
    @RequestMapping(name = "角色信息", value = Path.INFO, method = RequestMethod.GET)
    <T extends RoleDto> ResponseEntity<T> info(@PathVariable("tenantKey") String tenantKey,
                                               @PathVariable("roleKey") String roleKey,
                                               @RequestParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                               @RequestParam(value = "byRoleCode", required = false) Boolean byRoleCode);

    default <T extends RoleDto> ResponseEntity<T> info(long tenantId,
                                                       long roleId) {
        return this.info(String.valueOf(tenantId), String.valueOf(roleId), false, false);
    }

    @ResponseBody
    @RequestMapping(name = "删除角色", value = Path.DELETE, method = RequestMethod.DELETE)
    ResponseEntity<String> delete(@PathVariable("tenantKey") String tenantKey,
                                  @PathVariable("roleKey") String roleKey,
                                  @RequestParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                  @RequestParam(value = "byRoleCode", required = false) Boolean byRoleCode);

    default ResponseEntity<String> delete(long tenantId,
                                          long roleId) {
        return this.delete(String.valueOf(tenantId), String.valueOf(roleId), false, false);
    }

    @ResponseBody
    @RequestMapping(name = "角色用户列表", value = Path.USERS, method = RequestMethod.GET)
    <U extends UserDto<O, R, A>, O extends OrgDto<O>, R extends RoleDto, A extends AccountDto> ResponseEntity<Page<U>> users(
            @PathVariable("tenantKey") String tenantKey,
            @PathVariable("roleKey") String roleKey,
            @SpringQueryMap UserModel.Request.Query query,
            Pageable pageable);

    default <U extends UserDto<O, R, A>, O extends OrgDto<O>, R extends RoleDto, A extends AccountDto> ResponseEntity<Page<U>> users(
            long tenantId,
            long roleId,
            UserModel.Request.Query query,
            Pageable pageable) {
        query.setByTenantCode(false);
        query.setByRoleCode(false);
        return this.users(String.valueOf(tenantId), String.valueOf(roleId), query, pageable);
    }

    @ResponseBody
    @RequestMapping(name = "角色资源码列表", value = Path.RESOURCES, method = RequestMethod.GET)
    <T extends ResourceDto> ResponseEntity<Page<T>> resources(
            @PathVariable("tenantKey") String tenantKey,
            @PathVariable("roleKey") String roleKey,
            @RequestParam(value = "byTenantCode", required = false) Boolean byTenantCode,
            @RequestParam(value = "byRoleCode", required = false) Boolean byRoleCode,
            Pageable pageable);

    default <T extends ResourceDto> ResponseEntity<Page<T>> resources(
            long tenantId,
            long roleId,
            Pageable pageable) {
        return this.resources(String.valueOf(tenantId), String.valueOf(roleId), false, false, pageable);
    }

    @ResponseBody
    @RequestMapping(name = "角色批量绑定用户", value = Path.BIND_USERS, method = RequestMethod.POST)
    ResponseEntity<String> bindUsers(@PathVariable("tenantKey") String tenantKey,
                                     @PathVariable("roleKey") String roleKey,
                                     @RequestParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                     @RequestParam(value = "byRoleCode", required = false) Boolean byRoleCode,
                                     @RequestBody BindUsers bindUsers);


    default ResponseEntity<String> bindUsers(long tenantId,
                                             long roleId,
                                             BindUsers bindUsers) {
        return this.bindUsers(String.valueOf(tenantId), String.valueOf(roleId), false, false, bindUsers);
    }

    @ResponseBody
    @RequestMapping(name = "角色批量反绑定用户", value = Path.UNBIND_USERS, method = RequestMethod.POST)
    ResponseEntity<String> unbindUsers(@PathVariable("tenantKey") String tenantKey,
                                       @PathVariable("roleKey") String roleKey,
                                       @RequestParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                       @RequestParam(value = "byRoleCode", required = false) Boolean byRoleCode,
                                       @RequestBody UnbindUsers unbindUsers);


    default ResponseEntity<String> unbindUsers(long tenantId,
                                               long roleId,
                                               UnbindUsers unbindUsers) {
        return this.unbindUsers(String.valueOf(tenantId), String.valueOf(roleId), false, false, unbindUsers);
    }

    @ResponseBody
    @RequestMapping(name = "批量导入关联账号和角色", value = Path.BATCH_IMPORT, method = RequestMethod.POST)
    ResponseEntity<Long> batchImportBind(@PathVariable("tenantKey") String tenantKey,
                                         @RequestParam(value = "byTenantCode", required = false) Boolean byTenantCode,
                                         @RequestParam("file") MultipartFile file);

    @SuppressWarnings("all")
    interface Path extends Uri {
        String PAGE = Uri.PATH_PREFIX + "/{tenantKey}" + Uri.API_VERSION + "/roles";
        String LIST = Uri.PATH_PREFIX + "/{tenantKey}" + Uri.API_VERSION + "/roles/list";
        String CREATE = Uri.PATH_PREFIX + "/{tenantKey}" + Uri.API_VERSION + "/roles";
        String BATCH_CREATE = Uri.PATH_PREFIX + "/{tenantKey}" + Uri.API_VERSION + "/roles/batch";
        String UPDATE = Uri.PATH_PREFIX + "/{tenantKey}" + Uri.API_VERSION + "/roles/{roleKey}";
        String UPDATE_STATUS = Uri.PATH_PREFIX + "/{tenantKey}" + Uri.API_VERSION + "/roles/{roleKey}/status/{status}";
        String INFO = Uri.PATH_PREFIX + "/{tenantKey}" + Uri.API_VERSION + "/roles/{roleKey}";
        String DELETE = Uri.PATH_PREFIX + "/{tenantKey}" + Uri.API_VERSION + "/roles/{roleKey}";
        String USERS = Uri.PATH_PREFIX + "/{tenantKey}" + Uri.API_VERSION + "/roles/{roleKey}/users";
        String RESOURCES = Uri.PATH_PREFIX + "/{tenantKey}" + Uri.API_VERSION + "/roles/{roleKey}/resources";
        String BIND_RESOURCE_SET = Uri.PATH_PREFIX + "/{tenantKey}" + Uri.API_VERSION + "/roles/resource-sets";
        String BIND_USERS = Uri.PATH_PREFIX + "/{tenantKey}" + Uri.API_VERSION + "/roles/{roleKey}/users";
        String UNBIND_USERS = Uri.PATH_PREFIX + "/{tenantKey}" + Uri.API_VERSION + "/roles/{roleKey}/users/unbind";
        String BATCH_IMPORT = Uri.PATH_PREFIX + "/{tenantKey}" + Uri.API_VERSION + "/roles/bind/users";
    }
}
