package com.xforceplus.tenant.security.client.config;

import com.fasterxml.jackson.databind.Module;
import com.xforceplus.tenant.core.exception.ErrorCodes;
import com.xforceplus.tenant.core.exception.TenantFeignException;
import com.xforceplus.tenant.core.exception.response.ErrorResponse;
import com.xforceplus.tenant.security.client.interceptor.TenantTokenRequestInterceptor;
import com.xforceplus.tenant.security.client.interceptor.UserContextFeignInterceptor;
import com.xforceplus.tenant.security.client.service.ClientService;
import com.xforceplus.tenant.security.client.support.OauthClientProperties;

import feign.*;
import feign.codec.Decoder;
import feign.codec.Encoder;
import feign.codec.ErrorDecoder;
import feign.okhttp.OkHttpClient;
import feign.optionals.OptionalDecoder;
import io.geewit.web.utils.JsonUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.ObjectFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.data.web.SpringDataWebProperties;
import org.springframework.boot.autoconfigure.http.HttpMessageConverters;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.cloud.openfeign.AnnotatedParameterProcessor;
import org.springframework.cloud.openfeign.FeignAutoConfiguration;
import org.springframework.cloud.openfeign.FeignFormatterRegistrar;
import org.springframework.cloud.openfeign.support.*;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Scope;
import org.springframework.core.convert.ConversionService;
import org.springframework.format.support.DefaultFormattingConversionService;
import org.springframework.format.support.FormattingConversionService;
import org.springframework.http.HttpStatus;
import org.springframework.web.client.RestTemplate;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import static feign.FeignException.errorStatus;

@AutoConfigureAfter({FeignAutoConfiguration.class})
@Configuration(proxyBeanMethods = false)
@EnableConfigurationProperties({OauthClientProperties.class})
public class TenantFeignConfiguration {
    private static final Logger logger = LoggerFactory.getLogger(TenantFeignConfiguration.class);

    private final ObjectFactory<HttpMessageConverters> messageConverters;

    public TenantFeignConfiguration(ObjectFactory<HttpMessageConverters> messageConverters) {
        this.messageConverters = messageConverters;
        logger.info("TenantFeignConfiguration initializing");
    }

    @Autowired(required = false)
    private List<AnnotatedParameterProcessor> parameterProcessors = new ArrayList<>();

    @Autowired(required = false)
    private List<FeignFormatterRegistrar> feignFormatterRegistrars = new ArrayList<>();

    @Autowired(required = false)
    private SpringDataWebProperties springDataWebProperties;

    @Bean("tenantFeignDecoder")
    public Decoder feignDecoder() {
        return new OptionalDecoder(
                new ResponseEntityDecoder(new SpringDecoder(this.messageConverters)));
    }

    @Bean("tenantFeignEncoder")
    @ConditionalOnMissingClass("org.springframework.data.domain.Pageable")
    public Encoder feignEncoder() {
        return new SpringEncoder(this.messageConverters);
    }

    @Bean("tenantErrorDecoder")
    public ErrorDecoder errorDecoder() {
        return (methodKey, response) -> {
            logger.info("feign client response:", response);
            String body;
            try {
                body = Util.toString(response.body().asReader());
            } catch (IOException e) {
                logger.error("feign.IOException", e);
                throw new TenantFeignException(ErrorCodes.NOT_FOUND.getMessage(), ErrorCodes.NOT_FOUND.getCode(), response.status());
            }
            ErrorResponse errorResponse = JsonUtils.fromJson(body, ErrorResponse.class);
            if (response.status() >= HttpStatus.BAD_REQUEST.value() && response.status() <= HttpStatus.INTERNAL_SERVER_ERROR.value()) {
                logger.info(JsonUtils.toJson(errorResponse));
                throw new TenantFeignException(errorResponse.getMessage(), errorResponse.getCode(), response.status());
            }
            return errorStatus(methodKey, response);
        };
    }

    @Bean("tenantPageableSpringEncoder")
    @ConditionalOnClass(name = "org.springframework.data.domain.Pageable")
    @ConditionalOnMissingBean(PageableSpringEncoder.class)
    public PageableSpringEncoder feignEncoderPageable() {
        PageableSpringEncoder encoder = new PageableSpringEncoder(
                new SpringEncoder(this.messageConverters));
        if (springDataWebProperties != null) {
            encoder.setPageParameter(
                    springDataWebProperties.getPageable().getPageParameter());
            encoder.setSizeParameter(
                    springDataWebProperties.getPageable().getSizeParameter());
            encoder.setSortParameter(
                    springDataWebProperties.getSort().getSortParameter());
        }
        return encoder;
    }

    @Bean
    @ConditionalOnClass(name = {
            "org.springframework.data.domain.Page",
    })
    public Module pageJacksonModule() {
        return new PageJacksonModule();
    }

    @Bean("tenantSpringMvcContract")
    public Contract feignContract(@Qualifier("tenantFormattingConversionService") ConversionService feignConversionService) {
        return new SpringMvcContract(this.parameterProcessors, feignConversionService);
    }

    @Bean("tenantFormattingConversionService")
    public FormattingConversionService feignConversionService() {
        FormattingConversionService conversionService = new DefaultFormattingConversionService();
        this.feignFormatterRegistrars.forEach(feignFormatterRegistrar -> feignFormatterRegistrar.registerFormatters(conversionService));
        return conversionService;
    }

    @Bean
    @ConditionalOnMissingBean(Retryer.class)
    public Retryer feignRetryer() {
        return Retryer.NEVER_RETRY;
    }


    @Bean(name = "tenantFeignClient")
    public OkHttpClient tenantClient(okhttp3.OkHttpClient okHttpClient) {
        OkHttpClient client = new OkHttpClient(okHttpClient);
        return client;
    }

    @Bean
    @Scope("prototype")
    @ConditionalOnMissingBean
    public Feign.Builder feignBuilder(Retryer retryer, @Qualifier(value = "tenantOkHttpClient") okhttp3.OkHttpClient tenantOkHttpClient) {
        return Feign.builder().retryer(retryer).client(new OkHttpClient(tenantOkHttpClient));
    }

    @ConditionalOnProperty(name = "xforce.tenant.security.auth.enable", havingValue = "true", matchIfMissing = true)
    @Bean("tenantClientService")
    public ClientService clientService(@Qualifier(value = "tenantRestTemplate") RestTemplate okHttpRestTemplate,
                                       OauthClientProperties oauthClientProperties) {
        return new ClientService(okHttpRestTemplate, oauthClientProperties);
    }

    @ConditionalOnProperty(name = "xforce.tenant.security.auth.enable", havingValue = "true", matchIfMissing = true)
    @Bean(name = "tenantTokenRequestInterceptor")
    public TenantTokenRequestInterceptor tenantTokenRequestInterceptor(@Qualifier("tenantClientService") ClientService clientService) {
        return new TenantTokenRequestInterceptor(clientService);
    }

    @Bean(name = "userContextFeignInterceptor")
    public UserContextFeignInterceptor userContextFeignInterceptor() {
        return new UserContextFeignInterceptor();
    }


}
