package com.xforceplus.tenant.security.client.service;

import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.xforceplus.tenant.security.client.support.OauthClientProperties;
import io.geewit.web.utils.JsonUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.Map;


public class ClientService {
    private static final Logger logger = LoggerFactory.getLogger(ClientService.class);

    public ClientService(RestTemplate restTemplate, OauthClientProperties clientProperties) {
        this.restTemplate = restTemplate;
        this.clientProperties = clientProperties;
        this.cache = CacheBuilder.newBuilder().maximumSize(1).expireAfterAccess(Duration.of(clientProperties.getExpireHours(), ChronoUnit.HOURS)).build(loader);
    }

    private final RestTemplate restTemplate;

    private final OauthClientProperties clientProperties;

    private final LoadingCache<String, String> cache;

    private final CacheLoader<String, String> loader = new CacheLoader<String, String>() {
        @Override
        public String load(String key) {
            return fetchToken();
        }
    };

    public String token() {
        if(clientProperties.getParams() == null || StringUtils.isBlank(clientProperties.getParams().getClientId())) {
            return null;
        }
        return cache.getUnchecked(clientProperties.getParams().getClientId());
    }

    public void refresh() {
        cache.invalidateAll();
    }

    private String fetchToken() {
        try {
            ResponseEntity<String> responseEntity = restTemplate.postForEntity(clientProperties.getRequestUrl(), clientProperties.getParams(), String.class);
            if(responseEntity.getStatusCode().is2xxSuccessful()) {
                String body = responseEntity.getBody();
                Map<String, String> bodyMap = JsonUtils.fromJson(body, new TypeReference<Map<String, String>>(){});
                return bodyMap.get("data");
            } else {
                logger.warn("request " + clientProperties.getRequestUrl() + " faild (reponse.status: " + responseEntity.getStatusCodeValue());
                return null;
            }
        } catch (RestClientException e) {
            logger.warn(e.getMessage());
            return null;
        }
    }
}
